/**
 * WP Outreach - Post Subscription Form Handler
 *
 * Handles AJAX form submission for post subscription shortcodes.
 */
(function() {
    'use strict';

    // Wait for DOM ready
    document.addEventListener('DOMContentLoaded', function() {
        // Find all post subscription forms
        const forms = document.querySelectorAll('.wpo-post-subscribe-form form');

        forms.forEach(function(form) {
            form.addEventListener('submit', handleSubmit);
        });
    });

    /**
     * Handle form submission
     *
     * @param {Event} e Submit event
     */
    function handleSubmit(e) {
        e.preventDefault();

        const form = e.target;
        const button = form.querySelector('.wpo-form-button');
        const btnText = button.querySelector('.wpo-btn-text');
        const btnLoading = button.querySelector('.wpo-btn-loading');
        const messageEl = form.querySelector('.wpo-form-message');

        // Get form data
        const formData = new FormData(form);
        const data = {
            email: formData.get('email'),
            post_id: formData.get('post_id'),
            post_type: formData.get('post_type'),
            subscription_type: formData.get('subscription_type'),
            scope: formData.get('scope') || 'post',
            website: formData.get('website') || '' // Honeypot
        };

        // Validate email
        if (!data.email || !isValidEmail(data.email)) {
            showMessage(messageEl, 'Please enter a valid email address.', 'error');
            return;
        }

        // Show loading state
        button.disabled = true;
        if (btnText) btnText.style.display = 'none';
        if (btnLoading) btnLoading.style.display = 'inline-flex';
        if (messageEl) messageEl.style.display = 'none';

        // Check if wpOutreachPostSubscribe is defined
        if (typeof wpOutreachPostSubscribe === 'undefined') {
            console.error('WP Outreach: wpOutreachPostSubscribe not defined');
            showMessage(messageEl, 'Configuration error. Please contact site administrator.', 'error');
            resetButton(button, btnText, btnLoading);
            return;
        }

        // Send AJAX request
        fetch(wpOutreachPostSubscribe.apiUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpOutreachPostSubscribe.nonce
            },
            body: JSON.stringify(data)
        })
        .then(function(response) {
            return response.json();
        })
        .then(function(result) {
            if (result.success) {
                // Show success message
                const successMessage = form.dataset.successMessage || result.message || 'Subscribed successfully!';
                showMessage(messageEl, successMessage, 'success');

                // Clear email field
                form.querySelector('input[name="email"]').value = '';

                // Optionally hide form after success
                // form.style.display = 'none';
            } else {
                // Show error message
                showMessage(messageEl, result.message || 'An error occurred. Please try again.', 'error');
            }
        })
        .catch(function(error) {
            console.error('WP Outreach subscription error:', error);
            showMessage(messageEl, 'An error occurred. Please try again.', 'error');
        })
        .finally(function() {
            resetButton(button, btnText, btnLoading);
        });
    }

    /**
     * Validate email format
     *
     * @param {string} email Email address
     * @returns {boolean} Is valid
     */
    function isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    /**
     * Show message to user
     *
     * @param {Element} el Message element
     * @param {string} text Message text
     * @param {string} type Message type: 'success' or 'error'
     */
    function showMessage(el, text, type) {
        if (!el) return;

        el.textContent = text;
        el.className = 'wpo-form-message wpo-' + type;
        el.style.display = 'block';

        // Auto-hide success messages after 5 seconds
        if (type === 'success') {
            setTimeout(function() {
                el.style.display = 'none';
            }, 5000);
        }
    }

    /**
     * Reset button to default state
     *
     * @param {Element} button Button element
     * @param {Element} btnText Text span
     * @param {Element} btnLoading Loading span
     */
    function resetButton(button, btnText, btnLoading) {
        button.disabled = false;
        if (btnText) btnText.style.display = 'inline';
        if (btnLoading) btnLoading.style.display = 'none';
    }
})();
