<?php

namespace WPOutreach\Tracking;

/**
 * Handles unsubscribe requests
 *
 * Provides secure token-based unsubscribe functionality for email compliance
 * (CAN-SPAM, GDPR). Shows a subscription management page where users can
 * choose which subscriptions to keep or remove.
 *
 * @since 1.0.0
 */
class UnsubscribeHandler
{
    /**
     * Secret key for token generation
     */
    private static function getSecretKey(): string
    {
        $key = get_option('wp_outreach_unsubscribe_key');

        if (!$key) {
            $key = wp_generate_password(64, true, true);
            update_option('wp_outreach_unsubscribe_key', $key);
        }

        return $key;
    }

    /**
     * Generate a secure unsubscribe token for a subscriber
     *
     * @param int    $subscriberId Subscriber ID
     * @param string $email        Subscriber email
     * @return string Base64-encoded token
     */
    public static function generateToken(int $subscriberId, string $email): string
    {
        $data = $subscriberId . '|' . $email;
        $hash = hash_hmac('sha256', $data, self::getSecretKey());

        // Token format: base64(subscriber_id:hash)
        return base64_encode($subscriberId . ':' . substr($hash, 0, 32));
    }

    /**
     * Validate an unsubscribe token and extract subscriber ID
     *
     * @param string $token The token to validate
     * @return int|false Subscriber ID if valid, false otherwise
     */
    public static function validateToken(string $token): int|false
    {
        $decoded = base64_decode($token, true);

        if ($decoded === false) {
            return false;
        }

        $parts = explode(':', $decoded);

        if (count($parts) !== 2) {
            return false;
        }

        [$subscriberId, $tokenHash] = $parts;
        $subscriberId = (int) $subscriberId;

        if ($subscriberId <= 0) {
            return false;
        }

        // Get subscriber email to verify token
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, email FROM {$table} WHERE id = %d",
            $subscriberId
        ));

        if (!$subscriber) {
            return false;
        }

        // Regenerate hash and compare
        $expectedData = $subscriberId . '|' . $subscriber->email;
        $expectedHash = substr(hash_hmac('sha256', $expectedData, self::getSecretKey()), 0, 32);

        if (!hash_equals($expectedHash, $tokenHash)) {
            return false;
        }

        return $subscriberId;
    }

    /**
     * Get subscriber details
     */
    public static function getSubscriber(int $subscriberId): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $subscriberId
        ));
    }

    /**
     * Get subscriber's list subscriptions
     */
    public static function getListSubscriptions(int $subscriberId): array
    {
        global $wpdb;
        $lists_table = $wpdb->prefix . 'outreach_lists';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT l.id, l.name, l.description
             FROM {$lists_table} l
             INNER JOIN {$pivot_table} sl ON l.id = sl.list_id
             WHERE sl.subscriber_id = %d
             ORDER BY l.name ASC",
            $subscriberId
        ));
    }

    /**
     * Get subscriber's post subscriptions
     */
    public static function getPostSubscriptions(int $subscriberId): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';

        $subscriptions = $wpdb->get_results($wpdb->prepare(
            "SELECT ps.id, ps.post_id, ps.post_type, ps.subscription_type, p.post_title
             FROM {$table} ps
             LEFT JOIN {$wpdb->posts} p ON ps.post_id = p.ID
             WHERE ps.subscriber_id = %d AND ps.status = 1
             ORDER BY ps.created_at DESC",
            $subscriberId
        ));

        // Add post type labels
        foreach ($subscriptions as $sub) {
            $post_type_obj = get_post_type_object($sub->post_type);
            $sub->post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $sub->post_type;
        }

        return $subscriptions;
    }

    /**
     * Process subscription updates from the management form
     *
     * @param string $token           Unsubscribe token
     * @param array  $keepLists       List IDs to keep
     * @param array  $keepPosts       Post subscription IDs to keep
     * @param bool   $unsubscribeAll  Whether to unsubscribe from everything
     * @return array Result with success status and message
     */
    public static function processSubscriptionUpdate(
        string $token,
        array $keepLists = [],
        array $keepPosts = [],
        bool $unsubscribeAll = false
    ): array {
        $subscriberId = self::validateToken($token);

        if ($subscriberId === false) {
            return [
                'success' => false,
                'message' => 'Invalid or expired link.',
            ];
        }

        global $wpdb;
        $subscriber = self::getSubscriber($subscriberId);

        if (!$subscriber) {
            return [
                'success' => false,
                'message' => 'Subscriber not found.',
            ];
        }

        // Unsubscribe from everything
        if ($unsubscribeAll) {
            // Update subscriber status
            $wpdb->update(
                $wpdb->prefix . 'outreach_subscribers',
                ['status' => 'unsubscribed', 'updated_at' => current_time('mysql')],
                ['id' => $subscriberId]
            );

            // Deactivate all post subscriptions
            $wpdb->update(
                $wpdb->prefix . 'outreach_post_subscriptions',
                ['status' => 0],
                ['subscriber_id' => $subscriberId]
            );

            // Remove from all lists
            $wpdb->delete(
                $wpdb->prefix . 'outreach_subscriber_list',
                ['subscriber_id' => $subscriberId]
            );

            do_action('wp_outreach_subscriber_unsubscribed', $subscriberId, $subscriber);

            return [
                'success' => true,
                'message' => 'You have been unsubscribed from all communications.',
                'email' => $subscriber->email,
            ];
        }

        // Update list subscriptions - remove unchecked lists
        $currentLists = self::getListSubscriptions($subscriberId);
        $currentListIds = array_map(fn($l) => (int) $l->id, $currentLists);
        $keepListIds = array_map('intval', $keepLists);
        $listsToRemove = array_diff($currentListIds, $keepListIds);

        foreach ($listsToRemove as $listId) {
            $wpdb->delete(
                $wpdb->prefix . 'outreach_subscriber_list',
                ['subscriber_id' => $subscriberId, 'list_id' => $listId]
            );
        }

        // Update post subscriptions - deactivate unchecked ones
        $currentPosts = self::getPostSubscriptions($subscriberId);
        $currentPostIds = array_map(fn($p) => (int) $p->id, $currentPosts);
        $keepPostIds = array_map('intval', $keepPosts);
        $postsToRemove = array_diff($currentPostIds, $keepPostIds);

        foreach ($postsToRemove as $postSubId) {
            $wpdb->update(
                $wpdb->prefix . 'outreach_post_subscriptions',
                ['status' => 0],
                ['id' => $postSubId, 'subscriber_id' => $subscriberId]
            );
        }

        $removedCount = count($listsToRemove) + count($postsToRemove);

        return [
            'success' => true,
            'message' => $removedCount > 0
                ? "Your preferences have been updated. Removed {$removedCount} subscription(s)."
                : 'No changes were made to your subscriptions.',
            'email' => $subscriber->email,
        ];
    }

    /**
     * Generate the full unsubscribe URL for a subscriber
     *
     * @param int    $subscriberId Subscriber ID
     * @param string $email        Subscriber email
     * @return string Full unsubscribe URL
     */
    public static function getUnsubscribeUrl(int $subscriberId, string $email): string
    {
        $token = self::generateToken($subscriberId, $email);

        return rest_url('outreach/v1/unsubscribe') . '?token=' . urlencode($token);
    }

    /**
     * Render the subscription management page
     *
     * @param int    $subscriberId Subscriber ID
     * @param string $token        The unsubscribe token
     * @return string HTML content
     */
    public static function renderManagementPage(int $subscriberId, string $token): string
    {
        $subscriber = self::getSubscriber($subscriberId);
        $lists = self::getListSubscriptions($subscriberId);
        $postSubscriptions = self::getPostSubscriptions($subscriberId);
        $siteName = get_bloginfo('name');
        $actionUrl = rest_url('outreach/v1/unsubscribe');

        $hasSubscriptions = !empty($lists) || !empty($postSubscriptions);

        // Build lists HTML
        $listsHtml = '';
        foreach ($lists as $list) {
            $listsHtml .= '
                <label class="subscription-item">
                    <input type="checkbox" name="keep_lists[]" value="' . esc_attr($list->id) . '" checked>
                    <span class="checkbox-custom"></span>
                    <div class="subscription-info">
                        <span class="subscription-name">' . esc_html($list->name) . '</span>
                        ' . ($list->description ? '<span class="subscription-desc">' . esc_html($list->description) . '</span>' : '') . '
                    </div>
                </label>';
        }

        // Build post subscriptions HTML
        $postsHtml = '';
        foreach ($postSubscriptions as $sub) {
            $typeLabel = $sub->subscription_type === 'both' ? 'New & Updates' : ucfirst($sub->subscription_type) . 's';
            $postsHtml .= '
                <label class="subscription-item">
                    <input type="checkbox" name="keep_posts[]" value="' . esc_attr($sub->id) . '" checked>
                    <span class="checkbox-custom"></span>
                    <div class="subscription-info">
                        <span class="subscription-name">' . esc_html($sub->post_title ?: 'All ' . $sub->post_type_label . 's') . '</span>
                        <span class="subscription-desc">' . esc_html($sub->post_type_label) . ' &bull; ' . esc_html($typeLabel) . '</span>
                    </div>
                </label>';
        }

        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Subscriptions - ' . esc_html($siteName) . '</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: #ffffff;
            border-radius: 16px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            padding: 40px;
            max-width: 500px;
            width: 100%;
        }
        .header {
            text-align: center;
            margin-bottom: 32px;
        }
        .icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 20px;
            border-radius: 50%;
            background: #eef2ff;
            color: #6366f1;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .icon svg {
            width: 32px;
            height: 32px;
        }
        h1 {
            font-size: 24px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 8px;
        }
        .email {
            font-size: 14px;
            color: #6b7280;
        }
        .section {
            margin-bottom: 24px;
        }
        .section-title {
            font-size: 14px;
            font-weight: 600;
            color: #374151;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .section-title svg {
            width: 16px;
            height: 16px;
            color: #6b7280;
        }
        .subscription-list {
            background: #f9fafb;
            border-radius: 12px;
            padding: 8px;
        }
        .subscription-item {
            display: flex;
            align-items: flex-start;
            gap: 12px;
            padding: 12px;
            border-radius: 8px;
            cursor: pointer;
            transition: background 0.15s;
        }
        .subscription-item:hover {
            background: #f3f4f6;
        }
        .subscription-item input[type="checkbox"] {
            display: none;
        }
        .checkbox-custom {
            width: 20px;
            height: 20px;
            min-width: 20px;
            border: 2px solid #d1d5db;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.15s;
            margin-top: 2px;
        }
        .subscription-item input:checked + .checkbox-custom {
            background: #6366f1;
            border-color: #6366f1;
        }
        .checkbox-custom::after {
            content: "";
            width: 10px;
            height: 10px;
            background: url("data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 24 24\' fill=\'none\' stroke=\'white\' stroke-width=\'3\'%3E%3Cpath d=\'M20 6L9 17l-5-5\'/%3E%3C/svg%3E") center/contain no-repeat;
            opacity: 0;
            transition: opacity 0.15s;
        }
        .subscription-item input:checked + .checkbox-custom::after {
            opacity: 1;
        }
        .subscription-info {
            flex: 1;
        }
        .subscription-name {
            display: block;
            font-weight: 500;
            color: #1f2937;
            font-size: 15px;
        }
        .subscription-desc {
            display: block;
            font-size: 13px;
            color: #6b7280;
            margin-top: 2px;
        }
        .empty-message {
            text-align: center;
            padding: 24px;
            color: #6b7280;
        }
        .actions {
            display: flex;
            flex-direction: column;
            gap: 12px;
            margin-top: 24px;
        }
        .btn {
            display: block;
            width: 100%;
            padding: 14px 24px;
            border: none;
            border-radius: 10px;
            font-weight: 500;
            font-size: 15px;
            cursor: pointer;
            text-align: center;
            text-decoration: none;
            transition: all 0.2s;
        }
        .btn-primary {
            background: #6366f1;
            color: #ffffff;
        }
        .btn-primary:hover {
            background: #4f46e5;
        }
        .btn-danger {
            background: #fef2f2;
            color: #dc2626;
            border: 1px solid #fecaca;
        }
        .btn-danger:hover {
            background: #fee2e2;
        }
        .footer {
            margin-top: 32px;
            padding-top: 24px;
            border-top: 1px solid #e5e7eb;
            text-align: center;
            font-size: 13px;
            color: #9ca3af;
        }
        .hint {
            font-size: 13px;
            color: #6b7280;
            text-align: center;
            margin-top: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/>
                    <circle cx="12" cy="12" r="3"/>
                </svg>
            </div>
            <h1>Manage Subscriptions</h1>
            <p class="email">' . esc_html($subscriber->email) . '</p>
        </div>

        <form method="POST" action="' . esc_url($actionUrl) . '">
            <input type="hidden" name="token" value="' . esc_attr($token) . '">
            <input type="hidden" name="action" value="update">

            ' . (!empty($lists) ? '
            <div class="section">
                <div class="section-title">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M19 21l-7-5-7 5V5a2 2 0 012-2h10a2 2 0 012 2z"/>
                    </svg>
                    Mailing Lists
                </div>
                <div class="subscription-list">
                    ' . $listsHtml . '
                </div>
            </div>
            ' : '') . '

            ' . (!empty($postSubscriptions) ? '
            <div class="section">
                <div class="section-title">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"/>
                    </svg>
                    Content Notifications
                </div>
                <div class="subscription-list">
                    ' . $postsHtml . '
                </div>
            </div>
            ' : '') . '

            ' . (!$hasSubscriptions ? '
            <div class="empty-message">
                <p>You don\'t have any active subscriptions.</p>
            </div>
            ' : '') . '

            <div class="actions">
                ' . ($hasSubscriptions ? '
                <button type="submit" class="btn btn-primary">Save Preferences</button>
                <p class="hint">Uncheck items you no longer want to receive</p>
                ' : '') . '
                <button type="submit" name="unsubscribe_all" value="1" class="btn btn-danger" onclick="return confirm(\'Are you sure you want to unsubscribe from all communications?\')">
                    Unsubscribe from Everything
                </button>
            </div>
        </form>

        <div class="footer">
            &copy; ' . date('Y') . ' ' . esc_html($siteName) . '
        </div>
    </div>
</body>
</html>';
    }

    /**
     * Render the confirmation page after processing
     *
     * @param array $result Result from processSubscriptionUpdate
     * @return string HTML content
     */
    public static function renderConfirmationPage(array $result): string
    {
        $siteName = get_bloginfo('name');
        $homeUrl = home_url();

        $statusClass = $result['success'] ? 'success' : 'error';
        $iconSvg = $result['success']
            ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>'
            : '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/></svg>';

        $emailDisplay = isset($result['email']) ? '<p class="email">' . esc_html($result['email']) . '</p>' : '';

        return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscription Updated - ' . esc_html($siteName) . '</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: #ffffff;
            border-radius: 16px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            padding: 48px;
            max-width: 440px;
            width: 100%;
            text-align: center;
        }
        .icon {
            width: 64px;
            height: 64px;
            margin: 0 auto 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .icon svg { width: 32px; height: 32px; }
        .icon.success { background: #d1fae5; color: #059669; }
        .icon.error { background: #fee2e2; color: #dc2626; }
        h1 {
            font-size: 24px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 12px;
        }
        .message {
            font-size: 16px;
            color: #6b7280;
            line-height: 1.6;
            margin-bottom: 8px;
        }
        .email {
            font-size: 14px;
            color: #9ca3af;
            margin-bottom: 24px;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #6366f1;
            color: #ffffff;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 500;
            font-size: 14px;
            transition: background 0.2s;
        }
        .btn:hover { background: #4f46e5; }
        .footer {
            margin-top: 32px;
            padding-top: 24px;
            border-top: 1px solid #e5e7eb;
            font-size: 13px;
            color: #9ca3af;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="icon ' . $statusClass . '">' . $iconSvg . '</div>
        <h1>' . ($result['success'] ? 'Preferences Updated' : 'Error') . '</h1>
        <p class="message">' . esc_html($result['message']) . '</p>
        ' . $emailDisplay . '
        <a href="' . esc_url($homeUrl) . '" class="btn">Return to ' . esc_html($siteName) . '</a>
        <div class="footer">&copy; ' . date('Y') . ' ' . esc_html($siteName) . '</div>
    </div>
</body>
</html>';
    }
}
