<?php

namespace WPOutreach\Tracking;

/**
 * Handles email open tracking
 *
 * Uses a 1x1 transparent tracking pixel to detect when emails are opened.
 * Records opens in the logs table for campaign analytics.
 *
 * @since 1.0.0
 */
class OpenTracker
{
    /**
     * Generate a unique tracking ID
     *
     * @return string 32-character tracking ID
     */
    public static function generateTrackingId(): string
    {
        return bin2hex(random_bytes(16));
    }

    /**
     * Get the tracking pixel URL for a given tracking ID
     *
     * @param string $trackingId The tracking ID
     * @return string Full URL to the tracking pixel
     */
    public static function getPixelUrl(string $trackingId): string
    {
        return rest_url('outreach/v1/track/open') . '?t=' . urlencode($trackingId);
    }

    /**
     * Get the tracking pixel HTML to insert in emails
     *
     * @param string $trackingId The tracking ID
     * @return string HTML img tag
     */
    public static function getPixelHtml(string $trackingId): string
    {
        $url = self::getPixelUrl($trackingId);

        return '<img src="' . esc_url($url) . '" width="1" height="1" alt="" style="display:block;width:1px;height:1px;border:0;" />';
    }

    /**
     * Insert tracking pixel into email HTML content
     *
     * Inserts the pixel just before the closing </body> tag,
     * or at the end if no body tag is found.
     *
     * @param string $content Email HTML content
     * @param string $trackingId The tracking ID
     * @return string Content with tracking pixel inserted
     */
    public static function insertPixel(string $content, string $trackingId): string
    {
        $pixel = self::getPixelHtml($trackingId);

        // Try to insert before </body>
        if (stripos($content, '</body>') !== false) {
            return preg_replace(
                '/<\/body>/i',
                $pixel . '</body>',
                $content,
                1
            );
        }

        // Fallback: append to content
        return $content . $pixel;
    }

    /**
     * Record an email open event
     *
     * @param string $trackingId The tracking ID from the pixel request
     * @return bool True if open was recorded, false if invalid/already tracked unique
     */
    public static function recordOpen(string $trackingId): bool
    {
        global $wpdb;

        $trackingId = sanitize_text_field($trackingId);

        if (empty($trackingId)) {
            return false;
        }

        $logs_table = $wpdb->prefix . 'outreach_logs';

        // Find the log entry for this tracking ID
        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT id, opens, first_opened_at FROM {$logs_table} WHERE tracking_id = %s",
            $trackingId
        ));

        if (!$log) {
            return false;
        }

        $now = current_time('mysql');
        $isFirstOpen = empty($log->first_opened_at);

        // Update the log entry
        $updateData = [
            'opens' => $log->opens + 1,
            'last_opened_at' => $now,
        ];

        // Set first_opened_at if this is the first open
        if ($isFirstOpen) {
            $updateData['first_opened_at'] = $now;
            $updateData['status'] = 'opened';
        }

        $result = $wpdb->update(
            $logs_table,
            $updateData,
            ['id' => $log->id]
        );

        if ($result !== false) {
            // Fire hook for first open only (unique open)
            if ($isFirstOpen) {
                // Get the full log data for the hook
                $fullLog = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$logs_table} WHERE id = %d",
                    $log->id
                ));

                do_action('wp_outreach_email_opened', $fullLog);
            }

            // Fire hook for all opens (total opens)
            do_action('wp_outreach_email_open_recorded', $log->id, $log->opens + 1);

            return true;
        }

        return false;
    }

    /**
     * Create a log entry when an email is sent
     *
     * @param array $data Log data
     * @return int|false Log ID on success, false on failure
     */
    public static function createLogEntry(array $data): int|false
    {
        global $wpdb;

        $logs_table = $wpdb->prefix . 'outreach_logs';

        $result = $wpdb->insert($logs_table, [
            'campaign_id' => $data['campaign_id'] ?? null,
            'automation_id' => $data['automation_id'] ?? null,
            'subscriber_id' => $data['subscriber_id'],
            'email' => $data['email'],
            'subject' => $data['subject'] ?? null,
            'tracking_id' => $data['tracking_id'],
            'status' => 'sent',
            'opens' => 0,
            'clicks' => 0,
            'sent_at' => current_time('mysql'),
        ]);

        if ($result) {
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Get open statistics for a campaign
     *
     * @param int $campaignId Campaign ID
     * @return array{total_sent: int, unique_opens: int, total_opens: int, open_rate: float}
     */
    public static function getCampaignStats(int $campaignId): array
    {
        global $wpdb;

        $logs_table = $wpdb->prefix . 'outreach_logs';

        $stats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total_sent,
                SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as unique_opens,
                SUM(opens) as total_opens
             FROM {$logs_table}
             WHERE campaign_id = %d",
            $campaignId
        ));

        $totalSent = (int) ($stats->total_sent ?? 0);
        $uniqueOpens = (int) ($stats->unique_opens ?? 0);
        $totalOpens = (int) ($stats->total_opens ?? 0);

        return [
            'total_sent' => $totalSent,
            'unique_opens' => $uniqueOpens,
            'total_opens' => $totalOpens,
            'open_rate' => $totalSent > 0 ? round(($uniqueOpens / $totalSent) * 100, 2) : 0,
        ];
    }

    /**
     * Get open statistics for an automation
     *
     * @param int $automationId Automation ID
     * @return array{total_sent: int, unique_opens: int, total_opens: int, open_rate: float}
     */
    public static function getAutomationStats(int $automationId): array
    {
        global $wpdb;

        $logs_table = $wpdb->prefix . 'outreach_logs';

        $stats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total_sent,
                SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as unique_opens,
                SUM(opens) as total_opens
             FROM {$logs_table}
             WHERE automation_id = %d",
            $automationId
        ));

        $totalSent = (int) ($stats->total_sent ?? 0);
        $uniqueOpens = (int) ($stats->unique_opens ?? 0);
        $totalOpens = (int) ($stats->total_opens ?? 0);

        return [
            'total_sent' => $totalSent,
            'unique_opens' => $uniqueOpens,
            'total_opens' => $totalOpens,
            'open_rate' => $totalSent > 0 ? round(($uniqueOpens / $totalSent) * 100, 2) : 0,
        ];
    }
}
