<?php

namespace WPOutreach\Tracking;

/**
 * Handles email click tracking
 *
 * Rewrites links in emails and tracks clicks when users click through.
 *
 * @since 1.0.0
 */
class ClickTracker
{
    /**
     * Generate a hash for a URL
     *
     * @param string $url The original URL
     * @param int $campaignId Campaign ID for uniqueness
     * @return string 16-character hash
     */
    public static function generateLinkHash(string $url, int $campaignId): string
    {
        return substr(md5($url . $campaignId . wp_salt('auth')), 0, 16);
    }

    /**
     * Get the tracking redirect URL for a link
     *
     * @param string $trackingId Email tracking ID
     * @param string $linkHash Link hash
     * @return string Full URL to the click tracker
     */
    public static function getTrackingUrl(string $trackingId, string $linkHash): string
    {
        return rest_url('outreach/v1/track/click') . '?' . http_build_query([
            't' => $trackingId,
            'l' => $linkHash,
        ]);
    }

    /**
     * Store a link and return its hash
     *
     * @param string $url Original URL
     * @param int $campaignId Campaign ID
     * @return string Link hash
     */
    public static function storeLink(string $url, int $campaignId): string
    {
        global $wpdb;

        $linksTable = $wpdb->prefix . 'outreach_links';
        $hash = self::generateLinkHash($url, $campaignId);

        // Check if link already exists
        $existingId = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$linksTable} WHERE campaign_id = %d AND hash = %s",
            $campaignId,
            $hash
        ));

        if (!$existingId) {
            // Insert new link
            $wpdb->insert($linksTable, [
                'campaign_id' => $campaignId,
                'url' => $url,
                'hash' => $hash,
                'clicks' => 0,
            ]);
        }

        return $hash;
    }

    /**
     * Get original URL from hash
     *
     * @param string $hash Link hash
     * @param int|null $campaignId Campaign ID (optional for validation)
     * @return string|null Original URL or null if not found
     */
    public static function getOriginalUrl(string $hash, ?int $campaignId = null): ?string
    {
        global $wpdb;

        $linksTable = $wpdb->prefix . 'outreach_links';

        if ($campaignId !== null) {
            $url = $wpdb->get_var($wpdb->prepare(
                "SELECT url FROM {$linksTable} WHERE hash = %s AND campaign_id = %d",
                $hash,
                $campaignId
            ));
        } else {
            $url = $wpdb->get_var($wpdb->prepare(
                "SELECT url FROM {$linksTable} WHERE hash = %s",
                $hash
            ));
        }

        return $url ?: null;
    }

    /**
     * Rewrite all links in HTML content for tracking
     *
     * @param string $content HTML content
     * @param string $trackingId Email tracking ID
     * @param int $campaignId Campaign ID
     * @return string Content with rewritten links
     */
    public static function rewriteLinks(string $content, string $trackingId, int $campaignId): string
    {
        // Match all href attributes in anchor tags
        $pattern = '/<a\s+([^>]*?)href=["\']([^"\']+)["\']/i';

        return preg_replace_callback($pattern, function ($matches) use ($trackingId, $campaignId) {
            $attributes = $matches[1];
            $url = $matches[2];

            // Skip certain URLs
            if (self::shouldSkipUrl($url)) {
                return $matches[0];
            }

            // Store the link and get its hash
            $linkHash = self::storeLink($url, $campaignId);

            // Generate tracking URL
            $trackingUrl = self::getTrackingUrl($trackingId, $linkHash);

            return '<a ' . $attributes . 'href="' . esc_url($trackingUrl) . '"';
        }, $content);
    }

    /**
     * Check if a URL should be skipped for tracking
     *
     * @param string $url The URL to check
     * @return bool True if should skip
     */
    private static function shouldSkipUrl(string $url): bool
    {
        // Skip empty or invalid URLs
        if (empty($url)) {
            return true;
        }

        // Skip anchors and javascript
        if (str_starts_with($url, '#') || str_starts_with($url, 'javascript:')) {
            return true;
        }

        // Skip mailto and tel links
        if (str_starts_with($url, 'mailto:') || str_starts_with($url, 'tel:')) {
            return true;
        }

        // Skip unsubscribe URLs (already tracked differently)
        if (str_contains($url, '/unsubscribe') || str_contains($url, 'unsubscribe_url')) {
            return true;
        }

        // Skip tracking URLs (prevent double tracking)
        if (str_contains($url, '/track/')) {
            return true;
        }

        return false;
    }

    /**
     * Record a click event
     *
     * @param string $trackingId Email tracking ID
     * @param string $linkHash Link hash
     * @return string|null Original URL to redirect to, or null if invalid
     */
    public static function recordClick(string $trackingId, string $linkHash): ?string
    {
        global $wpdb;

        $trackingId = sanitize_text_field($trackingId);
        $linkHash = sanitize_text_field($linkHash);

        if (empty($trackingId) || empty($linkHash)) {
            return null;
        }

        $linksTable = $wpdb->prefix . 'outreach_links';
        $logsTable = $wpdb->prefix . 'outreach_logs';

        // Get the link
        $link = $wpdb->get_row($wpdb->prepare(
            "SELECT id, url, campaign_id FROM {$linksTable} WHERE hash = %s",
            $linkHash
        ));

        if (!$link) {
            return null;
        }

        // Increment link clicks
        $wpdb->query($wpdb->prepare(
            "UPDATE {$linksTable} SET clicks = clicks + 1 WHERE id = %d",
            $link->id
        ));

        // Find the log entry for this tracking ID
        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT id, clicks, clicked_links FROM {$logsTable} WHERE tracking_id = %s",
            $trackingId
        ));

        if ($log) {
            // Update log entry
            $clickedLinks = json_decode($log->clicked_links, true) ?: [];

            // Track which links were clicked and how many times
            if (!isset($clickedLinks[$linkHash])) {
                $clickedLinks[$linkHash] = 0;
            }
            $clickedLinks[$linkHash]++;

            $isFirstClick = $log->clicks === 0;

            $updateData = [
                'clicks' => $log->clicks + 1,
                'clicked_links' => wp_json_encode($clickedLinks),
            ];

            // Update status to clicked if this is the first click
            if ($isFirstClick) {
                $updateData['status'] = 'clicked';
            }

            $wpdb->update($logsTable, $updateData, ['id' => $log->id]);

            // Fire hook for first click only
            if ($isFirstClick) {
                $fullLog = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$logsTable} WHERE id = %d",
                    $log->id
                ));
                do_action('wp_outreach_email_clicked', $fullLog, $link->url);
            }

            // Fire hook for all clicks
            do_action('wp_outreach_click_recorded', $log->id, $linkHash, $link->url);
        }

        return $link->url;
    }

    /**
     * Get click statistics for a campaign
     *
     * @param int $campaignId Campaign ID
     * @return array{total_sent: int, unique_clicks: int, total_clicks: int, click_rate: float, links: array}
     */
    public static function getCampaignStats(int $campaignId): array
    {
        global $wpdb;

        $logsTable = $wpdb->prefix . 'outreach_logs';
        $linksTable = $wpdb->prefix . 'outreach_links';

        // Get email stats
        $emailStats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total_sent,
                SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END) as unique_clicks,
                SUM(clicks) as total_clicks
             FROM {$logsTable}
             WHERE campaign_id = %d",
            $campaignId
        ));

        // Get link breakdown
        $links = $wpdb->get_results($wpdb->prepare(
            "SELECT url, clicks FROM {$linksTable}
             WHERE campaign_id = %d
             ORDER BY clicks DESC",
            $campaignId
        ));

        $totalSent = (int) ($emailStats->total_sent ?? 0);
        $uniqueClicks = (int) ($emailStats->unique_clicks ?? 0);
        $totalClicks = (int) ($emailStats->total_clicks ?? 0);

        return [
            'total_sent' => $totalSent,
            'unique_clicks' => $uniqueClicks,
            'total_clicks' => $totalClicks,
            'click_rate' => $totalSent > 0 ? round(($uniqueClicks / $totalSent) * 100, 2) : 0,
            'links' => $links,
        ];
    }
}
