<?php

namespace WPOutreach\Subscriptions;

/**
 * Post Subscription Meta Box
 *
 * Adds a "Notify Subscribers" checkbox to the publish panel for enabled post types.
 *
 * @since 1.0.0
 */
class PostSubscriptionMetaBox
{
    /**
     * Settings for post subscriptions
     */
    private array $settings;

    /**
     * Handler instance for subscriber count
     */
    private PostSubscriptionHandler $handler;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->settings = get_option('wp_outreach_post_subscriptions', []);
        $this->handler = new PostSubscriptionHandler();
    }

    /**
     * Initialize WordPress hooks
     */
    public static function init(): void
    {
        $instance = new self();

        // Only proceed if post subscriptions are enabled
        if (empty($instance->settings['enabled'])) {
            return;
        }

        add_action('add_meta_boxes', [$instance, 'registerMetaBox']);
        add_action('save_post', [$instance, 'saveMetaBox'], 5, 2); // Priority 5 to run before notification handler
    }

    /**
     * Register the meta box on enabled post types
     */
    public function registerMetaBox(): void
    {
        $enabledTypes = $this->settings['post_types'] ?? ['post'];

        if (empty($enabledTypes)) {
            return;
        }

        foreach ($enabledTypes as $postType) {
            add_meta_box(
                'wp_outreach_notify_subscribers',
                __('Post Subscriptions', 'outreach'),
                [$this, 'renderMetaBox'],
                $postType,
                'side',
                'default'
            );
        }
    }

    /**
     * Render the meta box content
     *
     * @param \WP_Post $post Current post object
     */
    public function renderMetaBox(\WP_Post $post): void
    {
        // Get subscriber count for this post
        $subscriberCount = $this->handler->getSubscriptionCount($post->ID, $post->post_type);

        // Get the default notify setting
        $defaultNotify = $this->settings['notify_by_default'] ?? true;

        // Get the current value (default to the setting default)
        // This is intentionally NOT persistent - each update starts fresh with the default
        $notify = $defaultNotify;

        // Add nonce for security
        wp_nonce_field('wp_outreach_notify_nonce', 'wp_outreach_notify_nonce');

        ?>
        <div class="outreach-notify-box">
            <style>
                .outreach-notify-box {
                    padding: 8px 0;
                }
                .outreach-notify-checkbox {
                    display: flex;
                    align-items: flex-start;
                    gap: 8px;
                    margin-bottom: 10px;
                }
                .outreach-notify-checkbox input[type="checkbox"] {
                    margin-top: 2px;
                }
                .outreach-notify-label {
                    font-weight: 500;
                    color: #1e1e1e;
                    line-height: 1.4;
                }
                .outreach-subscriber-count {
                    display: flex;
                    align-items: center;
                    gap: 6px;
                    padding: 8px 10px;
                    background: #f0f6fc;
                    border-radius: 4px;
                    font-size: 12px;
                    color: #1e3a5f;
                }
                .outreach-subscriber-count svg {
                    flex-shrink: 0;
                }
                .outreach-subscriber-count.has-subscribers {
                    background: #e7f5e7;
                    color: #1e5631;
                }
                .outreach-subscriber-count.no-subscribers {
                    background: #f5f5f5;
                    color: #666;
                }
                .outreach-notify-help {
                    margin-top: 8px;
                    font-size: 11px;
                    color: #757575;
                    line-height: 1.4;
                }
            </style>

            <label class="outreach-notify-checkbox">
                <input
                    type="checkbox"
                    name="wp_outreach_notify_subscribers"
                    value="1"
                    <?php checked($notify, true); ?>
                />
                <span class="outreach-notify-label">
                    <?php _e('Notify subscribers of this update', 'outreach'); ?>
                </span>
            </label>

            <div class="outreach-subscriber-count <?php echo $subscriberCount > 0 ? 'has-subscribers' : 'no-subscribers'; ?>">
                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"></path>
                    <circle cx="9" cy="7" r="4"></circle>
                    <path d="M23 21v-2a4 4 0 0 0-3-3.87"></path>
                    <path d="M16 3.13a4 4 0 0 1 0 7.75"></path>
                </svg>
                <?php if ($subscriberCount > 0): ?>
                    <?php printf(
                        _n(
                            '%d subscriber will be notified',
                            '%d subscribers will be notified',
                            $subscriberCount,
                            'outreach'
                        ),
                        $subscriberCount
                    ); ?>
                <?php else: ?>
                    <?php _e('No subscribers yet', 'outreach'); ?>
                <?php endif; ?>
            </div>

            <?php if ($post->post_status !== 'publish'): ?>
                <p class="outreach-notify-help">
                    <?php _e('Notifications will be sent when this post is published.', 'outreach'); ?>
                </p>
            <?php else: ?>
                <p class="outreach-notify-help">
                    <?php _e('Uncheck to update without notifying subscribers.', 'outreach'); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Save the meta box value
     *
     * @param int      $postId Post ID
     * @param \WP_Post $post   Post object
     */
    public function saveMetaBox(int $postId, \WP_Post $post): void
    {
        // Security checks
        if (!isset($_POST['wp_outreach_notify_nonce'])) {
            return;
        }

        if (!wp_verify_nonce($_POST['wp_outreach_notify_nonce'], 'wp_outreach_notify_nonce')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $postId)) {
            return;
        }

        // Check if this post type is enabled
        $enabledTypes = $this->settings['post_types'] ?? ['post'];
        if (!in_array($post->post_type, $enabledTypes, true)) {
            return;
        }

        // Get the checkbox value
        $notify = isset($_POST['wp_outreach_notify_subscribers']) ? '1' : '0';

        // Store as transient meta - this will be checked by PostSubscriptionHandler
        // We use a transient-like approach: the value is only valid for this save operation
        update_post_meta($postId, '_wp_outreach_notify_update', $notify);
    }

    /**
     * Check if notifications should be sent for this post update
     *
     * @param int $postId Post ID
     * @return bool True if notifications should be sent
     */
    public static function shouldNotify(int $postId): bool
    {
        $notify = get_post_meta($postId, '_wp_outreach_notify_update', true);

        // If no meta set, use the default from settings
        if ($notify === '') {
            $settings = get_option('wp_outreach_post_subscriptions', []);
            return $settings['notify_by_default'] ?? true;
        }

        return $notify === '1';
    }

    /**
     * Clear the notify flag after processing
     * Call this after notifications are sent to reset for next update
     *
     * @param int $postId Post ID
     */
    public static function clearNotifyFlag(int $postId): void
    {
        delete_post_meta($postId, '_wp_outreach_notify_update');
    }
}
