<?php

namespace WPOutreach\Subscriptions;

/**
 * Post Subscription Injector
 *
 * Auto-injects subscription forms into post content based on admin settings.
 * Supports both "after content" and "floating modal" display modes.
 *
 * @since 1.0.0
 */
class PostSubscriptionInjector
{
    /**
     * Plugin settings
     */
    private static array $settings = [];

    /**
     * Whether assets have been enqueued
     */
    private static bool $assets_enqueued = false;

    /**
     * Initialize the injector
     */
    public static function init(): void
    {
        // Load settings
        self::$settings = get_option('wp_outreach_post_subscriptions', []);

        // Check if feature is enabled
        if (empty(self::$settings['enabled'])) {
            return;
        }

        // Hook into content for "after_content" mode
        add_filter('the_content', [self::class, 'maybeInjectForm'], 99);

        // Hook into footer for "floating_modal" mode
        add_action('wp_footer', [self::class, 'maybeInjectFloatingModal']);

        // Enqueue assets when needed
        add_action('wp_enqueue_scripts', [self::class, 'maybeEnqueueAssets']);
    }

    /**
     * Check if we should inject on the current page
     */
    private static function shouldInject(): bool
    {
        // Must be singular view (single post/page)
        if (!is_singular()) {
            return false;
        }

        // Don't inject in admin or REST API contexts
        if (is_admin() || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST)) {
            return false;
        }

        // Check if current post type is enabled
        $post_type = get_post_type();
        $enabled_post_types = self::$settings['post_types'] ?? ['post'];

        if (!in_array($post_type, $enabled_post_types, true)) {
            return false;
        }

        return true;
    }

    /**
     * Maybe enqueue assets
     */
    public static function maybeEnqueueAssets(): void
    {
        if (!self::shouldInject()) {
            return;
        }

        self::enqueueAssets();
    }

    /**
     * Enqueue form assets
     */
    private static function enqueueAssets(): void
    {
        if (self::$assets_enqueued) {
            return;
        }

        wp_enqueue_style(
            'outreach-form',
            WP_OUTREACH_URL . 'public/css/form.css',
            [],
            WP_OUTREACH_VERSION
        );

        wp_enqueue_script(
            'outreach-post-subscribe',
            WP_OUTREACH_URL . 'public/js/post-subscribe.js',
            [],
            WP_OUTREACH_VERSION,
            true
        );

        wp_localize_script('outreach-post-subscribe', 'wpOutreachPostSubscribe', [
            'apiUrl' => rest_url('outreach/v1/post-subscribe'),
            'nonce' => wp_create_nonce('wp_rest'),
        ]);

        self::$assets_enqueued = true;
    }

    /**
     * Maybe inject form after content
     *
     * @param string $content Post content
     * @return string Modified content
     */
    public static function maybeInjectForm(string $content): string
    {
        // Only inject in main query and singular
        if (!is_main_query() || !in_the_loop()) {
            return $content;
        }

        if (!self::shouldInject()) {
            return $content;
        }

        // Only inject for "after_content" mode
        $display_mode = self::$settings['display_mode'] ?? 'after_content';
        if ($display_mode !== 'after_content') {
            return $content;
        }

        // Make sure assets are enqueued
        self::enqueueAssets();

        // Get current post info
        $post = get_post();
        $post_type_obj = get_post_type_object($post->post_type);
        $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $post->post_type;

        // Get subscriber count if enabled
        $subscriber_count = 0;
        if (!empty(self::$settings['show_subscriber_count'])) {
            $subscriber_count = self::getSubscriberCount($post->ID, $post->post_type);
        }

        // Build the form HTML
        $form_html = self::renderAfterContentForm([
            'post_id' => $post->ID,
            'post_type' => $post->post_type,
            'post_type_label' => $post_type_label,
            'subscriber_count' => $subscriber_count,
        ]);

        return $content . $form_html;
    }

    /**
     * Maybe inject floating modal
     */
    public static function maybeInjectFloatingModal(): void
    {
        if (!self::shouldInject()) {
            return;
        }

        // Only inject for "floating_modal" mode
        $display_mode = self::$settings['display_mode'] ?? 'after_content';
        if ($display_mode !== 'floating_modal') {
            return;
        }

        // Make sure assets are enqueued
        self::enqueueAssets();

        // Get current post info
        $post = get_post();
        $post_type_obj = get_post_type_object($post->post_type);
        $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $post->post_type;

        // Get subscriber count if enabled
        $subscriber_count = 0;
        if (!empty(self::$settings['show_subscriber_count'])) {
            $subscriber_count = self::getSubscriberCount($post->ID, $post->post_type);
        }

        // Output the floating modal HTML
        echo self::renderFloatingModal([
            'post_id' => $post->ID,
            'post_type' => $post->post_type,
            'post_type_label' => $post_type_label,
            'subscriber_count' => $subscriber_count,
        ]);
    }

    /**
     * Get subscriber count for a post or post type
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return int Subscriber count
     */
    private static function getSubscriberCount(int $post_id, string $post_type): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';

        // Count subscribers for this specific post + all posts of this type
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT subscriber_id) FROM {$table}
             WHERE (post_id = %d OR (post_id = 0 AND post_type = %s))
             AND status = 1",
            $post_id,
            $post_type
        ));
    }

    /**
     * Render the after-content form
     *
     * @param array $data Form data
     * @return string HTML output
     */
    private static function renderAfterContentForm(array $data): string
    {
        $heading = self::$settings['heading'] ?? 'Get notified when this content is updated';
        $description = self::$settings['description'] ?? 'Enter your email to receive updates about this post.';
        $button_text = self::$settings['button_text'] ?? 'Notify Me';
        $success_message = self::$settings['success_message'] ?? "You'll be notified when this is updated!";
        $subscription_type = self::$settings['subscription_type'] ?? 'update';

        $form_id = 'wpo-auto-form-' . $data['post_id'];

        $subscriber_html = '';
        if (!empty(self::$settings['show_subscriber_count']) && $data['subscriber_count'] > 0) {
            $subscriber_html = sprintf(
                '<p class="wpo-subscriber-count"><span class="wpo-count">%d</span> %s already subscribed</p>',
                $data['subscriber_count'],
                $data['subscriber_count'] === 1 ? 'person' : 'people'
            );
        }

        ob_start();
        ?>
        <div id="<?php echo esc_attr($form_id); ?>" class="wpo-subscribe-form wpo-post-subscribe-form wpo-auto-inject wpo-after-content">
            <div class="wpo-form-header">
                <h4 class="wpo-form-heading"><?php echo esc_html($heading); ?></h4>
                <?php if ($description): ?>
                <p class="wpo-form-description"><?php echo esc_html($description); ?></p>
                <?php endif; ?>
                <?php echo $subscriber_html; ?>
            </div>
            <form class="wpo-form" data-success-message="<?php echo esc_attr($success_message); ?>">
                <input type="hidden" name="post_id" value="<?php echo esc_attr($data['post_id']); ?>" />
                <input type="hidden" name="post_type" value="<?php echo esc_attr($data['post_type']); ?>" />
                <input type="hidden" name="subscription_type" value="<?php echo esc_attr($subscription_type); ?>" />

                <div class="wpo-form-field">
                    <input
                        type="email"
                        name="email"
                        placeholder="Your email address"
                        class="wpo-form-input"
                        required
                    />
                </div>

                <!-- Honeypot field (hidden) -->
                <div class="wpo-hp-field" aria-hidden="true">
                    <input type="text" name="website" tabindex="-1" autocomplete="off" />
                </div>

                <div class="wpo-form-field">
                    <button type="submit" class="wpo-form-button">
                        <span class="wpo-btn-text"><?php echo esc_html($button_text); ?></span>
                        <span class="wpo-btn-loading" style="display:none;">
                            <svg class="wpo-spinner" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10" stroke-dasharray="32" stroke-linecap="round"/>
                            </svg>
                        </span>
                    </button>
                </div>

                <div class="wpo-form-message" style="display:none;"></div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render the floating modal
     *
     * @param array $data Form data
     * @return string HTML output
     */
    private static function renderFloatingModal(array $data): string
    {
        $float_button_text = self::$settings['float_button_text'] ?? 'Get Updates';
        $float_position = self::$settings['float_position'] ?? 'bottom-right';
        $heading = self::$settings['heading'] ?? 'Get notified when this content is updated';
        $description = self::$settings['description'] ?? 'Enter your email to receive updates about this post.';
        $button_text = self::$settings['button_text'] ?? 'Notify Me';
        $success_message = self::$settings['success_message'] ?? "You'll be notified when this is updated!";
        $subscription_type = self::$settings['subscription_type'] ?? 'update';

        $form_id = 'wpo-float-form-' . $data['post_id'];

        $subscriber_html = '';
        if (!empty(self::$settings['show_subscriber_count']) && $data['subscriber_count'] > 0) {
            $subscriber_html = sprintf(
                '<p class="wpo-subscriber-count"><span class="wpo-count">%d</span> %s already subscribed</p>',
                $data['subscriber_count'],
                $data['subscriber_count'] === 1 ? 'person' : 'people'
            );
        }

        ob_start();
        ?>
        <!-- Floating Button -->
        <button type="button" class="wpo-float-button wpo-float-<?php echo esc_attr($float_position); ?>" onclick="document.getElementById('wpo-float-modal').classList.add('wpo-modal-open')">
            <svg class="wpo-float-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M18 8A6 6 0 0 0 6 8c0 7-3 9-3 9h18s-3-2-3-9"/>
                <path d="M13.73 21a2 2 0 0 1-3.46 0"/>
            </svg>
            <span class="wpo-float-text"><?php echo esc_html($float_button_text); ?></span>
        </button>

        <!-- Modal Overlay -->
        <div id="wpo-float-modal" class="wpo-float-modal">
            <div class="wpo-modal-backdrop" onclick="document.getElementById('wpo-float-modal').classList.remove('wpo-modal-open')"></div>
            <div class="wpo-modal-content">
                <button type="button" class="wpo-modal-close" onclick="document.getElementById('wpo-float-modal').classList.remove('wpo-modal-open')">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M18 6L6 18M6 6l12 12"/>
                    </svg>
                </button>

                <div id="<?php echo esc_attr($form_id); ?>" class="wpo-subscribe-form wpo-post-subscribe-form wpo-auto-inject">
                    <div class="wpo-form-header">
                        <h4 class="wpo-form-heading"><?php echo esc_html($heading); ?></h4>
                        <?php if ($description): ?>
                        <p class="wpo-form-description"><?php echo esc_html($description); ?></p>
                        <?php endif; ?>
                        <?php echo $subscriber_html; ?>
                    </div>
                    <form class="wpo-form" data-success-message="<?php echo esc_attr($success_message); ?>">
                        <input type="hidden" name="post_id" value="<?php echo esc_attr($data['post_id']); ?>" />
                        <input type="hidden" name="post_type" value="<?php echo esc_attr($data['post_type']); ?>" />
                        <input type="hidden" name="subscription_type" value="<?php echo esc_attr($subscription_type); ?>" />

                        <div class="wpo-form-field">
                            <input
                                type="email"
                                name="email"
                                placeholder="Your email address"
                                class="wpo-form-input"
                                required
                            />
                        </div>

                        <!-- Honeypot field (hidden) -->
                        <div class="wpo-hp-field" aria-hidden="true">
                            <input type="text" name="website" tabindex="-1" autocomplete="off" />
                        </div>

                        <div class="wpo-form-field">
                            <button type="submit" class="wpo-form-button">
                                <span class="wpo-btn-text"><?php echo esc_html($button_text); ?></span>
                                <span class="wpo-btn-loading" style="display:none;">
                                    <svg class="wpo-spinner" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                        <circle cx="12" cy="12" r="10" stroke-dasharray="32" stroke-linecap="round"/>
                                    </svg>
                                </span>
                            </button>
                        </div>

                        <div class="wpo-form-message" style="display:none;"></div>
                    </form>
                </div>
            </div>
        </div>

        <style>
            /* Floating Button Styles */
            .wpo-float-button {
                position: fixed;
                z-index: 9999;
                display: flex;
                align-items: center;
                gap: 8px;
                padding: 12px 20px;
                background: linear-gradient(135deg, #6366f1 0%, #4f46e5 100%);
                color: #fff;
                border: none;
                border-radius: 50px;
                font-size: 14px;
                font-weight: 600;
                cursor: pointer;
                box-shadow: 0 4px 20px rgba(99, 102, 241, 0.4);
                transition: transform 0.15s ease, box-shadow 0.15s ease;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            }

            .wpo-float-button:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 24px rgba(99, 102, 241, 0.5);
            }

            .wpo-float-bottom-right {
                bottom: 24px;
                right: 24px;
            }

            .wpo-float-bottom-left {
                bottom: 24px;
                left: 24px;
            }

            .wpo-float-icon {
                width: 20px;
                height: 20px;
            }

            /* Modal Overlay */
            .wpo-float-modal {
                position: fixed;
                inset: 0;
                z-index: 10000;
                display: none;
            }

            .wpo-float-modal.wpo-modal-open {
                display: flex;
                align-items: center;
                justify-content: center;
            }

            .wpo-modal-backdrop {
                position: absolute;
                inset: 0;
                background: rgba(0, 0, 0, 0.5);
                backdrop-filter: blur(4px);
            }

            .wpo-modal-content {
                position: relative;
                background: #fff;
                border-radius: 16px;
                padding: 24px;
                max-width: 420px;
                width: calc(100% - 32px);
                box-shadow: 0 20px 50px rgba(0, 0, 0, 0.3);
                animation: wpo-modal-slide-up 0.2s ease;
            }

            @keyframes wpo-modal-slide-up {
                from {
                    opacity: 0;
                    transform: translateY(20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }

            .wpo-modal-close {
                position: absolute;
                top: 12px;
                right: 12px;
                width: 32px;
                height: 32px;
                display: flex;
                align-items: center;
                justify-content: center;
                background: #f1f5f9;
                border: none;
                border-radius: 8px;
                color: #64748b;
                cursor: pointer;
                transition: background 0.15s ease, color 0.15s ease;
            }

            .wpo-modal-close:hover {
                background: #e2e8f0;
                color: #1e293b;
            }

            .wpo-modal-close svg {
                width: 16px;
                height: 16px;
            }

            /* After Content Form Styles */
            .wpo-after-content {
                margin-top: 40px;
                padding: 24px;
                background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
                border-radius: 16px;
                border: 1px solid #e2e8f0;
            }

            .wpo-form-header {
                margin-bottom: 16px;
            }

            .wpo-form-heading {
                margin: 0 0 8px 0;
                font-size: 18px;
                font-weight: 600;
                color: #1e293b;
            }

            .wpo-form-description {
                margin: 0 0 8px 0;
                font-size: 14px;
                color: #64748b;
            }

            .wpo-subscriber-count {
                margin: 0;
                font-size: 13px;
                color: #6366f1;
            }

            .wpo-subscriber-count .wpo-count {
                font-weight: 600;
            }

            /* Modal form adjustments */
            .wpo-modal-content .wpo-post-subscribe-form {
                padding: 0;
                background: none;
                border: none;
            }

            /* Dark mode for modal */
            @media (prefers-color-scheme: dark) {
                .wpo-modal-content {
                    background: #1e293b;
                }

                .wpo-modal-close {
                    background: #334155;
                    color: #94a3b8;
                }

                .wpo-modal-close:hover {
                    background: #475569;
                    color: #f1f5f9;
                }

                .wpo-after-content {
                    background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%);
                    border-color: #334155;
                }

                .wpo-form-heading {
                    color: #f1f5f9;
                }

                .wpo-form-description {
                    color: #94a3b8;
                }
            }
        </style>

        <script>
            // Close modal on Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    document.getElementById('wpo-float-modal').classList.remove('wpo-modal-open');
                }
            });
        </script>
        <?php
        return ob_get_clean();
    }
}
