<?php

namespace WPOutreach\Subscriptions;

use WPOutreach\Automation\TriggerRegistry;
use WPOutreach\Mailer\Template\VariableParser;

/**
 * Post Subscription Handler
 *
 * Handles user subscriptions to post/page/CPT updates and triggers automations
 * when content is published or updated.
 *
 * @since 1.0.0
 */
class PostSubscriptionHandler
{
    /**
     * Settings for post subscriptions
     */
    private array $settings;
    /**
     * Database table name
     */
    private string $table;

    /**
     * Subscribers table name
     */
    private string $subscribers_table;

    /**
     * Excluded post types
     */
    private array $excluded_post_types = [
        'revision',
        'nav_menu_item',
        'custom_css',
        'customize_changeset',
        'oembed_cache',
        'user_request',
        'wp_block',
        'wp_template',
        'wp_template_part',
        'wp_global_styles',
        'wp_navigation',
    ];

    /**
     * Constructor
     */
    public function __construct()
    {
        global $wpdb;
        $this->table = $wpdb->prefix . 'outreach_post_subscriptions';
        $this->subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $this->settings = get_option('wp_outreach_post_subscriptions', []);
    }

    /**
     * Check if auto-notify is enabled
     */
    private function isAutoNotifyEnabled(): bool
    {
        return !empty($this->settings['auto_notify']);
    }

    /**
     * Get notification subject template
     */
    private function getNotificationSubject(): string
    {
        return $this->settings['notification_subject'] ?? '{{post_title}} has been updated';
    }

    /**
     * Get notification content template
     */
    private function getNotificationContent(): string
    {
        return $this->settings['notification_content'] ?? '';
    }

    /**
     * Initialize WordPress hooks
     */
    public static function init(): void
    {
        $instance = new self();

        // Hook into post status transitions
        add_action('transition_post_status', [$instance, 'onPostStatusChange'], 10, 3);

        // Hook into post updates (for content changes on already-published posts)
        add_action('post_updated', [$instance, 'onPostUpdated'], 10, 3);
    }

    /**
     * Handle post status transitions (for new publish events)
     *
     * @param string   $new_status New post status
     * @param string   $old_status Old post status
     * @param \WP_Post $post       Post object
     */
    public function onPostStatusChange(string $new_status, string $old_status, \WP_Post $post): void
    {
        // Skip excluded post types
        if (in_array($post->post_type, $this->excluded_post_types, true)) {
            return;
        }

        // Only trigger on publish (new or from draft/pending)
        if ($new_status === 'publish' && $old_status !== 'publish') {
            // Check if admin opted out of notifications for this publish
            if (!PostSubscriptionMetaBox::shouldNotify($post->ID)) {
                PostSubscriptionMetaBox::clearNotifyFlag($post->ID);
                return;
            }

            $this->triggerPostPublished($post);
            PostSubscriptionMetaBox::clearNotifyFlag($post->ID);
        }
    }

    /**
     * Handle post content updates (for already-published posts)
     *
     * @param int      $post_id     Post ID
     * @param \WP_Post $post_after  Post object after update
     * @param \WP_Post $post_before Post object before update
     */
    public function onPostUpdated(int $post_id, \WP_Post $post_after, \WP_Post $post_before): void
    {
        // Skip excluded post types
        if (in_array($post_after->post_type, $this->excluded_post_types, true)) {
            return;
        }

        // Check if admin opted out of notifications for this update
        if (!PostSubscriptionMetaBox::shouldNotify($post_id)) {
            // Clear the flag for next time
            PostSubscriptionMetaBox::clearNotifyFlag($post_id);
            return;
        }

        // Only trigger if post was already published and content changed
        if ($post_before->post_status === 'publish' && $post_after->post_status === 'publish') {
            // Check if content actually changed
            $contentChanged = $post_before->post_content !== $post_after->post_content;
            $titleChanged = $post_before->post_title !== $post_after->post_title;

            if ($contentChanged || $titleChanged) {
                $this->triggerPostUpdated($post_after);
                // Clear the notify flag after processing
                PostSubscriptionMetaBox::clearNotifyFlag($post_id);
            }
        }
    }

    /**
     * Trigger post_published for all subscribed users
     *
     * @param \WP_Post $post Post object
     */
    private function triggerPostPublished(\WP_Post $post): void
    {
        $subscribers = $this->getSubscribersForPost($post->ID, $post->post_type, 'new');

        if (empty($subscribers)) {
            return;
        }

        $context = $this->buildPostContext($post);

        // Send auto-notifications if enabled
        if ($this->isAutoNotifyEnabled()) {
            $this->sendNotifications($subscribers, $context, 'published');
        }

        // Fire automation triggers for advanced workflows
        foreach ($subscribers as $subscriber) {
            TriggerRegistry::fire('post_published', (int) $subscriber->id, array_merge(
                $context,
                $this->getSubscriberData($subscriber)
            ));
        }

        do_action('wp_outreach_post_published_triggered', $post, count($subscribers));
    }

    /**
     * Trigger post_updated for all subscribed users
     *
     * @param \WP_Post $post Post object
     */
    private function triggerPostUpdated(\WP_Post $post): void
    {
        $subscribers = $this->getSubscribersForPost($post->ID, $post->post_type, 'update');

        if (empty($subscribers)) {
            return;
        }

        $context = $this->buildPostContext($post);

        // Send auto-notifications if enabled
        if ($this->isAutoNotifyEnabled()) {
            $this->sendNotifications($subscribers, $context, 'updated');
        }

        // Fire automation triggers for advanced workflows
        foreach ($subscribers as $subscriber) {
            TriggerRegistry::fire('post_updated', (int) $subscriber->id, array_merge(
                $context,
                $this->getSubscriberData($subscriber)
            ));
        }

        do_action('wp_outreach_post_updated_triggered', $post, count($subscribers));
    }

    /**
     * Send notification emails to subscribers
     *
     * @param array  $subscribers List of subscriber objects
     * @param array  $context     Post context data
     * @param string $event_type  'published' or 'updated'
     */
    private function sendNotifications(array $subscribers, array $context, string $event_type): void
    {
        global $wpdb;

        $subjectTemplate = $this->getNotificationSubject();
        $contentTemplate = $this->getNotificationContent();

        if (empty($contentTemplate)) {
            return;
        }

        $parser = new VariableParser();
        $queueTable = $wpdb->prefix . 'outreach_queue';

        // Get general settings for from name/email
        $generalSettings = get_option('wp_outreach_general', []);
        $fromName = $generalSettings['from_name'] ?? get_bloginfo('name');
        $fromEmail = $generalSettings['from_email'] ?? get_bloginfo('admin_email');

        // Add site-wide context
        $siteContext = [
            'site_name' => get_bloginfo('name'),
            'site_url' => get_site_url(),
            'current_date' => current_time('Y-m-d'),
            'current_year' => current_time('Y'),
        ];

        foreach ($subscribers as $subscriber) {
            $subscriberData = $this->getSubscriberData($subscriber);

            // Generate unsubscribe URL for this specific post subscription
            $unsubscribeUrl = $this->generateUnsubscribeUrl($subscriber->id, $context['post_id'], $context['post_type']);

            // Merge all context data
            $mergeData = array_merge(
                $context,
                $subscriberData,
                $siteContext,
                ['unsubscribe_url' => $unsubscribeUrl]
            );

            // Parse subject and content
            $subject = $parser->parse($subjectTemplate, $mergeData);
            $content = $parser->parse($contentTemplate, $mergeData);

            // Convert content to HTML (simple markdown-like conversion)
            $htmlContent = $this->convertToHtml($content);

            // Queue the email
            $wpdb->insert($queueTable, [
                'subscriber_id' => $subscriber->id,
                'email' => $subscriber->email,
                'subject' => $subject,
                'content' => $htmlContent,
                'status' => 'pending',
                'source' => 'post_subscription',
                'source_id' => $context['post_id'],
                'created_at' => current_time('mysql'),
            ]);
        }
    }

    /**
     * Generate unsubscribe URL for post subscription
     *
     * @param int    $subscriberId Subscriber ID
     * @param int    $postId       Post ID
     * @param string $postType     Post type
     * @return string Unsubscribe URL
     */
    private function generateUnsubscribeUrl(int $subscriberId, int $postId, string $postType): string
    {
        global $wpdb;

        // Get the subscription token
        $token = $wpdb->get_var($wpdb->prepare(
            "SELECT token FROM {$this->table}
             WHERE subscriber_id = %d AND post_id = %d AND post_type = %s AND status = 1",
            $subscriberId,
            $postId,
            $postType
        ));

        if (!$token) {
            // Fallback to general unsubscribe
            $subscriber = $wpdb->get_row($wpdb->prepare(
                "SELECT token FROM {$this->subscribers_table} WHERE id = %d",
                $subscriberId
            ));
            $token = $subscriber->token ?? '';
            return add_query_arg([
                'wpo_action' => 'unsubscribe',
                'token' => $token,
            ], home_url());
        }

        return add_query_arg([
            'wpo_action' => 'unsubscribe_post',
            'token' => $token,
        ], home_url());
    }

    /**
     * Convert plain text content to HTML
     *
     * @param string $content Plain text content
     * @return string HTML content
     */
    private function convertToHtml(string $content): string
    {
        // Escape HTML
        $content = esc_html($content);

        // Convert markdown-like bold **text**
        $content = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $content);

        // Convert URLs to links
        $content = preg_replace(
            '/(https?:\/\/[^\s<]+)/',
            '<a href="$1">$1</a>',
            $content
        );

        // Convert newlines to <br> and wrap in paragraphs
        $paragraphs = preg_split('/\n\s*\n/', $content);
        $html = '';
        foreach ($paragraphs as $para) {
            $para = trim($para);
            if (!empty($para)) {
                $para = nl2br($para);
                $html .= "<p>{$para}</p>\n";
            }
        }

        // Wrap in basic email template
        return $this->wrapInEmailTemplate($html);
    }

    /**
     * Wrap content in a basic email template
     *
     * @param string $content HTML content
     * @return string Full HTML email
     */
    private function wrapInEmailTemplate(string $content): string
    {
        $generalSettings = get_option('wp_outreach_general', []);
        $companyName = $generalSettings['company_name'] ?? get_bloginfo('name');

        ob_start();
        include WP_OUTREACH_PATH . 'templates/emails/wrapper.php';
        return ob_get_clean();
    }

    /**
     * Build context data from post for merge tags
     *
     * @param \WP_Post $post Post object
     * @return array Context data
     */
    private function buildPostContext(\WP_Post $post): array
    {
        $author = get_userdata($post->post_author);
        $thumbnail_id = get_post_thumbnail_id($post->ID);

        return [
            'post_id'        => $post->ID,
            'post_type'      => $post->post_type,
            'post_title'     => $post->post_title,
            'post_url'       => get_permalink($post->ID),
            'post_excerpt'   => wp_trim_words($post->post_excerpt ?: $post->post_content, 55, '...'),
            'post_content'   => $post->post_content,
            'post_author'    => $author ? $author->display_name : '',
            'post_date'      => get_the_date('', $post),
            'post_thumbnail' => $thumbnail_id ? wp_get_attachment_image_url($thumbnail_id, 'large') : '',
            'post_type_label' => get_post_type_object($post->post_type)->labels->singular_name ?? $post->post_type,
        ];
    }

    /**
     * Get subscriber data for context
     *
     * @param object $subscriber Subscriber row
     * @return array Subscriber data
     */
    private function getSubscriberData(object $subscriber): array
    {
        return [
            'email'      => $subscriber->email,
            'first_name' => $subscriber->first_name ?? '',
            'last_name'  => $subscriber->last_name ?? '',
        ];
    }

    /**
     * Get subscribers who are subscribed to a post or post type
     *
     * @param int    $post_id           Post ID
     * @param string $post_type         Post type
     * @param string $subscription_type 'new', 'update', or 'both'
     * @return array Subscriber objects
     */
    public function getSubscribersForPost(int $post_id, string $post_type, string $subscription_type = 'both'): array
    {
        global $wpdb;

        // Get subscribers who:
        // 1. Subscribed to this specific post (post_id = $post_id)
        // 2. Subscribed to all posts of this type (post_id = 0, post_type = $post_type)
        // 3. Have matching subscription_type ('both' or matching type)

        $type_condition = $subscription_type === 'both'
            ? "ps.subscription_type IN ('both', 'new', 'update')"
            : "ps.subscription_type IN ('both', %s)";

        $sql = "SELECT DISTINCT s.*
                FROM {$this->subscribers_table} s
                INNER JOIN {$this->table} ps ON s.id = ps.subscriber_id
                WHERE ps.status = 1
                AND s.status = 'active'
                AND (
                    (ps.post_id = %d)
                    OR (ps.post_id = 0 AND ps.post_type = %s)
                )";

        if ($subscription_type !== 'both') {
            $sql .= " AND {$type_condition}";
            return $wpdb->get_results($wpdb->prepare($sql, $post_id, $post_type, $subscription_type));
        }

        return $wpdb->get_results($wpdb->prepare($sql, $post_id, $post_type));
    }

    /**
     * Subscribe a user to post updates
     *
     * @param int    $subscriber_id    Subscriber ID
     * @param int    $post_id          Post ID (0 for all posts of type)
     * @param string $post_type        Post type
     * @param string $subscription_type 'new', 'update', or 'both'
     * @return int|false Insert ID or false on failure
     */
    public function subscribe(int $subscriber_id, int $post_id, string $post_type, string $subscription_type = 'both'): int|false
    {
        global $wpdb;

        // Check if already subscribed
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->table}
             WHERE subscriber_id = %d AND post_id = %d AND post_type = %s",
            $subscriber_id,
            $post_id,
            $post_type
        ));

        if ($existing) {
            // Update existing subscription
            $wpdb->update(
                $this->table,
                [
                    'subscription_type' => $subscription_type,
                    'status' => 1,
                ],
                ['id' => $existing]
            );
            return (int) $existing;
        }

        // Create new subscription
        $token = bin2hex(random_bytes(16));

        $result = $wpdb->insert($this->table, [
            'subscriber_id'     => $subscriber_id,
            'post_id'           => $post_id,
            'post_type'         => $post_type,
            'subscription_type' => $subscription_type,
            'token'             => $token,
            'status'            => 1,
            'created_at'        => current_time('mysql'),
        ]);

        if ($result) {
            do_action('wp_outreach_post_subscription_created', $subscriber_id, [
                'subscription_id'   => $wpdb->insert_id,
                'post_id'           => $post_id,
                'post_type'         => $post_type,
                'subscription_type' => $subscription_type,
            ]);
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Unsubscribe a user from post updates
     *
     * @param int    $subscriber_id Subscriber ID
     * @param int    $post_id       Post ID
     * @param string $post_type     Post type
     * @return bool True on success
     */
    public function unsubscribe(int $subscriber_id, int $post_id, string $post_type): bool
    {
        global $wpdb;

        $result = $wpdb->update(
            $this->table,
            ['status' => 0],
            [
                'subscriber_id' => $subscriber_id,
                'post_id'       => $post_id,
                'post_type'     => $post_type,
            ]
        );

        if ($result !== false) {
            do_action('wp_outreach_post_subscription_removed', $subscriber_id, $post_id, $post_type);
            return true;
        }

        return false;
    }

    /**
     * Unsubscribe via token (for email links)
     *
     * @param string $token Subscription token
     * @return bool True on success
     */
    public function unsubscribeByToken(string $token): bool
    {
        global $wpdb;

        $subscription = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table} WHERE token = %s AND status = 1",
            $token
        ));

        if (!$subscription) {
            return false;
        }

        return $this->unsubscribe(
            (int) $subscription->subscriber_id,
            (int) $subscription->post_id,
            $subscription->post_type
        );
    }

    /**
     * Check if a subscriber is subscribed to a post
     *
     * @param int    $subscriber_id Subscriber ID
     * @param int    $post_id       Post ID
     * @param string $post_type     Post type
     * @return bool
     */
    public function isSubscribed(int $subscriber_id, int $post_id, string $post_type): bool
    {
        global $wpdb;

        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->table}
             WHERE subscriber_id = %d
             AND (post_id = %d OR (post_id = 0 AND post_type = %s))
             AND status = 1",
            $subscriber_id,
            $post_id,
            $post_type
        ));

        return (bool) $exists;
    }

    /**
     * Get all subscriptions for a subscriber
     *
     * @param int $subscriber_id Subscriber ID
     * @return array Subscription objects
     */
    public function getSubscriberSubscriptions(int $subscriber_id): array
    {
        global $wpdb;

        return $wpdb->get_results($wpdb->prepare(
            "SELECT ps.*, p.post_title
             FROM {$this->table} ps
             LEFT JOIN {$wpdb->posts} p ON ps.post_id = p.ID
             WHERE ps.subscriber_id = %d AND ps.status = 1
             ORDER BY ps.created_at DESC",
            $subscriber_id
        ));
    }

    /**
     * Get subscription count for a post
     *
     * @param int    $post_id   Post ID
     * @param string $post_type Post type
     * @return int
     */
    public function getSubscriptionCount(int $post_id, string $post_type): int
    {
        global $wpdb;

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT ps.subscriber_id)
             FROM {$this->table} ps
             INNER JOIN {$this->subscribers_table} s ON ps.subscriber_id = s.id
             WHERE ps.status = 1
             AND s.status = 'active'
             AND (
                 ps.post_id = %d
                 OR (ps.post_id = 0 AND ps.post_type = %s)
             )",
            $post_id,
            $post_type
        ));
    }

    /**
     * Get available post types for subscription
     *
     * @return array Post types with labels
     */
    public static function getAvailablePostTypes(): array
    {
        $post_types = get_post_types(['public' => true], 'objects');
        $result = [];

        foreach ($post_types as $post_type) {
            if ($post_type->name === 'attachment') {
                continue;
            }

            $result[] = [
                'value' => $post_type->name,
                'label' => $post_type->labels->singular_name,
            ];
        }

        return $result;
    }
}
