<?php

namespace WPOutreach\Mailer;

use WPOutreach\Mailer\SESMailer;

/**
 * Override WordPress wp_mail() to use WP Outreach mail settings
 *
 * When enabled, all WordPress emails (password resets, notifications, etc.)
 * will be sent using the configured WP Outreach mail driver (SMTP or SES).
 */
class WPMailOverride
{
    /**
     * @var bool Whether logging is enabled
     */
    private static bool $logging_enabled = false;

    /**
     * Initialize the override if enabled
     */
    public static function init(): void
    {
        $mailer_settings = get_option('wp_outreach_mailer', []);

        // Check if override is enabled
        if (empty($mailer_settings['handle_all_wp_mail'])) {
            return;
        }

        // Enable logging for WordPress emails
        self::$logging_enabled = true;
        add_action('wp_mail_succeeded', [self::class, 'log_success'], 10, 1);
        add_action('wp_mail_failed', [self::class, 'log_failure'], 10, 1);

        $driver = $mailer_settings['driver'] ?? 'wp_mail';

        // No need to override PHPMailer if using default wp_mail
        if ($driver === 'wp_mail') {
            return;
        }

        // For SES, use the API directly instead of SMTP (more reliable)
        if ($driver === 'ses') {
            add_filter('pre_wp_mail', [self::class, 'send_via_ses'], 10, 2);
            return;
        }

        // For SMTP, hook into PHPMailer initialization
        add_action('phpmailer_init', [self::class, 'configure_phpmailer'], 10, 1);
    }

    /**
     * Send email via SES API directly (bypasses PHPMailer)
     *
     * @param null|bool $return Short-circuit return value
     * @param array $atts Email attributes (to, subject, message, headers, attachments)
     * @return bool|null True if sent, false if failed, null to continue with wp_mail
     */
    public static function send_via_ses($return, $atts): ?bool
    {
        // Skip if this is from WP Outreach queue (already handled)
        if (!empty($atts['headers']) && is_array($atts['headers'])) {
            foreach ($atts['headers'] as $header) {
                if (stripos($header, 'X-WP-Outreach') !== false) {
                    return null; // Let wp_mail handle it
                }
            }
        }

        $sesMailer = new SESMailer();
        if (!$sesMailer->isConfigured()) {
            return null; // Fall back to wp_mail
        }

        $general = get_option('wp_outreach_general', []);
        $from_email = $general['from_email'] ?? get_option('admin_email');
        $from_name = $general['from_name'] ?? get_option('blogname');

        // Parse recipients
        $to = is_array($atts['to']) ? $atts['to'] : [$atts['to']];

        // Build HTML message
        $message = $atts['message'] ?? '';
        $is_html = stripos($message, '<html') !== false || stripos($message, '<body') !== false || stripos($message, '<p>') !== false;

        if (!$is_html) {
            $message = nl2br(esc_html($message));
        }

        // Parse headers for Reply-To, CC, etc.
        $reply_to = null;
        $headers = $atts['headers'] ?? [];
        if (is_string($headers)) {
            $headers = explode("\n", str_replace("\r\n", "\n", $headers));
        }
        foreach ($headers as $header) {
            if (stripos($header, 'Reply-To:') === 0) {
                $reply_to = trim(substr($header, 9));
            }
        }

        try {
            $options = [
                'from_email' => $from_email,
                'from_name' => $from_name,
            ];
            if ($reply_to) {
                $options['reply_to'] = $reply_to;
            }

            foreach ($to as $recipient) {
                $recipient_email = self::extract_email($recipient);
                if (empty($recipient_email)) {
                    continue;
                }

                $success = $sesMailer->send(
                    $recipient_email,
                    $atts['subject'] ?? '(No subject)',
                    $message,
                    $options
                );

                if (!$success) {
                    // Log the failure
                    self::log_ses_failure($recipient_email, $atts['subject'] ?? '', $sesMailer->getLastError() ?? 'Unknown SES error');
                    return false;
                }

                // Log success
                self::log_ses_success($recipient_email, $atts['subject'] ?? '');
            }

            return true;
        } catch (\Exception $e) {
            // Log and fall back to wp_mail
            error_log('WP Outreach SES Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Log successful SES send
     */
    private static function log_ses_success(string $email, string $subject): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        $subscriber_id = self::get_subscriber_id($email);

        $insert_data = [
            'type' => 'wordpress',
            'email' => $email,
            'subject' => $subject,
            'status' => 'sent',
            'sent_at' => current_time('mysql'),
        ];

        if ($subscriber_id !== null) {
            $insert_data['subscriber_id'] = $subscriber_id;
        }

        $wpdb->insert($table, $insert_data);
    }

    /**
     * Log failed SES send
     */
    private static function log_ses_failure(string $email, string $subject, string $error): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        $subscriber_id = self::get_subscriber_id($email);

        $insert_data = [
            'type' => 'wordpress',
            'email' => $email,
            'subject' => $subject,
            'status' => 'bounced',
            'error' => $error,
            'sent_at' => current_time('mysql'),
        ];

        if ($subscriber_id !== null) {
            $insert_data['subscriber_id'] = $subscriber_id;
        }

        $wpdb->insert($table, $insert_data);
    }

    /**
     * Configure PHPMailer with WP Outreach settings (for SMTP only)
     *
     * @param \PHPMailer\PHPMailer\PHPMailer $phpmailer
     */
    public static function configure_phpmailer($phpmailer): void
    {
        $mailer_settings = get_option('wp_outreach_mailer', []);
        $driver = $mailer_settings['driver'] ?? 'wp_mail';

        if ($driver === 'smtp') {
            self::configure_smtp($phpmailer);
        }

        // Apply general settings (from name/email)
        self::apply_general_settings($phpmailer);
    }

    /**
     * Configure PHPMailer for SMTP
     */
    private static function configure_smtp($phpmailer): void
    {
        $smtp = get_option('wp_outreach_smtp', []);

        if (empty($smtp['host']) || empty($smtp['port'])) {
            return;
        }

        $phpmailer->isSMTP();
        $phpmailer->Host = $smtp['host'];
        $phpmailer->Port = (int) $smtp['port'];

        if (!empty($smtp['username']) && !empty($smtp['password'])) {
            $phpmailer->SMTPAuth = true;
            $phpmailer->Username = $smtp['username'];
            $phpmailer->Password = $smtp['password'];
        }

        $encryption = $smtp['encryption'] ?? 'tls';
        if ($encryption === 'tls') {
            $phpmailer->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
        } elseif ($encryption === 'ssl') {
            $phpmailer->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
        } else {
            $phpmailer->SMTPSecure = '';
        }

        $phpmailer->Timeout = 30;
    }

    /**
     * Apply general email settings (from name/email)
     * Note: This is only called for SMTP driver. SES uses the API directly.
     */
    private static function apply_general_settings($phpmailer): void
    {
        $general = get_option('wp_outreach_general', []);

        if (!empty($general['from_email'])) {
            $phpmailer->From = $general['from_email'];
            // Also set Sender (envelope from) to match for SPF alignment
            $phpmailer->Sender = $general['from_email'];
        }

        if (!empty($general['from_name'])) {
            $phpmailer->FromName = $general['from_name'];
        }
    }

    /**
     * Log successful email send
     *
     * @param array $mail_data Email data from wp_mail_succeeded action
     */
    public static function log_success(array $mail_data): void
    {
        if (!self::$logging_enabled) {
            return;
        }

        // Skip logging if this is from WP Outreach queue (already logged there)
        if (!empty($mail_data['headers']) && is_array($mail_data['headers'])) {
            foreach ($mail_data['headers'] as $header) {
                if (stripos($header, 'X-WP-Outreach') !== false) {
                    return;
                }
            }
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        // Handle multiple recipients
        $recipients = is_array($mail_data['to']) ? $mail_data['to'] : [$mail_data['to']];

        foreach ($recipients as $to) {
            // Extract email from "Name <email>" format if needed
            $email = self::extract_email($to);
            if (empty($email)) {
                continue;
            }

            // Check if this email belongs to a subscriber
            $subscriber_id = self::get_subscriber_id($email);

            $insert_data = [
                'type' => 'wordpress',
                'email' => $email,
                'subject' => $mail_data['subject'] ?? '',
                'status' => 'sent',
                'sent_at' => current_time('mysql'),
            ];

            // Only include subscriber_id if it exists
            if ($subscriber_id !== null) {
                $insert_data['subscriber_id'] = $subscriber_id;
            }

            $wpdb->insert($table, $insert_data);
        }
    }

    /**
     * Log failed email send
     *
     * @param \WP_Error $error Error object from wp_mail_failed action
     */
    public static function log_failure($error): void
    {
        if (!self::$logging_enabled) {
            return;
        }

        if (!($error instanceof \WP_Error)) {
            return;
        }

        $mail_data = $error->get_error_data();
        if (empty($mail_data) || !is_array($mail_data)) {
            return;
        }

        // Skip logging if this is from WP Outreach queue
        if (!empty($mail_data['headers']) && is_array($mail_data['headers'])) {
            foreach ($mail_data['headers'] as $header) {
                if (stripos($header, 'X-WP-Outreach') !== false) {
                    return;
                }
            }
        }

        // Extract error message
        $error_message = $error->get_error_message();
        $error_code = $error->get_error_code();
        $full_error = $error_code ? "[{$error_code}] {$error_message}" : $error_message;

        // Check for PHPMailer debug info in the error data
        if (!empty($mail_data['phpmailer_exception_code'])) {
            $full_error .= " (PHPMailer code: {$mail_data['phpmailer_exception_code']})";
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        $recipients = isset($mail_data['to']) ? (is_array($mail_data['to']) ? $mail_data['to'] : [$mail_data['to']]) : [];

        foreach ($recipients as $to) {
            $email = self::extract_email($to);
            if (empty($email)) {
                continue;
            }

            $subscriber_id = self::get_subscriber_id($email);

            $insert_data = [
                'type' => 'wordpress',
                'email' => $email,
                'subject' => $mail_data['subject'] ?? '',
                'status' => 'bounced',
                'error' => $full_error,
                'sent_at' => current_time('mysql'),
            ];

            // Only include subscriber_id if it exists
            if ($subscriber_id !== null) {
                $insert_data['subscriber_id'] = $subscriber_id;
            }

            $wpdb->insert($table, $insert_data);
        }
    }

    /**
     * Extract email from "Name <email>" format
     */
    private static function extract_email(string $to): string
    {
        $to = trim($to);

        // Check for "Name <email>" format
        if (preg_match('/<([^>]+)>/', $to, $matches)) {
            return sanitize_email($matches[1]);
        }

        return sanitize_email($to);
    }

    /**
     * Get subscriber ID by email if exists
     */
    private static function get_subscriber_id(string $email): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s LIMIT 1",
            $email
        ));

        return $id ? (int) $id : null;
    }
}
