<?php

namespace WPOutreach\Mailer\Template;

/**
 * Template variable parser
 *
 * Parses {{variable}} and {variable} placeholders in email templates
 * Supports fallback values: {{first_name|Subscriber}}
 */
class VariableParser
{
    /**
     * @var string Variable pattern - matches both {{var}} and {var}
     * Also supports fallback: {{var|fallback}} or {var|fallback}
     */
    private string $pattern = '/\{\{?\s*([a-z_\.]+)(?:\s*\|\s*([^}]*))?\s*\}?\}/i';

    /**
     * @var array Registered custom variables
     */
    private static array $customVariables = [];

    /**
     * Parse variables in content
     *
     * @param string $content Content with variables
     * @param array $data Data to replace variables with
     * @return string Parsed content
     */
    public function parse(string $content, array $data = []): string
    {
        // Add system variables
        $data = array_merge($this->getSystemVariables(), $data);

        // Add computed variables based on data
        $data = $this->addComputedVariables($data);

        // Add custom registered variables
        foreach (self::$customVariables as $key => $callback) {
            if (!isset($data[$key]) && is_callable($callback)) {
                $data[$key] = $callback($data);
            }
        }

        // Replace variables - supports both {{var}} and {var} syntax
        // Also supports fallback: {{first_name|Friend}}
        return preg_replace_callback($this->pattern, function ($matches) use ($data) {
            $key = $matches[1];
            $fallback = $matches[2] ?? '';
            return $this->getValue($key, $data, trim($fallback));
        }, $content);
    }

    /**
     * Add computed variables derived from base data
     */
    private function addComputedVariables(array $data): array
    {
        // Full name from first + last
        if (!isset($data['full_name'])) {
            $firstName = $data['first_name'] ?? '';
            $lastName = $data['last_name'] ?? '';
            $data['full_name'] = trim($firstName . ' ' . $lastName);
        }

        // Name (alias for first_name or full_name)
        if (!isset($data['name'])) {
            $data['name'] = $data['first_name'] ?? $data['full_name'] ?? '';
        }

        // Email username (part before @)
        if (!isset($data['email_username']) && !empty($data['email'])) {
            $data['email_username'] = strstr($data['email'], '@', true);
        }

        // Subscriber ID alias
        if (!isset($data['subscriber_id']) && isset($data['id'])) {
            $data['subscriber_id'] = $data['id'];
        }

        return $data;
    }

    /**
     * Get value for a variable key (supports dot notation and fallback)
     *
     * @param string $key Variable key (e.g., "subscriber.first_name")
     * @param array $data Data array
     * @param string $fallback Fallback value if variable is empty
     * @return string Value, fallback, or empty string
     */
    private function getValue(string $key, array $data, string $fallback = ''): string
    {
        $value = null;

        // Direct key match
        if (isset($data[$key])) {
            $value = $data[$key];
        } else {
            // Dot notation support (e.g., subscriber.email)
            $parts = explode('.', $key);
            $value = $data;

            foreach ($parts as $part) {
                if (is_array($value) && isset($value[$part])) {
                    $value = $value[$part];
                } elseif (is_object($value) && isset($value->$part)) {
                    $value = $value->$part;
                } else {
                    $value = null;
                    break;
                }
            }
        }

        $formatted = $this->formatValue($value);

        // Return fallback if value is empty
        if ($formatted === '' && $fallback !== '') {
            return $fallback;
        }

        return $formatted;
    }

    /**
     * Format value for output
     */
    private function formatValue($value): string
    {
        if (is_array($value) || is_object($value)) {
            return '';
        }

        return (string) $value;
    }

    /**
     * Get system variables
     */
    private function getSystemVariables(): array
    {
        return [
            'site_name' => get_bloginfo('name'),
            'site_url' => home_url(),
            'site_description' => get_bloginfo('description'),
            'admin_email' => get_option('admin_email'),
            'current_date' => wp_date(get_option('date_format')),
            'current_time' => wp_date(get_option('time_format')),
            'current_year' => wp_date('Y'),
            'current_month' => wp_date('F'),
        ];
    }

    /**
     * Extract all variables from content
     *
     * @param string $content Content to scan
     * @return array List of variable names
     */
    public function extractVariables(string $content): array
    {
        preg_match_all($this->pattern, $content, $matches);
        return array_unique($matches[1] ?? []);
    }

    /**
     * Register a custom variable
     *
     * @param string $name Variable name
     * @param callable $callback Callback to generate value
     */
    public static function register(string $name, callable $callback): void
    {
        self::$customVariables[$name] = $callback;
    }

    /**
     * Get all available variable names for documentation
     */
    public static function getAvailableVariables(): array
    {
        return [
            // Subscriber variables
            'subscriber' => [
                'label' => __('Subscriber', 'outreach'),
                'variables' => [
                    ['tag' => '{{email}}', 'description' => __('Email address', 'outreach')],
                    ['tag' => '{{first_name}}', 'description' => __('First name', 'outreach')],
                    ['tag' => '{{last_name}}', 'description' => __('Last name', 'outreach')],
                    ['tag' => '{{full_name}}', 'description' => __('Full name (first + last)', 'outreach')],
                    ['tag' => '{{name}}', 'description' => __('Name (first name or full name)', 'outreach')],
                    ['tag' => '{{email_username}}', 'description' => __('Email username (before @)', 'outreach')],
                ],
            ],

            // Personalization with fallbacks
            'personalization' => [
                'label' => __('Personalization', 'outreach'),
                'variables' => [
                    ['tag' => '{{first_name|Friend}}', 'description' => __('First name with fallback', 'outreach')],
                    ['tag' => '{{name|Subscriber}}', 'description' => __('Name with fallback', 'outreach')],
                ],
            ],

            // Site variables
            'site' => [
                'label' => __('Site', 'outreach'),
                'variables' => [
                    ['tag' => '{{site_name}}', 'description' => __('Website name', 'outreach')],
                    ['tag' => '{{site_url}}', 'description' => __('Website URL', 'outreach')],
                    ['tag' => '{{site_description}}', 'description' => __('Website tagline', 'outreach')],
                ],
            ],

            // Date/Time variables
            'date' => [
                'label' => __('Date & Time', 'outreach'),
                'variables' => [
                    ['tag' => '{{current_date}}', 'description' => __('Current date', 'outreach')],
                    ['tag' => '{{current_time}}', 'description' => __('Current time', 'outreach')],
                    ['tag' => '{{current_year}}', 'description' => __('Current year', 'outreach')],
                    ['tag' => '{{current_month}}', 'description' => __('Current month name', 'outreach')],
                ],
            ],

            // Link variables
            'links' => [
                'label' => __('Links', 'outreach'),
                'variables' => [
                    ['tag' => '{{unsubscribe_url}}', 'description' => __('Unsubscribe link', 'outreach')],
                    ['tag' => '{{preferences_url}}', 'description' => __('Preferences page link', 'outreach')],
                ],
            ],

            // Post/Content variables (for post_published/post_updated automations)
            'content' => [
                'label' => __('Content', 'outreach'),
                'variables' => [
                    ['tag' => '{{post_title}}', 'description' => __('Post/page title', 'outreach')],
                    ['tag' => '{{post_url}}', 'description' => __('Post/page URL', 'outreach')],
                    ['tag' => '{{post_excerpt}}', 'description' => __('Post excerpt (55 words)', 'outreach')],
                    ['tag' => '{{post_content}}', 'description' => __('Full post content', 'outreach')],
                    ['tag' => '{{post_author}}', 'description' => __('Post author name', 'outreach')],
                    ['tag' => '{{post_date}}', 'description' => __('Post publish date', 'outreach')],
                    ['tag' => '{{post_thumbnail}}', 'description' => __('Featured image URL', 'outreach')],
                    ['tag' => '{{post_type}}', 'description' => __('Post type slug', 'outreach')],
                    ['tag' => '{{post_type_label}}', 'description' => __('Post type label (e.g., "Post", "Page")', 'outreach')],
                    ['tag' => '{{post_id}}', 'description' => __('Post ID', 'outreach')],
                ],
            ],
        ];
    }
}
