<?php

namespace WPOutreach\Mailer\Template;

/**
 * Email template engine
 *
 * Renders email templates with variable substitution and layout wrapping
 */
class TemplateEngine
{
    /**
     * @var VariableParser Variable parser instance
     */
    private VariableParser $parser;

    /**
     * @var string Templates directory
     */
    private string $templatesDir;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->parser = new VariableParser();
        $this->templatesDir = WP_OUTREACH_PATH . 'templates/emails/';
    }

    /**
     * Render content with variables
     *
     * @param string $content Raw content
     * @param array $data Variable data
     * @param bool $wrap Wrap in email layout (auto-detects if content already has HTML structure)
     * @return string Rendered HTML
     */
    public function render(string $content, array $data = [], bool $wrap = true): string
    {
        // Parse variables
        $content = $this->parseVariables($content, $data);

        // Wrap in email layout if requested AND content doesn't already have HTML structure
        if ($wrap && !$this->hasHtmlStructure($content)) {
            $content = $this->wrapInLayout($content, $data);
        }

        return $content;
    }

    /**
     * Check if content already has complete HTML structure
     *
     * @param string $content Content to check
     * @return bool True if content has HTML structure
     */
    private function hasHtmlStructure(string $content): bool
    {
        $content = trim($content);

        // Check for common HTML document indicators
        return (
            stripos($content, '<!DOCTYPE') !== false ||
            stripos($content, '<html') !== false ||
            // Also check for email-wrapper class which is used in our templates
            stripos($content, 'class="email-wrapper"') !== false ||
            stripos($content, "class='email-wrapper'") !== false
        );
    }

    /**
     * Render content for a specific subscriber
     *
     * @param string $content Raw content
     * @param int $subscriberId Subscriber ID
     * @param bool $wrap Wrap in email layout
     * @return string Rendered HTML
     */
    public function renderForSubscriber(string $content, int $subscriberId, bool $wrap = true): string
    {
        $subscriberData = $this->getSubscriberData($subscriberId);
        return $this->render($content, $subscriberData, $wrap);
    }

    /**
     * Parse variables in content
     *
     * @param string $content Content with variables
     * @param array $data Variable data
     * @return string Parsed content
     */
    public function parseVariables(string $content, array $data = []): string
    {
        // Add link variables
        $data = array_merge($data, $this->generateLinkVariables($data));

        return $this->parser->parse($content, $data);
    }

    /**
     * Get subscriber data
     */
    private function getSubscriberData(int $subscriberId): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_subscribers';
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $subscriberId
        ), ARRAY_A);

        if (!$subscriber) {
            return [];
        }

        // Parse custom fields JSON
        if (!empty($subscriber['custom_fields'])) {
            $customFields = json_decode($subscriber['custom_fields'], true) ?: [];
            $subscriber = array_merge($subscriber, $customFields);
        }

        return $subscriber;
    }

    /**
     * Generate link variables for subscriber
     */
    private function generateLinkVariables(array $data): array
    {
        $subscriberId = (int) ($data['id'] ?? $data['subscriber_id'] ?? 0);
        $email = $data['email'] ?? '';

        // Generate proper unsubscribe URL with secure token
        $unsubscribeUrl = '#';
        if ($subscriberId > 0 && !empty($email)) {
            $unsubscribeUrl = \WPOutreach\Tracking\UnsubscribeHandler::getUnsubscribeUrl($subscriberId, $email);
        }

        // Preferences URL (to be implemented)
        $preferencesUrl = '#';

        return [
            'unsubscribe_url' => $unsubscribeUrl,
            'preferences_url' => $preferencesUrl,
            'view_online_url' => '#', // Would need campaign ID
        ];
    }

    /**
     * Wrap content in email layout
     *
     * @param string $content Email body content
     * @param array $data Additional data
     * @return string Complete email HTML
     */
    public function wrapInLayout(string $content, array $data = []): string
    {
        $settings = get_option('wp_outreach_general', []);

        $companyName = $settings['company_name'] ?? get_bloginfo('name');
        $companyAddress = $settings['company_address'] ?? '';

        $unsubscribeUrl = $data['unsubscribe_url'] ?? '#';
        $preferencesUrl = $data['preferences_url'] ?? '#';

        // Default email layout
        $layout = $this->getDefaultLayout();

        // Replace layout variables
        $layout = str_replace(
            [
                '{{content}}',
                '{{company_name}}',
                '{{company_address}}',
                '{{unsubscribe_url}}',
                '{{preferences_url}}',
                '{{current_year}}',
            ],
            [
                $content,
                esc_html($companyName),
                nl2br(esc_html($companyAddress)),
                esc_url($unsubscribeUrl),
                esc_url($preferencesUrl),
                wp_date('Y'),
            ],
            $layout
        );

        return $layout;
    }

    /**
     * Get default email layout
     */
    private function getDefaultLayout(): string
    {
        // Check for custom layout file
        $customLayout = $this->templatesDir . 'layout.html';
        if (file_exists($customLayout)) {
            return file_get_contents($customLayout);
        }

        // Default minimal layout
        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <title></title>
    <style>
        body { margin: 0; padding: 0; background-color: #f4f4f5; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; }
        .email-wrapper { width: 100%; background-color: #f4f4f5; padding: 40px 0; }
        .email-container { max-width: 600px; margin: 0 auto; background-color: #ffffff; border-radius: 8px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
        .email-header { background-color: #6366f1; padding: 30px 40px; text-align: center; }
        .email-header h1 { color: #ffffff; margin: 0; font-size: 24px; font-weight: 600; }
        .email-body { padding: 40px; color: #374151; font-size: 16px; line-height: 1.6; }
        .email-footer { background-color: #f9fafb; padding: 30px 40px; text-align: center; color: #6b7280; font-size: 13px; }
        .email-footer a { color: #6366f1; text-decoration: none; }
        .email-footer a:hover { text-decoration: underline; }
        @media only screen and (max-width: 600px) {
            .email-container { width: 100% !important; border-radius: 0 !important; }
            .email-body { padding: 30px 20px !important; }
        }
    </style>
</head>
<body>
    <div class="email-wrapper">
        <div class="email-container">
            <div class="email-body">
                {{content}}
            </div>
            <div class="email-footer">
                <p>&copy; {{current_year}} {{company_name}}</p>
                {{company_address}}
                <p style="margin-top: 15px;">
                    <a href="{{unsubscribe_url}}">Unsubscribe</a> &bull;
                    <a href="{{preferences_url}}">Email Preferences</a>
                </p>
            </div>
        </div>
    </div>
</body>
</html>';
    }

    /**
     * Render a saved template by ID
     *
     * @param int $templateId Template ID
     * @param array $data Variable data
     * @return string|null Rendered HTML or null if not found
     */
    public function renderTemplate(int $templateId, array $data = []): ?string
    {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_templates';
        $template = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $templateId
        ));

        if (!$template) {
            return null;
        }

        return $this->render($template->content, $data);
    }

    /**
     * Preview email with sample data
     *
     * @param string $content Email content
     * @return string Rendered preview HTML
     */
    public function preview(string $content): string
    {
        $sampleData = [
            'email' => 'subscriber@example.com',
            'first_name' => 'John',
            'last_name' => 'Doe',
            'token' => 'sample_token_12345',
        ];

        return $this->render($content, $sampleData);
    }

    /**
     * Inline CSS styles for email clients
     *
     * @param string $html HTML with <style> tags
     * @return string HTML with inlined styles
     */
    public function inlineStyles(string $html): string
    {
        // Basic CSS inliner - for production, use a library like Emogrifier
        // This is a simplified version that handles common cases

        // Extract styles
        preg_match_all('/<style[^>]*>(.*?)<\/style>/is', $html, $matches);

        if (empty($matches[1])) {
            return $html;
        }

        $styles = implode("\n", $matches[1]);

        // Parse CSS rules (simplified)
        preg_match_all('/([^{]+)\{([^}]+)\}/s', $styles, $rules);

        if (empty($rules[0])) {
            return $html;
        }

        // Apply inline styles (simplified - handles classes and elements)
        foreach ($rules[1] as $index => $selector) {
            $selector = trim($selector);
            $cssRules = trim($rules[2][$index]);

            // Skip media queries and complex selectors
            if (strpos($selector, '@') !== false || strpos($selector, ':') !== false) {
                continue;
            }

            // Handle class selectors
            if (strpos($selector, '.') === 0) {
                $className = substr($selector, 1);
                $html = preg_replace_callback(
                    '/class=["\']([^"\']*\b' . preg_quote($className, '/') . '\b[^"\']*)["\']/',
                    function ($match) use ($cssRules) {
                        // This is simplified - real implementation would merge styles
                        return $match[0];
                    },
                    $html
                );
            }
        }

        return $html;
    }
}
