<?php

namespace WPOutreach\Mailer;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

/**
 * Direct SMTP mailer using PHPMailer
 */
class SMTPMailer extends AbstractMailer
{
    /**
     * @var array SMTP configuration
     */
    private array $config = [];

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->loadConfig();
    }

    /**
     * Load SMTP configuration
     */
    private function loadConfig(): void
    {
        $this->config = get_option('wp_outreach_smtp', []);
    }

    /**
     * Get mailer name
     */
    public function getName(): string
    {
        return 'smtp';
    }

    /**
     * Check if SMTP is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->config['host'])
            && !empty($this->config['port'])
            && !empty($this->config['username'])
            && !empty($this->config['password']);
    }

    /**
     * Send email via SMTP
     */
    public function send(string $to, string $subject, string $body, array $options = []): bool
    {
        $this->clearError();

        if (!$this->isConfigured()) {
            $this->setError(__('SMTP is not configured', 'outreach'));
            return false;
        }

        $options = $this->mergeOptions($options);

        // Load WordPress PHPMailer
        require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
        require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
        require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';

        $mail = new PHPMailer(true);

        try {
            // SMTP settings
            $mail->isSMTP();
            $mail->Host = $this->config['host'];
            $mail->Port = (int) $this->config['port'];
            $mail->SMTPAuth = true;
            $mail->Username = $this->config['username'];
            $mail->Password = $this->config['password'];

            // Encryption
            $encryption = $this->config['encryption'] ?? 'tls';
            if ($encryption === 'tls') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            } elseif ($encryption === 'ssl') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            }

            // Timeout
            $mail->Timeout = 30;

            // From
            $mail->setFrom(
                $options['from_email'],
                $options['from_name'] ?? ''
            );

            // Reply-To
            if (!empty($options['reply_to'])) {
                $mail->addReplyTo($options['reply_to']);
            }

            // Recipient
            $mail->addAddress($to);

            // Content
            $mail->isHTML(true);
            $mail->CharSet = 'UTF-8';
            $mail->Subject = $subject;
            $mail->Body = $this->wrapHtml($body);
            $mail->AltBody = wp_strip_all_tags($body);

            // Attachments
            if (!empty($options['attachments'])) {
                foreach ($options['attachments'] as $attachment) {
                    if (is_string($attachment) && file_exists($attachment)) {
                        $mail->addAttachment($attachment);
                    }
                }
            }

            $mail->send();
            return true;

        } catch (Exception $e) {
            $this->setError($mail->ErrorInfo ?: $e->getMessage());
            return false;
        }
    }

    /**
     * Send batch emails via SMTP with connection reuse
     */
    public function sendBatch(array $recipients, string $subject, string $body, array $options = []): array
    {
        $this->clearError();
        $results = [];

        if (!$this->isConfigured()) {
            $this->setError(__('SMTP is not configured', 'outreach'));
            return array_fill_keys($recipients, false);
        }

        $options = $this->mergeOptions($options);

        require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
        require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
        require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';

        $mail = new PHPMailer(true);

        try {
            // SMTP settings
            $mail->isSMTP();
            $mail->Host = $this->config['host'];
            $mail->Port = (int) $this->config['port'];
            $mail->SMTPAuth = true;
            $mail->Username = $this->config['username'];
            $mail->Password = $this->config['password'];
            $mail->SMTPKeepAlive = true; // Keep connection open for batch

            $encryption = $this->config['encryption'] ?? 'tls';
            if ($encryption === 'tls') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            } elseif ($encryption === 'ssl') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            }

            $mail->Timeout = 30;
            $mail->setFrom($options['from_email'], $options['from_name'] ?? '');

            if (!empty($options['reply_to'])) {
                $mail->addReplyTo($options['reply_to']);
            }

            $mail->isHTML(true);
            $mail->CharSet = 'UTF-8';
            $mail->Subject = $subject;
            $mail->Body = $this->wrapHtml($body);
            $mail->AltBody = wp_strip_all_tags($body);

            foreach ($recipients as $email) {
                try {
                    $mail->clearAddresses();
                    $mail->addAddress($email);
                    $mail->send();
                    $results[$email] = true;
                } catch (Exception $e) {
                    $results[$email] = false;
                    $this->setError($e->getMessage());
                }
            }

            $mail->smtpClose();

        } catch (Exception $e) {
            $this->setError($e->getMessage());
            return array_fill_keys($recipients, false);
        }

        return $results;
    }
}
