<?php

namespace WPOutreach\Mailer;

/**
 * Amazon SES mailer using AWS SDK or REST API
 */
class SESMailer extends AbstractMailer
{
    /**
     * @var array SES configuration
     */
    private array $config = [];

    /**
     * @var string AWS SES API version
     */
    private string $apiVersion = '2010-12-01';

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->loadConfig();
    }

    /**
     * Load SES configuration
     * Checks wp-config constants first, then falls back to database option
     */
    private function loadConfig(): void
    {
        $this->config = get_option('wp_outreach_ses', []);

        // Override with wp-config constants if defined
        if (defined('WP_OUTREACH_SES_ACCESS_KEY') && WP_OUTREACH_SES_ACCESS_KEY) {
            $this->config['access_key'] = WP_OUTREACH_SES_ACCESS_KEY;
        }
        if (defined('WP_OUTREACH_SES_SECRET_KEY') && WP_OUTREACH_SES_SECRET_KEY) {
            $this->config['secret_key'] = WP_OUTREACH_SES_SECRET_KEY;
        }
        if (defined('WP_OUTREACH_SES_REGION') && WP_OUTREACH_SES_REGION) {
            $this->config['region'] = WP_OUTREACH_SES_REGION;
        }
    }

    /**
     * Get mailer name
     */
    public function getName(): string
    {
        return 'ses';
    }

    /**
     * Check if SES is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->config['access_key'])
            && !empty($this->config['secret_key'])
            && !empty($this->config['region']);
    }

    /**
     * Send email via Amazon SES
     */
    public function send(string $to, string $subject, string $body, array $options = []): bool
    {
        $this->clearError();

        if (!$this->isConfigured()) {
            $this->setError(__('Amazon SES is not configured', 'outreach'));
            return false;
        }

        $options = $this->mergeOptions($options);

        $params = [
            'Action' => 'SendEmail',
            'Version' => $this->apiVersion,
            'Source' => $this->formatAddress($options['from_email'], $options['from_name'] ?? ''),
            'Destination.ToAddresses.member.1' => $to,
            'Message.Subject.Data' => $subject,
            'Message.Subject.Charset' => 'UTF-8',
            'Message.Body.Html.Data' => $this->wrapHtml($body),
            'Message.Body.Html.Charset' => 'UTF-8',
            'Message.Body.Text.Data' => wp_strip_all_tags($body),
            'Message.Body.Text.Charset' => 'UTF-8',
        ];

        // Reply-To
        if (!empty($options['reply_to'])) {
            $params['ReplyToAddresses.member.1'] = $options['reply_to'];
        }

        $response = $this->makeRequest($params);

        if ($response === false) {
            return false;
        }

        // Check for SES errors
        if (isset($response['Error'])) {
            $this->setError($response['Error']['Message'] ?? __('Unknown SES error', 'outreach'));
            return false;
        }

        return isset($response['SendEmailResult']['MessageId']);
    }

    /**
     * Send batch emails via SES (up to 50 per request)
     */
    public function sendBatch(array $recipients, string $subject, string $body, array $options = []): array
    {
        $results = [];

        // SES SendEmail doesn't support bulk, so send individually
        // For true bulk, would need SendBulkTemplatedEmail
        foreach ($recipients as $email) {
            $results[$email] = $this->send($email, $subject, $body, $options);
        }

        return $results;
    }

    /**
     * Format email address with name
     */
    private function formatAddress(string $email, string $name = ''): string
    {
        if ($name) {
            return sprintf('"%s" <%s>', str_replace('"', '', $name), $email);
        }
        return $email;
    }

    /**
     * Make signed request to SES API
     */
    private function makeRequest(array $params): array|false
    {
        $region = $this->config['region'];
        $host = "email.{$region}.amazonaws.com";
        $endpoint = "https://{$host}/";

        // Build query string
        $queryString = http_build_query($params, '', '&', PHP_QUERY_RFC3986);

        // Create signature
        $date = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');

        $headers = [
            'host' => $host,
            'x-amz-date' => $date,
            'content-type' => 'application/x-www-form-urlencoded',
        ];

        // Create canonical request
        $canonicalHeaders = '';
        $signedHeaders = [];
        ksort($headers);
        foreach ($headers as $key => $value) {
            $canonicalHeaders .= strtolower($key) . ':' . trim($value) . "\n";
            $signedHeaders[] = strtolower($key);
        }
        $signedHeadersStr = implode(';', $signedHeaders);

        $canonicalRequest = implode("\n", [
            'POST',
            '/',
            '',
            $canonicalHeaders,
            $signedHeadersStr,
            hash('sha256', $queryString),
        ]);

        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $credentialScope = "{$dateStamp}/{$region}/ses/aws4_request";
        $stringToSign = implode("\n", [
            $algorithm,
            $date,
            $credentialScope,
            hash('sha256', $canonicalRequest),
        ]);

        // Calculate signature
        $signingKey = $this->getSigningKey($dateStamp, $region);
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);

        // Build authorization header
        $authorization = sprintf(
            '%s Credential=%s/%s, SignedHeaders=%s, Signature=%s',
            $algorithm,
            $this->config['access_key'],
            $credentialScope,
            $signedHeadersStr,
            $signature
        );

        // Make request
        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Host' => $host,
                'X-Amz-Date' => $date,
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Authorization' => $authorization,
            ],
            'body' => $queryString,
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            $this->setError($response->get_error_message());
            return false;
        }

        $body = wp_remote_retrieve_body($response);

        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);

        if ($xml === false) {
            $this->setError(__('Invalid response from SES', 'outreach'));
            return false;
        }

        return $this->xmlToArray($xml);
    }

    /**
     * Generate AWS4 signing key
     */
    private function getSigningKey(string $dateStamp, string $region): string
    {
        $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $this->config['secret_key'], true);
        $kRegion = hash_hmac('sha256', $region, $kDate, true);
        $kService = hash_hmac('sha256', 'ses', $kRegion, true);
        return hash_hmac('sha256', 'aws4_request', $kService, true);
    }

    /**
     * Convert SimpleXMLElement to array
     */
    private function xmlToArray(\SimpleXMLElement $xml): array
    {
        $json = json_encode($xml);
        return json_decode($json, true);
    }
}
