<?php

declare(strict_types=1);

namespace WPOutreach\Mailer\SES;

/**
 * AWS SES Regions
 *
 * Static list of all AWS regions that support SES
 */
class SESRegions
{
    /**
     * All AWS SES regions
     */
    public const REGIONS = [
        'us-east-1'      => 'US East (N. Virginia)',
        'us-east-2'      => 'US East (Ohio)',
        'us-west-1'      => 'US West (N. California)',
        'us-west-2'      => 'US West (Oregon)',
        'eu-west-1'      => 'Europe (Ireland)',
        'eu-west-2'      => 'Europe (London)',
        'eu-west-3'      => 'Europe (Paris)',
        'eu-central-1'   => 'Europe (Frankfurt)',
        'eu-north-1'     => 'Europe (Stockholm)',
        'eu-south-1'     => 'Europe (Milan)',
        'ap-south-1'     => 'Asia Pacific (Mumbai)',
        'ap-northeast-1' => 'Asia Pacific (Tokyo)',
        'ap-northeast-2' => 'Asia Pacific (Seoul)',
        'ap-northeast-3' => 'Asia Pacific (Osaka)',
        'ap-southeast-1' => 'Asia Pacific (Singapore)',
        'ap-southeast-2' => 'Asia Pacific (Sydney)',
        'ca-central-1'   => 'Canada (Central)',
        'me-south-1'     => 'Middle East (Bahrain)',
        'sa-east-1'      => 'South America (Sao Paulo)',
        'af-south-1'     => 'Africa (Cape Town)',
        'il-central-1'   => 'Israel (Tel Aviv)',
    ];

    /**
     * Get all regions
     */
    public static function getAll(): array
    {
        return self::REGIONS;
    }

    /**
     * Get all regions as array for API response
     */
    public static function toArray(): array
    {
        $result = [];
        foreach (self::REGIONS as $code => $name) {
            $result[] = [
                'code' => $code,
                'name' => $name,
            ];
        }
        return $result;
    }

    /**
     * Check if a region is valid
     */
    public static function isValid(string $region): bool
    {
        return isset(self::REGIONS[$region]);
    }

    /**
     * Get region name by code
     */
    public static function getName(string $code): ?string
    {
        return self::REGIONS[$code] ?? null;
    }

    /**
     * Get SES endpoint for a region
     */
    public static function getEndpoint(string $region): string
    {
        return sprintf('email.%s.amazonaws.com', $region);
    }
}
