<?php

declare(strict_types=1);

namespace WPOutreach\Mailer\SES;

/**
 * SES Manager
 *
 * Core SES API operations for identity management and account status
 * Uses AWS SES API v1 (2010-12-01) with Signature v4 signing
 */
class SESManager
{
    private const API_VERSION = '2010-12-01';

    private SESCredentials $credentials;
    private ?string $lastError = null;

    public function __construct(?SESCredentials $credentials = null)
    {
        $this->credentials = $credentials ?? new SESCredentials();
    }

    /**
     * Check if SES is configured with valid credentials
     */
    public function isConfigured(): bool
    {
        return $this->credentials->isConfigured();
    }

    /**
     * Test connection by getting send quota
     */
    public function testConnection(): array
    {
        if (!$this->credentials->isConfigured()) {
            return [
                'success' => false,
                'error' => __('SES credentials not configured', 'outreach'),
            ];
        }

        $response = $this->makeRequest([
            'Action' => 'GetSendQuota',
        ]);

        if ($response === false) {
            return [
                'success' => false,
                'error' => $this->lastError,
            ];
        }

        if (isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? __('Unknown SES error', 'outreach'),
                'code' => $response['Error']['Code'] ?? null,
            ];
        }

        return [
            'success' => true,
            'message' => __('Connection successful', 'outreach'),
            'quota' => $response['GetSendQuotaResult'] ?? null,
        ];
    }

    /**
     * Get account status including quota and sandbox detection
     */
    public function getAccountStatus(): array
    {
        $baseResponse = [
            'source' => $this->credentials->getSource(),
            'region' => $this->credentials->getRegion(),
            'config_constants' => $this->credentials->getConfigConstants(),
        ];

        if (!$this->credentials->isConfigured()) {
            return array_merge($baseResponse, [
                'configured' => false,
            ]);
        }

        $response = $this->makeRequest([
            'Action' => 'GetSendQuota',
        ], 10); // Use shorter timeout for status check

        if ($response === false || isset($response['Error'])) {
            return array_merge($baseResponse, [
                'configured' => true,
                'error' => $response['Error']['Message'] ?? $this->lastError,
                'quota' => null,
            ]);
        }

        $quota = $response['GetSendQuotaResult'] ?? [];
        $max24HourSend = floatval($quota['Max24HourSend'] ?? 0);

        return array_merge($baseResponse, [
            'configured' => true,
            'is_sandbox' => $max24HourSend <= 200,
            'quota' => [
                'max_24_hour_send' => $max24HourSend,
                'max_send_rate' => floatval($quota['MaxSendRate'] ?? 0),
                'sent_last_24_hours' => floatval($quota['SentLast24Hours'] ?? 0),
            ],
        ]);
    }

    /**
     * Get sending statistics for the past two weeks
     *
     * Returns bounce rate, complaint rate, and delivery metrics
     */
    public function getSendStatistics(): array
    {
        if (!$this->credentials->isConfigured()) {
            return [
                'success' => false,
                'error' => __('SES not configured', 'outreach'),
            ];
        }

        $response = $this->makeRequest([
            'Action' => 'GetSendStatistics',
        ], 15);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        $dataPoints = $response['GetSendStatisticsResult']['SendDataPoints']['member'] ?? [];

        // Normalize to array (single item returns as string/object)
        if (!is_array($dataPoints) || (isset($dataPoints['Timestamp']))) {
            $dataPoints = [$dataPoints];
        }

        // Aggregate totals
        $totals = [
            'delivery_attempts' => 0,
            'bounces' => 0,
            'complaints' => 0,
            'rejects' => 0,
        ];

        foreach ($dataPoints as $point) {
            if (!is_array($point)) continue;
            $totals['delivery_attempts'] += intval($point['DeliveryAttempts'] ?? 0);
            $totals['bounces'] += intval($point['Bounces'] ?? 0);
            $totals['complaints'] += intval($point['Complaints'] ?? 0);
            $totals['rejects'] += intval($point['Rejects'] ?? 0);
        }

        // Calculate rates
        $deliveryAttempts = $totals['delivery_attempts'];
        $bounceRate = $deliveryAttempts > 0 ? ($totals['bounces'] / $deliveryAttempts) * 100 : 0;
        $complaintRate = $deliveryAttempts > 0 ? ($totals['complaints'] / $deliveryAttempts) * 100 : 0;
        $deliveryRate = $deliveryAttempts > 0
            ? (($deliveryAttempts - $totals['bounces'] - $totals['rejects']) / $deliveryAttempts) * 100
            : 0;

        // Determine health status
        $bounceStatus = 'healthy';
        if ($bounceRate >= 5) {
            $bounceStatus = 'danger';
        } elseif ($bounceRate >= 3) {
            $bounceStatus = 'warning';
        }

        $complaintStatus = 'healthy';
        if ($complaintRate >= 0.1) {
            $complaintStatus = 'danger';
        } elseif ($complaintRate >= 0.05) {
            $complaintStatus = 'warning';
        }

        return [
            'success' => true,
            'period' => __('Last 14 days', 'outreach'),
            'totals' => $totals,
            'rates' => [
                'bounce_rate' => round($bounceRate, 3),
                'bounce_status' => $bounceStatus,
                'complaint_rate' => round($complaintRate, 4),
                'complaint_status' => $complaintStatus,
                'delivery_rate' => round($deliveryRate, 2),
            ],
            'data_points' => count($dataPoints),
        ];
    }

    /**
     * List email or domain identities
     *
     * @param string $type 'EmailAddress' or 'Domain'
     */
    public function listIdentities(string $type = ''): array
    {
        if (!$this->credentials->isConfigured()) {
            return ['success' => false, 'error' => __('SES not configured', 'outreach')];
        }

        $params = ['Action' => 'ListIdentities'];
        if ($type) {
            $params['IdentityType'] = $type;
        }

        $response = $this->makeRequest($params);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        $members = $response['ListIdentitiesResult']['Identities']['member'] ?? [];

        // Normalize to array (single item returns as string)
        if (is_string($members)) {
            $members = [$members];
        }

        return [
            'success' => true,
            'identities' => $members,
        ];
    }

    /**
     * Get verification status for identities
     */
    public function getIdentityVerificationStatus(array $identities): array
    {
        if (empty($identities)) {
            return ['success' => true, 'statuses' => []];
        }

        $params = ['Action' => 'GetIdentityVerificationAttributes'];
        foreach (array_values($identities) as $i => $identity) {
            $params['Identities.member.' . ($i + 1)] = $identity;
        }

        $response = $this->makeRequest($params);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        $entries = $response['GetIdentityVerificationAttributesResult']['VerificationAttributes']['entry'] ?? [];

        // Normalize single entry to array
        if (isset($entries['key'])) {
            $entries = [$entries];
        }

        $statuses = [];
        foreach ($entries as $entry) {
            $key = $entry['key'] ?? '';
            $value = $entry['value'] ?? [];
            $statuses[$key] = [
                'status' => $value['VerificationStatus'] ?? 'NotStarted',
                'token' => $value['VerificationToken'] ?? null,
            ];
        }

        return [
            'success' => true,
            'statuses' => $statuses,
        ];
    }

    /**
     * Get DKIM attributes for domains
     */
    public function getDkimAttributes(array $domains): array
    {
        if (empty($domains)) {
            return ['success' => true, 'attributes' => []];
        }

        $params = ['Action' => 'GetIdentityDkimAttributes'];
        foreach (array_values($domains) as $i => $domain) {
            $params['Identities.member.' . ($i + 1)] = $domain;
        }

        $response = $this->makeRequest($params);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        $entries = $response['GetIdentityDkimAttributesResult']['DkimAttributes']['entry'] ?? [];

        // Normalize single entry to array
        if (isset($entries['key'])) {
            $entries = [$entries];
        }

        $attributes = [];
        foreach ($entries as $entry) {
            $key = $entry['key'] ?? '';
            $value = $entry['value'] ?? [];
            $tokens = $value['DkimTokens']['member'] ?? [];

            // Normalize tokens
            if (is_string($tokens)) {
                $tokens = [$tokens];
            }

            $attributes[$key] = [
                'dkim_enabled' => ($value['DkimEnabled'] ?? 'false') === 'true',
                'dkim_verification_status' => $value['DkimVerificationStatus'] ?? 'NotStarted',
                'dkim_tokens' => $tokens,
            ];
        }

        return [
            'success' => true,
            'attributes' => $attributes,
        ];
    }

    /**
     * Verify an email identity (sends verification email)
     */
    public function verifyEmail(string $email): array
    {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'error' => __('Invalid email address', 'outreach')];
        }

        $response = $this->makeRequest([
            'Action' => 'VerifyEmailIdentity',
            'EmailAddress' => $email,
        ]);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        return [
            'success' => true,
            'message' => sprintf(__('Verification email sent to %s', 'outreach'), $email),
            'email' => $email,
        ];
    }

    /**
     * Verify a domain identity
     */
    public function verifyDomain(string $domain): array
    {
        // Basic domain validation
        $domain = strtolower(trim($domain));
        if (!preg_match('/^[a-z0-9]([a-z0-9-]*[a-z0-9])?(\.[a-z0-9]([a-z0-9-]*[a-z0-9])?)+$/', $domain)) {
            return ['success' => false, 'error' => __('Invalid domain name', 'outreach')];
        }

        // Verify domain identity
        $response = $this->makeRequest([
            'Action' => 'VerifyDomainIdentity',
            'Domain' => $domain,
        ]);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        $verificationToken = $response['VerifyDomainIdentityResult']['VerificationToken'] ?? '';

        // Also initiate DKIM verification
        $dkimResponse = $this->makeRequest([
            'Action' => 'VerifyDomainDkim',
            'Domain' => $domain,
        ]);

        $dkimTokens = [];
        if ($dkimResponse !== false && !isset($dkimResponse['Error'])) {
            $dkimTokens = $dkimResponse['VerifyDomainDkimResult']['DkimTokens']['member'] ?? [];
            if (is_string($dkimTokens)) {
                $dkimTokens = [$dkimTokens];
            }
        }

        // Build DNS records
        $dnsRecords = $this->buildDnsRecords($domain, $verificationToken, $dkimTokens);

        return [
            'success' => true,
            'domain' => $domain,
            'verification_token' => $verificationToken,
            'dkim_tokens' => $dkimTokens,
            'dns_records' => $dnsRecords,
        ];
    }

    /**
     * Delete an identity (email or domain)
     */
    public function deleteIdentity(string $identity): array
    {
        $response = $this->makeRequest([
            'Action' => 'DeleteIdentity',
            'Identity' => $identity,
        ]);

        if ($response === false || isset($response['Error'])) {
            return [
                'success' => false,
                'error' => $response['Error']['Message'] ?? $this->lastError,
            ];
        }

        return [
            'success' => true,
            'message' => sprintf(__('Identity %s deleted', 'outreach'), $identity),
        ];
    }

    /**
     * Get all identities with their verification status
     */
    public function getAllIdentitiesWithStatus(): array
    {
        // Get emails
        $emailsResult = $this->listIdentities('EmailAddress');
        $emails = $emailsResult['success'] ? $emailsResult['identities'] : [];

        // Get domains
        $domainsResult = $this->listIdentities('Domain');
        $domains = $domainsResult['success'] ? $domainsResult['identities'] : [];

        // Get verification status for all
        $allIdentities = array_merge($emails, $domains);
        $statusResult = $this->getIdentityVerificationStatus($allIdentities);
        $statuses = $statusResult['success'] ? $statusResult['statuses'] : [];

        // Get DKIM for domains
        $dkimResult = $this->getDkimAttributes($domains);
        $dkimAttributes = $dkimResult['success'] ? $dkimResult['attributes'] : [];

        // Build response
        $emailData = [];
        foreach ($emails as $email) {
            $emailData[] = [
                'identity' => $email,
                'status' => $statuses[$email]['status'] ?? 'NotStarted',
            ];
        }

        $domainData = [];
        foreach ($domains as $domain) {
            $dkim = $dkimAttributes[$domain] ?? [];
            $domainData[] = [
                'identity' => $domain,
                'status' => $statuses[$domain]['status'] ?? 'NotStarted',
                'verification_token' => $statuses[$domain]['token'] ?? null,
                'dkim_enabled' => $dkim['dkim_enabled'] ?? false,
                'dkim_status' => $dkim['dkim_verification_status'] ?? 'NotStarted',
                'dkim_tokens' => $dkim['dkim_tokens'] ?? [],
            ];
        }

        return [
            'success' => true,
            'emails' => $emailData,
            'domains' => $domainData,
        ];
    }

    /**
     * Refresh status for a single identity
     */
    public function refreshIdentityStatus(string $identity): array
    {
        $statusResult = $this->getIdentityVerificationStatus([$identity]);

        if (!$statusResult['success']) {
            return $statusResult;
        }

        $status = $statusResult['statuses'][$identity] ?? null;

        if (!$status) {
            return [
                'success' => false,
                'error' => __('Identity not found', 'outreach'),
            ];
        }

        // Check if it's a domain and get DKIM
        $isDomain = !filter_var($identity, FILTER_VALIDATE_EMAIL);
        $dkim = null;

        if ($isDomain) {
            $dkimResult = $this->getDkimAttributes([$identity]);
            if ($dkimResult['success'] && isset($dkimResult['attributes'][$identity])) {
                $dkim = $dkimResult['attributes'][$identity];
            }
        }

        $result = [
            'success' => true,
            'identity' => $identity,
            'status' => $status['status'],
            'verification_token' => $status['token'],
        ];

        if ($dkim) {
            $result['dkim_enabled'] = $dkim['dkim_enabled'];
            $result['dkim_status'] = $dkim['dkim_verification_status'];
            $result['dkim_tokens'] = $dkim['dkim_tokens'];
        }

        return $result;
    }

    /**
     * Build DNS records for domain verification
     */
    private function buildDnsRecords(string $domain, string $verificationToken, array $dkimTokens): array
    {
        $records = [];

        // Domain verification TXT record
        $records[] = [
            'type' => 'TXT',
            'name' => "_amazonses.{$domain}",
            'value' => $verificationToken,
            'purpose' => __('Domain Verification', 'outreach'),
        ];

        // DKIM CNAME records
        foreach ($dkimTokens as $token) {
            $records[] = [
                'type' => 'CNAME',
                'name' => "{$token}._domainkey.{$domain}",
                'value' => "{$token}.dkim.amazonses.com",
                'purpose' => __('DKIM Authentication', 'outreach'),
            ];
        }

        // SPF recommendation (informational)
        $records[] = [
            'type' => 'TXT',
            'name' => $domain,
            'value' => 'v=spf1 include:amazonses.com ~all',
            'purpose' => __('SPF (Recommended)', 'outreach'),
            'optional' => true,
        ];

        // DMARC recommendation (informational)
        $records[] = [
            'type' => 'TXT',
            'name' => "_dmarc.{$domain}",
            'value' => "v=DMARC1; p=none; rua=mailto:dmarc@{$domain}",
            'purpose' => __('DMARC (Recommended)', 'outreach'),
            'optional' => true,
        ];

        return $records;
    }

    /**
     * Make signed request to SES API
     *
     * @param array $params Request parameters
     * @param int $timeout Request timeout in seconds
     */
    private function makeRequest(array $params, int $timeout = 30): array|false
    {
        $params['Version'] = self::API_VERSION;

        $region = $this->credentials->getRegion();
        $host = SESRegions::getEndpoint($region);
        $endpoint = "https://{$host}/";

        // Build query string
        $queryString = http_build_query($params, '', '&', PHP_QUERY_RFC3986);

        // Create signature
        $date = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');

        $headers = [
            'host' => $host,
            'x-amz-date' => $date,
            'content-type' => 'application/x-www-form-urlencoded',
        ];

        // Create canonical request
        $canonicalHeaders = '';
        $signedHeaders = [];
        ksort($headers);
        foreach ($headers as $key => $value) {
            $canonicalHeaders .= strtolower($key) . ':' . trim($value) . "\n";
            $signedHeaders[] = strtolower($key);
        }
        $signedHeadersStr = implode(';', $signedHeaders);

        $canonicalRequest = implode("\n", [
            'POST',
            '/',
            '',
            $canonicalHeaders,
            $signedHeadersStr,
            hash('sha256', $queryString),
        ]);

        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $credentialScope = "{$dateStamp}/{$region}/ses/aws4_request";
        $stringToSign = implode("\n", [
            $algorithm,
            $date,
            $credentialScope,
            hash('sha256', $canonicalRequest),
        ]);

        // Calculate signature
        $signingKey = $this->getSigningKey($dateStamp, $region);
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);

        // Build authorization header
        $authorization = sprintf(
            '%s Credential=%s/%s, SignedHeaders=%s, Signature=%s',
            $algorithm,
            $this->credentials->getAccessKey(),
            $credentialScope,
            $signedHeadersStr,
            $signature
        );

        // Make request
        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Host' => $host,
                'X-Amz-Date' => $date,
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Authorization' => $authorization,
            ],
            'body' => $queryString,
            'timeout' => $timeout,
        ]);

        if (is_wp_error($response)) {
            $this->lastError = $response->get_error_message();
            return false;
        }

        $body = wp_remote_retrieve_body($response);

        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);

        if ($xml === false) {
            $this->lastError = __('Invalid response from SES', 'outreach');
            return false;
        }

        return $this->xmlToArray($xml);
    }

    /**
     * Generate AWS4 signing key
     */
    private function getSigningKey(string $dateStamp, string $region): string
    {
        $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $this->credentials->getSecretKey(), true);
        $kRegion = hash_hmac('sha256', $region, $kDate, true);
        $kService = hash_hmac('sha256', 'ses', $kRegion, true);
        return hash_hmac('sha256', 'aws4_request', $kService, true);
    }

    /**
     * Convert SimpleXMLElement to array
     */
    private function xmlToArray(\SimpleXMLElement $xml): array
    {
        $json = json_encode($xml);
        return json_decode($json, true);
    }

    /**
     * Get last error message
     */
    public function getLastError(): ?string
    {
        return $this->lastError;
    }

    // =========================================================================
    // Suppression List Methods (SES v2 API)
    // =========================================================================

    /**
     * List suppressed email addresses
     *
     * @param string $reason Filter by reason: 'BOUNCE', 'COMPLAINT', or empty for all
     * @param string $nextToken Pagination token
     * @param int $pageSize Number of results per page (max 1000)
     */
    public function listSuppressedDestinations(string $reason = '', string $nextToken = '', int $pageSize = 100): array
    {
        if (!$this->credentials->isConfigured()) {
            return ['success' => false, 'error' => __('SES not configured', 'outreach')];
        }

        $queryParams = ['PageSize' => min($pageSize, 1000)];

        if ($reason && in_array($reason, ['BOUNCE', 'COMPLAINT'])) {
            $queryParams['Reasons'] = $reason;
        }

        if ($nextToken) {
            $queryParams['NextToken'] = $nextToken;
        }

        $response = $this->makeV2Request('GET', '/v2/email/suppression/addresses', $queryParams);

        if ($response === false || isset($response['error'])) {
            return [
                'success' => false,
                'error' => $response['error'] ?? $this->lastError,
            ];
        }

        $items = $response['SuppressedDestinationSummaries'] ?? [];

        return [
            'success' => true,
            'items' => $items,
            'next_token' => $response['NextToken'] ?? null,
        ];
    }

    /**
     * Get details for a suppressed email address
     */
    public function getSuppressedDestination(string $email): array
    {
        if (!$this->credentials->isConfigured()) {
            return ['success' => false, 'error' => __('SES not configured', 'outreach')];
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'error' => __('Invalid email address', 'outreach')];
        }

        $response = $this->makeV2Request('GET', '/v2/email/suppression/addresses/' . rawurlencode($email));

        if ($response === false || isset($response['error'])) {
            return [
                'success' => false,
                'error' => $response['error'] ?? $this->lastError,
            ];
        }

        return [
            'success' => true,
            'destination' => $response['SuppressedDestination'] ?? null,
        ];
    }

    /**
     * Add an email address to the suppression list
     *
     * @param string $email Email address to suppress
     * @param string $reason 'BOUNCE' or 'COMPLAINT'
     */
    public function putSuppressedDestination(string $email, string $reason = 'BOUNCE'): array
    {
        if (!$this->credentials->isConfigured()) {
            return ['success' => false, 'error' => __('SES not configured', 'outreach')];
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'error' => __('Invalid email address', 'outreach')];
        }

        if (!in_array($reason, ['BOUNCE', 'COMPLAINT'])) {
            return ['success' => false, 'error' => __('Reason must be BOUNCE or COMPLAINT', 'outreach')];
        }

        $body = [
            'EmailAddress' => $email,
            'Reason' => $reason,
        ];

        $response = $this->makeV2Request('PUT', '/v2/email/suppression/addresses/' . rawurlencode($email), [], $body);

        if ($response === false || isset($response['error'])) {
            return [
                'success' => false,
                'error' => $response['error'] ?? $this->lastError,
            ];
        }

        return [
            'success' => true,
            'message' => sprintf(__('%s added to suppression list', 'outreach'), $email),
        ];
    }

    /**
     * Remove an email address from the suppression list
     */
    public function deleteSuppressedDestination(string $email): array
    {
        if (!$this->credentials->isConfigured()) {
            return ['success' => false, 'error' => __('SES not configured', 'outreach')];
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'error' => __('Invalid email address', 'outreach')];
        }

        $response = $this->makeV2Request('DELETE', '/v2/email/suppression/addresses/' . rawurlencode($email));

        if ($response === false || isset($response['error'])) {
            return [
                'success' => false,
                'error' => $response['error'] ?? $this->lastError,
            ];
        }

        return [
            'success' => true,
            'message' => sprintf(__('%s removed from suppression list', 'outreach'), $email),
        ];
    }

    /**
     * Make signed request to SES v2 API (REST/JSON)
     *
     * @param string $method HTTP method (GET, PUT, DELETE)
     * @param string $path API path (e.g., /v2/email/suppression/addresses)
     * @param array $queryParams Query string parameters
     * @param array|null $body Request body for PUT/POST
     * @param int $timeout Request timeout in seconds
     */
    private function makeV2Request(string $method, string $path, array $queryParams = [], ?array $body = null, int $timeout = 30): array|false
    {
        $region = $this->credentials->getRegion();
        $host = "email.{$region}.amazonaws.com";
        $endpoint = "https://{$host}{$path}";

        // Add query string
        $queryString = '';
        if (!empty($queryParams)) {
            $queryString = http_build_query($queryParams, '', '&', PHP_QUERY_RFC3986);
            $endpoint .= '?' . $queryString;
        }

        // Prepare body
        $bodyContent = '';
        if ($body !== null && in_array($method, ['PUT', 'POST'])) {
            $bodyContent = json_encode($body);
        }

        // Create signature
        $date = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');

        $headers = [
            'host' => $host,
            'x-amz-date' => $date,
        ];

        if ($bodyContent) {
            $headers['content-type'] = 'application/json';
        }

        // Create canonical request
        $canonicalHeaders = '';
        $signedHeaders = [];
        ksort($headers);
        foreach ($headers as $key => $value) {
            $canonicalHeaders .= strtolower($key) . ':' . trim($value) . "\n";
            $signedHeaders[] = strtolower($key);
        }
        $signedHeadersStr = implode(';', $signedHeaders);

        $canonicalUri = $path;
        $canonicalQueryString = $queryString;

        $canonicalRequest = implode("\n", [
            $method,
            $canonicalUri,
            $canonicalQueryString,
            $canonicalHeaders,
            $signedHeadersStr,
            hash('sha256', $bodyContent),
        ]);

        // Create string to sign
        $algorithm = 'AWS4-HMAC-SHA256';
        $credentialScope = "{$dateStamp}/{$region}/ses/aws4_request";
        $stringToSign = implode("\n", [
            $algorithm,
            $date,
            $credentialScope,
            hash('sha256', $canonicalRequest),
        ]);

        // Calculate signature
        $signingKey = $this->getSigningKey($dateStamp, $region);
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);

        // Build authorization header
        $authorization = sprintf(
            '%s Credential=%s/%s, SignedHeaders=%s, Signature=%s',
            $algorithm,
            $this->credentials->getAccessKey(),
            $credentialScope,
            $signedHeadersStr,
            $signature
        );

        // Build request headers
        $requestHeaders = [
            'Host' => $host,
            'X-Amz-Date' => $date,
            'Authorization' => $authorization,
        ];

        if ($bodyContent) {
            $requestHeaders['Content-Type'] = 'application/json';
        }

        // Make request
        $args = [
            'method' => $method,
            'headers' => $requestHeaders,
            'timeout' => $timeout,
        ];

        if ($bodyContent) {
            $args['body'] = $bodyContent;
        }

        $response = wp_remote_request($endpoint, $args);

        if (is_wp_error($response)) {
            $this->lastError = $response->get_error_message();
            return false;
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Parse JSON response
        $data = json_decode($body, true);

        // Handle errors
        if ($statusCode >= 400) {
            $errorMessage = $data['message'] ?? $data['Message'] ?? __('Unknown SES error', 'outreach');
            return ['error' => $errorMessage, 'code' => $statusCode];
        }

        // For DELETE requests, empty response is success
        if ($method === 'DELETE' && empty($body)) {
            return [];
        }

        return $data ?: [];
    }
}
