<?php

declare(strict_types=1);

namespace WPOutreach\Mailer\SES;

/**
 * SES Credentials Manager
 *
 * Handles AWS SES credential loading with support for:
 * - wp-config.php constants (priority)
 * - Database storage via WordPress options
 */
class SESCredentials
{
    private const CONFIG_ACCESS_KEY = 'WP_OUTREACH_SES_ACCESS_KEY';
    private const CONFIG_SECRET_KEY = 'WP_OUTREACH_SES_SECRET_KEY';
    private const CONFIG_REGION = 'WP_OUTREACH_SES_REGION';
    private const OPTION_NAME = 'wp_outreach_ses';
    private const DEFAULT_REGION = 'us-east-1';

    private ?string $accessKey = null;
    private ?string $secretKey = null;
    private string $region;
    private string $source = 'none';
    private ?string $configurationSet = null;

    public function __construct()
    {
        $this->loadCredentials();
    }

    /**
     * Load credentials from config or database
     */
    private function loadCredentials(): void
    {
        // Priority 1: Check wp-config.php constants
        if ($this->loadFromConfig()) {
            $this->source = 'config';
            return;
        }

        // Priority 2: Load from database
        if ($this->loadFromDatabase()) {
            $this->source = 'database';
            return;
        }

        $this->source = 'none';
    }

    /**
     * Load credentials from wp-config.php constants
     */
    private function loadFromConfig(): bool
    {
        $hasAccessKey = defined(self::CONFIG_ACCESS_KEY) && !empty(constant(self::CONFIG_ACCESS_KEY));
        $hasSecretKey = defined(self::CONFIG_SECRET_KEY) && !empty(constant(self::CONFIG_SECRET_KEY));

        if ($hasAccessKey && $hasSecretKey) {
            $this->accessKey = constant(self::CONFIG_ACCESS_KEY);
            $this->secretKey = constant(self::CONFIG_SECRET_KEY);
            $this->region = defined(self::CONFIG_REGION) && !empty(constant(self::CONFIG_REGION))
                ? constant(self::CONFIG_REGION)
                : self::DEFAULT_REGION;
            return true;
        }

        return false;
    }

    /**
     * Load credentials from database
     */
    private function loadFromDatabase(): bool
    {
        $options = get_option(self::OPTION_NAME, []);

        if (!is_array($options)) {
            return false;
        }

        $accessKey = $options['access_key'] ?? '';
        $secretKey = $options['secret_key'] ?? '';

        if (!empty($accessKey) && !empty($secretKey) && $secretKey !== '********') {
            $this->accessKey = $accessKey;
            $this->secretKey = $secretKey;
            $this->region = !empty($options['region']) ? $options['region'] : self::DEFAULT_REGION;
            $this->configurationSet = !empty($options['configuration_set']) ? $options['configuration_set'] : null;
            return true;
        }

        // Load region even if credentials aren't set
        $this->region = !empty($options['region']) ? $options['region'] : self::DEFAULT_REGION;
        $this->configurationSet = !empty($options['configuration_set']) ? $options['configuration_set'] : null;

        return false;
    }

    /**
     * Check if credentials are configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->accessKey) && !empty($this->secretKey);
    }

    /**
     * Get credential source
     */
    public function getSource(): string
    {
        return $this->source;
    }

    /**
     * Check if credentials come from wp-config.php
     */
    public function isFromConfig(): bool
    {
        return $this->source === 'config';
    }

    /**
     * Get access key
     */
    public function getAccessKey(): ?string
    {
        return $this->accessKey;
    }

    /**
     * Get secret key
     */
    public function getSecretKey(): ?string
    {
        return $this->secretKey;
    }

    /**
     * Get region
     */
    public function getRegion(): string
    {
        return $this->region;
    }

    /**
     * Get configuration set (optional)
     */
    public function getConfigurationSet(): ?string
    {
        return $this->configurationSet;
    }

    /**
     * Get which config constants are defined
     */
    public function getConfigConstants(): array
    {
        return [
            'access_key' => defined(self::CONFIG_ACCESS_KEY) && !empty(constant(self::CONFIG_ACCESS_KEY)),
            'secret_key' => defined(self::CONFIG_SECRET_KEY) && !empty(constant(self::CONFIG_SECRET_KEY)),
            'region' => defined(self::CONFIG_REGION) && !empty(constant(self::CONFIG_REGION)),
        ];
    }

    /**
     * Get configuration as array for API response
     */
    public function toArray(): array
    {
        return [
            'configured' => $this->isConfigured(),
            'source' => $this->source,
            'region' => $this->region,
            'access_key' => $this->accessKey ? substr($this->accessKey, 0, 8) . '...' : null,
            'config_constants' => $this->getConfigConstants(),
            'configuration_set' => $this->configurationSet,
        ];
    }
}
