<?php

namespace WPOutreach\Mailer\Queue;

use WPOutreach\Mailer\MailerFactory;
use WPOutreach\Mailer\Template\TemplateEngine;

/**
 * Queue worker - processes email queue via cron
 */
class QueueWorker
{
    /**
     * @var QueueManager Queue manager instance
     */
    private QueueManager $queue;

    /**
     * @var TemplateEngine Template engine
     */
    private TemplateEngine $templateEngine;

    /**
     * @var int Batch size per run
     */
    private int $batchSize = 50;

    /**
     * @var int Rate limit (emails per minute)
     */
    private int $rateLimit = 100;

    /**
     * @var int Max retries per email
     */
    private int $maxRetries = 3;

    /**
     * @var bool Is worker currently running
     */
    private static bool $running = false;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->queue = new QueueManager();
        $this->templateEngine = new TemplateEngine();
        $this->loadSettings();
    }

    /**
     * Load settings from options
     */
    private function loadSettings(): void
    {
        $settings = get_option('wp_outreach_mailer', []);

        $this->batchSize = (int) ($settings['batch_size'] ?? 50);
        $this->rateLimit = (int) ($settings['rate_limit'] ?? 100);
        $this->maxRetries = (int) ($settings['max_retries'] ?? 3);
    }

    /**
     * Register cron hooks
     */
    public static function register(): void
    {
        // Register cron hook
        add_action('wp_outreach_process_queue', [self::class, 'processQueue']);

        // Schedule cron if not scheduled
        if (!wp_next_scheduled('wp_outreach_process_queue')) {
            wp_schedule_event(time(), 'wp_outreach_interval', 'wp_outreach_process_queue');
        }

        // Register custom cron interval
        add_filter('cron_schedules', function ($schedules) {
            $schedules['wp_outreach_interval'] = [
                'interval' => 60, // Every minute
                'display' => __('Every Minute (WP Outreach)', 'outreach'),
            ];
            return $schedules;
        });

        // Cleanup old sent emails daily
        add_action('wp_outreach_cleanup_queue', [self::class, 'cleanup']);

        if (!wp_next_scheduled('wp_outreach_cleanup_queue')) {
            wp_schedule_daily_event(time(), 'wp_outreach_cleanup_queue');
        }
    }

    /**
     * Unregister cron hooks (for deactivation)
     */
    public static function unregister(): void
    {
        wp_clear_scheduled_hook('wp_outreach_process_queue');
        wp_clear_scheduled_hook('wp_outreach_cleanup_queue');
    }

    /**
     * Process email queue (static entry point for cron)
     *
     * @return array{sent: int, failed: int, skipped: int, duration?: float}
     */
    public static function processQueue(): array
    {
        $worker = new self();
        return $worker->run();
    }

    /**
     * Run the queue worker
     */
    public function run(): array
    {
        // Prevent concurrent runs
        if (self::$running) {
            return ['skipped' => true, 'reason' => 'Already running'];
        }

        self::$running = true;
        $startTime = microtime(true);
        $results = ['sent' => 0, 'failed' => 0, 'skipped' => 0];

        try {
            // Get pending emails
            $items = $this->queue->getPending($this->batchSize, $this->maxRetries);

            if (empty($items)) {
                self::$running = false;
                return ['sent' => 0, 'message' => 'Queue empty'];
            }

            // Get mailer
            $mailer = MailerFactory::getInstance();

            // Calculate delay between emails for rate limiting
            $delay = $this->rateLimit > 0 ? (60 / $this->rateLimit) * 1000000 : 0; // microseconds

            foreach ($items as $item) {
                // Check execution time (max 50 seconds to stay under minute cron)
                if ((microtime(true) - $startTime) > 50) {
                    break;
                }

                // Mark as processing
                $this->queue->markProcessing($item->id);

                // Personalize content if subscriber exists
                $content = $item->content;
                $subject = $item->subject;

                if ($item->subscriber_id) {
                    $content = $this->templateEngine->renderForSubscriber(
                        $content,
                        $item->subscriber_id
                    );
                    $subject = $this->templateEngine->parseVariables(
                        $subject,
                        $this->getSubscriberData($item->subscriber_id)
                    );
                }

                // Send email
                $sent = $mailer->send($item->to_email, $subject, $content);

                if ($sent) {
                    $this->queue->markSent($item->id);
                    $this->logSuccess($item);
                    $results['sent']++;
                } else {
                    $error = $mailer->getLastError() ?? 'Unknown error';
                    $this->queue->markFailed($item->id, $error);
                    $this->logFailure($item, $error);
                    $results['failed']++;
                }

                // Rate limiting
                if ($delay > 0) {
                    usleep((int) $delay);
                }
            }

        } catch (\Throwable $e) {
            error_log('WP Outreach Queue Error: ' . $e->getMessage());
            $results['error'] = $e->getMessage();
        }

        self::$running = false;
        $results['duration'] = round(microtime(true) - $startTime, 2);

        return $results;
    }

    /**
     * Get subscriber data for template parsing
     */
    private function getSubscriberData(int $subscriberId): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_subscribers';
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $subscriberId
        ), ARRAY_A);

        if (!$subscriber) {
            return [];
        }

        // Parse custom fields
        if (!empty($subscriber['custom_fields'])) {
            $customFields = json_decode($subscriber['custom_fields'], true) ?: [];
            $subscriber = array_merge($subscriber, $customFields);
        }

        return $subscriber;
    }

    /**
     * Log successful send to logs table
     */
    private function logSuccess(object $item): void
    {
        global $wpdb;

        // If tracking_id exists, a log entry was already created by OpenTracker::createLogEntry()
        // during queueing, so we don't need to create a duplicate
        if (!empty($item->tracking_id)) {
            return;
        }

        // Determine email type based on source
        $type = 'campaign';
        if (!empty($item->automation_id)) {
            $type = 'automation';
        }

        $wpdb->insert($wpdb->prefix . 'outreach_logs', [
            'campaign_id' => $item->campaign_id,
            'automation_id' => $item->automation_id,
            'subscriber_id' => $item->subscriber_id,
            'email' => $item->to_email,
            'subject' => $item->subject,
            'status' => 'sent',
            'sent_at' => current_time('mysql'),
            'type' => $type,
        ]);
    }

    /**
     * Log failed send
     */
    private function logFailure(object $item, string $error): void
    {
        // Log to error log for debugging
        error_log(sprintf(
            'WP Outreach: Failed to send email to %s - %s',
            $item->to_email,
            $error
        ));
    }

    /**
     * Cleanup old sent emails
     */
    public static function cleanup(): void
    {
        $manager = new QueueManager();
        $deleted = $manager->cleanup(30);

        if ($deleted > 0) {
            error_log(sprintf('WP Outreach: Cleaned up %d old queue entries', $deleted));
        }
    }

    /**
     * Manually trigger queue processing (for testing or immediate send)
     *
     * @return array{sent: int, failed: int, skipped: int, duration?: float}
     */
    public static function trigger(): array
    {
        $worker = new self();
        return $worker->run();
    }

    /**
     * Check if queue is paused
     */
    public static function isPaused(): bool
    {
        return (bool) get_option('wp_outreach_queue_paused', false);
    }

    /**
     * Pause queue processing
     */
    public static function pause(): void
    {
        update_option('wp_outreach_queue_paused', true);
    }

    /**
     * Resume queue processing
     */
    public static function resume(): void
    {
        update_option('wp_outreach_queue_paused', false);
    }
}
