<?php

namespace WPOutreach\Mailer\Queue;

use WPOutreach\Mailer\Template\TemplateEngine;
use WPOutreach\Tracking\OpenTracker;
use WPOutreach\Tracking\ClickTracker;

/**
 * Email queue manager
 *
 * Handles adding emails to queue and managing queue state
 */
class QueueManager
{
    /**
     * @var string Queue table name
     */
    private string $table;

    /**
     * @var TemplateEngine Template engine for personalization
     */
    private TemplateEngine $templateEngine;

    /**
     * @var array Debug info from last queueCampaign call
     */
    public array $lastDebug = [];

    /**
     * Constructor
     */
    public function __construct()
    {
        global $wpdb;
        $this->table = $wpdb->prefix . 'outreach_queue';
        $this->templateEngine = new TemplateEngine();
    }

    /**
     * Add email to queue
     *
     * @param array $data Email data
     * @return int|false Insert ID or false on failure
     */
    public function add(array $data): int|false
    {
        global $wpdb;

        $defaults = [
            'campaign_id' => null,
            'automation_id' => null,
            'subscriber_id' => 0,
            'to_email' => '',
            'subject' => '',
            'content' => '',
            'tracking_id' => null,
            'priority' => 5,
            'status' => 'pending',
            'attempts' => 0,
            'scheduled_at' => current_time('mysql'),
        ];

        $data = array_merge($defaults, $data);

        // Validate required fields
        if (empty($data['to_email']) || empty($data['subject']) || empty($data['content'])) {
            return false;
        }

        $result = $wpdb->insert($this->table, [
            'campaign_id' => $data['campaign_id'],
            'automation_id' => $data['automation_id'],
            'subscriber_id' => $data['subscriber_id'],
            'to_email' => sanitize_email($data['to_email']),
            'subject' => sanitize_text_field($data['subject']),
            'content' => $data['content'],
            'tracking_id' => $data['tracking_id'],
            'priority' => (int) $data['priority'],
            'status' => $data['status'],
            'attempts' => (int) $data['attempts'],
            'scheduled_at' => $data['scheduled_at'],
        ]);

        if ($result === false) {
            error_log("WP Outreach Queue: Insert failed - " . $wpdb->last_error);
        }

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Add multiple emails to queue
     *
     * @param array $emails Array of email data
     * @return int Number of emails added
     */
    public function addBatch(array $emails): int
    {
        $count = 0;

        foreach ($emails as $email) {
            if ($this->add($email)) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Queue campaign to all subscribers with personalization
     *
     * @param int $campaignId Campaign ID
     * @param array $subscriberIds Array of subscriber IDs
     * @param string $subject Email subject (may contain merge tags)
     * @param string $content Email content (may contain merge tags)
     * @param bool $enableTracking Whether to add open tracking pixel
     * @return int Number of emails queued
     */
    public function queueCampaign(int $campaignId, array $subscriberIds, string $subject, string $content, bool $enableTracking = true): int
    {
        global $wpdb;

        // Initialize debug info
        $this->lastDebug = [
            'campaign_id' => $campaignId,
            'input_subscriber_count' => count($subscriberIds),
        ];

        if (empty($subscriberIds)) {
            $this->lastDebug['error'] = 'No subscriber IDs provided';
            return 0;
        }

        if (empty($subject)) {
            $this->lastDebug['error'] = 'Empty subject';
            return 0;
        }

        if (empty($content)) {
            $this->lastDebug['error'] = 'Empty content';
            return 0;
        }

        $subscribersTable = $wpdb->prefix . 'outreach_subscribers';
        $now = current_time('mysql');
        $count = 0;

        // Check if tracking is enabled in settings
        $trackingSettings = get_option('wp_outreach_tracking', []);
        $openTrackingEnabled = $enableTracking && ($trackingSettings['open_tracking'] ?? true);
        $clickTrackingEnabled = $enableTracking && ($trackingSettings['click_tracking'] ?? true);

        // Get full subscriber data for personalization
        $placeholders = implode(',', array_fill(0, count($subscriberIds), '%d'));
        $subscribers = $wpdb->get_results($wpdb->prepare(
            "SELECT id, email, first_name, last_name, token, custom_fields
             FROM {$subscribersTable}
             WHERE id IN ({$placeholders}) AND status = 'active'",
            ...$subscriberIds
        ), ARRAY_A);

        $this->lastDebug['subscribers_found'] = count($subscribers);

        if (empty($subscribers)) {
            $this->lastDebug['error'] = 'No subscribers found after query';
            return 0;
        }

        foreach ($subscribers as $subscriber) {
            // Parse custom fields JSON if present
            if (!empty($subscriber['custom_fields'])) {
                $customFields = json_decode($subscriber['custom_fields'], true) ?: [];
                $subscriber = array_merge($subscriber, $customFields);
            }

            // Personalize subject and content for this subscriber
            $personalizedSubject = $this->templateEngine->parseVariables($subject, $subscriber);
            $personalizedContent = $this->templateEngine->render($content, $subscriber, false);

            // Generate tracking ID for both open and click tracking
            $trackingId = null;
            if ($openTrackingEnabled || $clickTrackingEnabled) {
                $trackingId = OpenTracker::generateTrackingId();
            }

            // Insert open tracking pixel
            if ($openTrackingEnabled && $trackingId) {
                $personalizedContent = OpenTracker::insertPixel($personalizedContent, $trackingId);
            }

            // Rewrite links for click tracking
            if ($clickTrackingEnabled && $trackingId) {
                $personalizedContent = ClickTracker::rewriteLinks($personalizedContent, $trackingId, $campaignId);
            }

            $addResult = $this->add([
                'campaign_id' => $campaignId,
                'subscriber_id' => $subscriber['id'],
                'to_email' => $subscriber['email'],
                'subject' => $personalizedSubject,
                'content' => $personalizedContent,
                'tracking_id' => $trackingId,
                'scheduled_at' => $now,
            ]);

            if ($addResult) {
                $count++;

                // Create log entry for tracking
                if ($trackingId) {
                    OpenTracker::createLogEntry([
                        'campaign_id' => $campaignId,
                        'subscriber_id' => $subscriber['id'],
                        'email' => $subscriber['email'],
                        'subject' => $personalizedSubject,
                        'tracking_id' => $trackingId,
                    ]);
                }
            }
        }

        return $count;
    }

    /**
     * Get pending emails from queue
     *
     * @param int $limit Maximum emails to fetch
     * @param int $maxAttempts Maximum retry attempts
     * @return array Queue items
     */
    public function getPending(int $limit = 50, int $maxAttempts = 3): array
    {
        global $wpdb;

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table}
             WHERE status IN ('pending', 'failed')
             AND attempts < %d
             AND scheduled_at <= %s
             ORDER BY priority ASC, scheduled_at ASC
             LIMIT %d",
            $maxAttempts,
            current_time('mysql'),
            $limit
        ));
    }

    /**
     * Mark queue item as processing
     */
    public function markProcessing(int $id): bool
    {
        global $wpdb;

        return (bool) $wpdb->update(
            $this->table,
            ['status' => 'processing'],
            ['id' => $id]
        );
    }

    /**
     * Mark queue item as sent
     */
    public function markSent(int $id): bool
    {
        global $wpdb;

        return (bool) $wpdb->update(
            $this->table,
            [
                'status' => 'sent',
                'sent_at' => current_time('mysql'),
                'error' => null,
            ],
            ['id' => $id]
        );
    }

    /**
     * Mark queue item as failed
     */
    public function markFailed(int $id, string $error): bool
    {
        global $wpdb;

        // Increment attempts
        $wpdb->query($wpdb->prepare(
            "UPDATE {$this->table} SET attempts = attempts + 1, status = 'failed', error = %s WHERE id = %d",
            $error,
            $id
        ));

        return true;
    }

    /**
     * Get queue statistics
     */
    public function getStats(): array
    {
        global $wpdb;

        $stats = $wpdb->get_results(
            "SELECT status, COUNT(*) as count FROM {$this->table} GROUP BY status",
            OBJECT_K
        );

        return [
            'pending' => (int) ($stats['pending']->count ?? 0),
            'processing' => (int) ($stats['processing']->count ?? 0),
            'sent' => (int) ($stats['sent']->count ?? 0),
            'failed' => (int) ($stats['failed']->count ?? 0),
            'total' => array_sum(array_column((array) $stats, 'count')),
        ];
    }

    /**
     * Get campaign queue progress
     */
    public function getCampaignProgress(int $campaignId): array
    {
        global $wpdb;

        $stats = $wpdb->get_results($wpdb->prepare(
            "SELECT status, COUNT(*) as count FROM {$this->table} WHERE campaign_id = %d GROUP BY status",
            $campaignId
        ), OBJECT_K);

        $total = array_sum(array_column((array) $stats, 'count'));
        $sent = (int) ($stats['sent']->count ?? 0);

        return [
            'total' => $total,
            'sent' => $sent,
            'pending' => (int) ($stats['pending']->count ?? 0),
            'failed' => (int) ($stats['failed']->count ?? 0),
            'progress' => $total > 0 ? round(($sent / $total) * 100, 1) : 0,
        ];
    }

    /**
     * Delete sent emails older than X days
     */
    public function cleanup(int $days = 30): int
    {
        global $wpdb;

        $date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        return (int) $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table} WHERE status = 'sent' AND sent_at < %s",
            $date
        ));
    }

    /**
     * Cancel pending campaign emails
     */
    public function cancelCampaign(int $campaignId): int
    {
        global $wpdb;

        return (int) $wpdb->delete($this->table, [
            'campaign_id' => $campaignId,
            'status' => 'pending',
        ]);
    }

    /**
     * Retry failed emails for a campaign
     */
    public function retryCampaign(int $campaignId): int
    {
        global $wpdb;

        return (int) $wpdb->update(
            $this->table,
            [
                'status' => 'pending',
                'attempts' => 0,
                'error' => null,
            ],
            [
                'campaign_id' => $campaignId,
                'status' => 'failed',
            ]
        );
    }
}
