<?php

namespace WPOutreach\Mailer;

use WPOutreach\License\FeatureManager;

/**
 * Factory for creating mailer instances
 */
class MailerFactory
{
    /**
     * @var array Registered mailer classes
     */
    private static array $mailers = [
        'wp_mail' => WPMailer::class,
        'smtp' => SMTPMailer::class,
        'ses' => SESMailer::class,
    ];

    /**
     * @var MailerInterface|null Cached mailer instance
     */
    private static ?MailerInterface $instance = null;

    /**
     * Get the configured mailer
     */
    public static function create(?string $driver = null): MailerInterface
    {
        if ($driver === null) {
            $settings = get_option('wp_outreach_mailer', []);
            $driver = $settings['driver'] ?? 'wp_mail';
        }

        // Validate driver
        if (!isset(self::$mailers[$driver])) {
            $driver = 'wp_mail';
        }

        // License enforcement: Free users can only use wp_mail
        // Even if they have SMTP/SES configured, force wp_mail
        if ($driver !== 'wp_mail' && !FeatureManager::canUseMailer($driver)) {
            $driver = 'wp_mail';
        }

        $class = self::$mailers[$driver];
        $mailer = new $class();

        // Fall back to wp_mail if configured mailer isn't ready
        if (!$mailer->isConfigured() && $driver !== 'wp_mail') {
            return new WPMailer();
        }

        return $mailer;
    }

    /**
     * Get singleton mailer instance
     */
    public static function getInstance(): MailerInterface
    {
        if (self::$instance === null) {
            self::$instance = self::create();
        }

        return self::$instance;
    }

    /**
     * Reset singleton (for testing or config changes)
     */
    public static function reset(): void
    {
        self::$instance = null;
    }

    /**
     * Register a custom mailer
     */
    public static function register(string $name, string $class): void
    {
        if (!is_subclass_of($class, MailerInterface::class)) {
            throw new \InvalidArgumentException(
                sprintf('%s must implement MailerInterface', $class)
            );
        }

        self::$mailers[$name] = $class;
    }

    /**
     * Get all available mailers
     */
    public static function getAvailableMailers(): array
    {
        $available = [];

        foreach (self::$mailers as $name => $class) {
            $mailer = new $class();
            $available[$name] = [
                'name' => $name,
                'configured' => $mailer->isConfigured(),
                'label' => self::getMailerLabel($name),
            ];
        }

        return $available;
    }

    /**
     * Get human-readable label for mailer
     */
    private static function getMailerLabel(string $name): string
    {
        $labels = [
            'wp_mail' => __('WordPress Mail', 'outreach'),
            'smtp' => __('SMTP', 'outreach'),
            'ses' => __('Amazon SES', 'outreach'),
        ];

        return $labels[$name] ?? $name;
    }
}
