<?php

namespace WPOutreach\Mailer;

/**
 * Abstract base class for mailers
 */
abstract class AbstractMailer implements MailerInterface
{
    /**
     * @var string|null Last error message
     */
    protected ?string $lastError = null;

    /**
     * @var array Default options
     */
    protected array $defaults = [];

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->loadDefaults();
    }

    /**
     * Load default settings from WordPress options
     */
    protected function loadDefaults(): void
    {
        $general = get_option('wp_outreach_general', []);

        $this->defaults = [
            'from_name' => $general['from_name'] ?? get_bloginfo('name'),
            'from_email' => $general['from_email'] ?? get_option('admin_email'),
            'reply_to' => $general['reply_to'] ?? '',
        ];
    }

    /**
     * Merge options with defaults
     */
    protected function mergeOptions(array $options): array
    {
        return array_merge($this->defaults, $options);
    }

    /**
     * Get the last error message
     */
    public function getLastError(): ?string
    {
        return $this->lastError;
    }

    /**
     * Set error message
     */
    protected function setError(string $message): void
    {
        $this->lastError = $message;
    }

    /**
     * Clear error
     */
    protected function clearError(): void
    {
        $this->lastError = null;
    }

    /**
     * Send email to multiple recipients (default implementation)
     */
    public function sendBatch(array $recipients, string $subject, string $body, array $options = []): array
    {
        $results = [];

        foreach ($recipients as $email) {
            $results[$email] = $this->send($email, $subject, $body, $options);
        }

        return $results;
    }

    /**
     * Build email headers array
     */
    protected function buildHeaders(array $options): array
    {
        $headers = [];

        // From header
        if (!empty($options['from_email'])) {
            $from_name = $options['from_name'] ?? '';
            if ($from_name) {
                $headers[] = sprintf('From: %s <%s>', $from_name, $options['from_email']);
            } else {
                $headers[] = sprintf('From: %s', $options['from_email']);
            }
        }

        // Reply-To header
        if (!empty($options['reply_to'])) {
            $headers[] = sprintf('Reply-To: %s', $options['reply_to']);
        }

        // Content-Type
        $headers[] = 'Content-Type: text/html; charset=UTF-8';

        // Custom headers
        if (!empty($options['headers']) && is_array($options['headers'])) {
            $headers = array_merge($headers, $options['headers']);
        }

        return $headers;
    }

    /**
     * Wrap HTML content in email template
     */
    protected function wrapHtml(string $body): string
    {
        // If body already has doctype, return as-is
        if (stripos($body, '<!DOCTYPE') !== false || stripos($body, '<html') !== false) {
            return $body;
        }

        // Wrap in basic HTML structure
        return sprintf(
            '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, sans-serif;">
    %s
</body>
</html>',
            $body
        );
    }
}
