<?php

namespace WPOutreach\License;

/**
 * Feature Manager
 *
 * Handles license-based feature restrictions for free vs paid users.
 *
 * FREE PLAN Restrictions:
 * - Subscribers: Up to 3,000
 * - Templates: Cannot save custom templates
 * - Import/Export: Disabled
 * - Automations: Basic triggers only (subscriber_created, subscriber_joins_list, wp_user_registered, post_published)
 * - Post Subscriptions: Disabled
 * - Email Driver: WP Mail only
 * - Tracking: Disabled
 * - Analytics: Disabled
 *
 * PAID PLAN: Everything unlimited, all features enabled
 *
 * @since 1.0.0
 */
class FeatureManager
{
    /**
     * Maximum subscribers allowed for free plan
     */
    public const FREE_SUBSCRIBER_LIMIT = 3000;

    /**
     * Allowed automation triggers for free plan
     */
    public const FREE_TRIGGERS = [
        'subscriber_created',
        'subscriber_joins_list',
        'wp_user_registered',
        'post_published',
    ];

    /**
     * Features that require a paid license
     */
    public const PREMIUM_FEATURES = [
        'templates',           // Save custom templates
        'import_export',       // Import/export subscribers
        'advanced_triggers',   // All triggers beyond basic ones
        'post_subscriptions',  // Post/page subscription notifications
        'smtp',                // SMTP email driver
        'ses',                 // Amazon SES email driver
        'tracking',            // Open/click tracking
        'analytics',           // Analytics dashboard
    ];

    /**
     * Cache for license status
     */
    private static ?bool $isPro = null;

    /**
     * Check if the current installation has a valid license
     *
     * @return bool True if license is valid and not expired
     */
    public static function hasValidLicense(): bool
    {
        if (self::$isPro !== null) {
            return self::$isPro;
        }

        // Allow bypass via constant for development/owner sites
        if (defined('WP_OUTREACH_PRO') && WP_OUTREACH_PRO === true) {
            self::$isPro = true;
            return true;
        }

        // Allow bypass via filter
        if (apply_filters('wp_outreach_is_pro', false)) {
            self::$isPro = true;
            return true;
        }

        // Check new format first
        $license = get_option('wp_outreach_license', []);

        // Check if license exists and is active
        if (!empty($license['key']) && !empty($license['status'])) {
            if ($license['status'] === 'active') {
                // Check expiration
                if (!empty($license['expires_at'])) {
                    $expires = strtotime($license['expires_at']);
                    if ($expires && $expires < time()) {
                        self::$isPro = false;
                        return false;
                    }
                }

                self::$isPro = true;
                return true;
            }
        }

        // Fallback: Check legacy format (wp_outreach_license_status)
        $legacyStatus = get_option('wp_outreach_license_status', '');
        if ($legacyStatus === 'valid') {
            // Check legacy expiration
            $legacyData = get_option('wp_outreach_license_data', []);
            if (!empty($legacyData['expire_timestamp']) && $legacyData['expire_timestamp'] > 0) {
                if ($legacyData['expire_timestamp'] < time()) {
                    self::$isPro = false;
                    return false;
                }
            }

            self::$isPro = true;
            return true;
        }

        self::$isPro = false;
        return false;
    }

    /**
     * Check if a specific feature is available
     *
     * @param string $feature Feature identifier
     * @return bool True if feature is available
     */
    public static function can(string $feature): bool
    {
        // Pro users have access to everything
        if (self::hasValidLicense()) {
            return true;
        }

        // Free users don't have access to premium features
        return !in_array($feature, self::PREMIUM_FEATURES, true);
    }

    /**
     * Check if a specific automation trigger is available
     *
     * @param string $trigger Trigger identifier
     * @return bool True if trigger is available
     */
    public static function canUseTrigger(string $trigger): bool
    {
        // Pro users have access to all triggers
        if (self::hasValidLicense()) {
            return true;
        }

        // Free users only have access to basic triggers
        return in_array($trigger, self::FREE_TRIGGERS, true);
    }

    /**
     * Get the subscriber limit for the current plan
     *
     * @return int Subscriber limit (-1 for unlimited)
     */
    public static function getSubscriberLimit(): int
    {
        if (self::hasValidLicense()) {
            return -1; // Unlimited
        }

        return self::FREE_SUBSCRIBER_LIMIT;
    }

    /**
     * Check if the subscriber limit has been reached
     *
     * @return bool True if limit reached
     */
    public static function isSubscriberLimitReached(): bool
    {
        if (self::hasValidLicense()) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");

        return $count >= self::FREE_SUBSCRIBER_LIMIT;
    }

    /**
     * Get the current subscriber count
     *
     * @return int Current number of subscribers
     */
    public static function getSubscriberCount(): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    }

    /**
     * Get the number of remaining subscriber slots
     *
     * @return int Remaining slots (-1 for unlimited)
     */
    public static function getRemainingSubscriberSlots(): int
    {
        if (self::hasValidLicense()) {
            return -1; // Unlimited
        }

        $count = self::getSubscriberCount();
        $remaining = self::FREE_SUBSCRIBER_LIMIT - $count;

        return max(0, $remaining);
    }

    /**
     * Get allowed email drivers for current plan
     *
     * @return array List of allowed driver identifiers
     */
    public static function getAllowedMailers(): array
    {
        if (self::hasValidLicense()) {
            return ['wp_mail', 'smtp', 'ses'];
        }

        return ['wp_mail'];
    }

    /**
     * Check if a specific mailer is allowed
     *
     * @param string $mailer Mailer identifier
     * @return bool True if mailer is allowed
     */
    public static function canUseMailer(string $mailer): bool
    {
        return in_array($mailer, self::getAllowedMailers(), true);
    }

    /**
     * Get available triggers for current plan
     *
     * @return array List of available trigger identifiers (empty = all available)
     */
    public static function getAvailableTriggers(): array
    {
        if (self::hasValidLicense()) {
            return []; // Empty means all available
        }

        return self::FREE_TRIGGERS;
    }

    /**
     * Get license status information for display
     *
     * @return array License status details
     */
    public static function getStatus(): array
    {
        $license = get_option('wp_outreach_license', []);
        $isValid = self::hasValidLicense();

        return [
            'is_pro' => $isValid,
            'plan' => $isValid ? 'pro' : 'free',
            'key' => !empty($license['key']) ? self::maskLicenseKey($license['key']) : null,
            'status' => $license['status'] ?? 'inactive',
            'expires_at' => $license['expires_at'] ?? null,
            'subscriber_limit' => self::getSubscriberLimit(),
            'subscriber_count' => self::getSubscriberCount(),
            'remaining_slots' => self::getRemainingSubscriberSlots(),
            'allowed_mailers' => self::getAllowedMailers(),
            'available_triggers' => self::getAvailableTriggers(),
            'features' => [
                'templates' => self::can('templates'),
                'import_export' => self::can('import_export'),
                'advanced_triggers' => self::can('advanced_triggers'),
                'post_subscriptions' => self::can('post_subscriptions'),
                'smtp' => self::can('smtp'),
                'ses' => self::can('ses'),
                'tracking' => self::can('tracking'),
                'analytics' => self::can('analytics'),
            ],
        ];
    }

    /**
     * Mask a license key for display
     *
     * @param string $key Full license key
     * @return string Masked key (e.g., "WPO-XXXX-****-****")
     */
    private static function maskLicenseKey(string $key): string
    {
        if (strlen($key) <= 8) {
            return str_repeat('*', strlen($key));
        }

        return substr($key, 0, 8) . str_repeat('*', strlen($key) - 8);
    }

    /**
     * Clear the cached license status
     * Call this after license changes
     */
    public static function clearCache(): void
    {
        self::$isPro = null;
    }

    /**
     * Get upgrade URL
     *
     * @return string URL to upgrade page
     */
    public static function getUpgradeUrl(): string
    {
        return 'https://wpoutreach.com/pricing/';
    }

    /**
     * Get a user-friendly error message for a restricted feature
     *
     * @param string $feature Feature identifier
     * @return string Error message
     */
    public static function getRestrictionMessage(string $feature): string
    {
        $messages = [
            'templates' => 'Saving custom templates requires a Pro license.',
            'import_export' => 'Import/Export requires a Pro license.',
            'advanced_triggers' => 'This automation trigger requires a Pro license.',
            'post_subscriptions' => 'Post Subscriptions require a Pro license.',
            'smtp' => 'SMTP email driver requires a Pro license.',
            'ses' => 'Amazon SES integration requires a Pro license.',
            'tracking' => 'Email tracking requires a Pro license.',
            'analytics' => 'Analytics dashboard requires a Pro license.',
            'subscriber_limit' => sprintf(
                'Free plan is limited to %s subscribers. Upgrade to Pro for unlimited subscribers.',
                number_format(self::FREE_SUBSCRIBER_LIMIT)
            ),
        ];

        return $messages[$feature] ?? 'This feature requires a Pro license.';
    }
}
