<?php

namespace WPOutreach\Integrations;

use WPOutreach\Automation\TriggerRegistry;

/**
 * WordPress Download Manager Integration
 *
 * Integrates WP Outreach with WordPress Download Manager to provide
 * automation triggers for downloads, email locks, and purchases.
 *
 * @since 1.0.0
 */
class WPDMIntegration
{
    /**
     * Initialize the integration
     */
    public static function init(): void
    {
        // Only load if WPDM is active
        if (!self::isWPDMCoreActive()) {
            return;
        }

        // Register WPDM triggers
        add_action('wp_outreach_register_triggers', [self::class, 'registerTriggers']);

        // Core WPDM hooks
        add_action('wpdm_before_email_download_link', [self::class, 'onEmailLockSubmitted'], 10, 2);

        // Premium Packages hooks (only if Premium Packages is active)
        if (self::isWPDMPremiumActive()) {
            add_action('wpdmpp_payment_completed', [self::class, 'onPaymentCompleted'], 10, 1);
        }
    }

    /**
     * Check if WPDM Core is active
     *
     * @return bool
     */
    public static function isWPDMCoreActive(): bool
    {
        return defined('WPDM_VERSION') || class_exists('WPDM');
    }

    /**
     * Check if WPDM Premium Packages is active
     *
     * @return bool
     */
    public static function isWPDMPremiumActive(): bool
    {
        return class_exists('WPDMPP\Libs\Order') || function_exists('wpdmpp_get_order');
    }

    /**
     * Register WPDM-specific triggers
     *
     * @param TriggerRegistry $registry
     */
    public static function registerTriggers(TriggerRegistry $registry): void
    {
        // Email Lock trigger (Core WPDM)
        $registry->register([
            'id'          => 'wpdm_email_lock_submitted',
            'name'        => __('Email Lock Submitted', 'outreach'),
            'description' => __('Fires when someone submits their email to unlock a download', 'outreach'),
            'group'       => __('Download Manager', 'outreach'),
            'icon'        => 'mail',
            'source'      => 'integration',
            'config_fields' => [],
        ]);

        // Purchase trigger (Premium Packages only)
        if (self::isWPDMPremiumActive()) {
            $registry->register([
                'id'          => 'wpdm_new_purchase',
                'name'        => __('WPDM Purchase', 'outreach'),
                'description' => __('Fires when a customer completes a Download Manager purchase', 'outreach'),
                'group'       => __('Download Manager', 'outreach'),
                'icon'        => 'shopping-cart',
                'source'      => 'integration',
                'config_fields' => [],
            ]);
        }
    }

    /**
     * Handle email lock submission
     *
     * @param array $postData POST data containing email
     * @param array $context  Context with package ID
     */
    public static function onEmailLockSubmitted(array $postData, array $context): void
    {
        $email = sanitize_email($postData['email'] ?? '');
        $packageId = intval($context['ID'] ?? 0);

        if (empty($email) || !$packageId) {
            return;
        }

        // Get package info
        $package = get_post($packageId);
        if (!$package) {
            return;
        }

        // Find or create subscriber
        $subscriberId = self::findOrCreateSubscriberByEmail($email, 'wpdm_email_lock');

        if (!$subscriberId) {
            return;
        }

        // Fire the trigger with package context
        TriggerRegistry::fire('wpdm_email_lock_submitted', $subscriberId, [
            'package_id'    => $packageId,
            'package_title' => $package->post_title,
            'package_url'   => get_permalink($packageId),
        ]);
    }

    /**
     * Find or create subscriber by email
     *
     * @param string $email
     * @param string $source
     * @return int|null
     */
    private static function findOrCreateSubscriberByEmail(string $email, string $source = 'wpdm'): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'status'     => 'active', // Email lock = consent
            'source'     => $source,
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriberId = $wpdb->insert_id;

        // Fire subscriber created action
        do_action('wp_outreach_subscriber_created', $subscriberId, [
            'email'  => $email,
            'source' => $source,
        ]);

        return $subscriberId;
    }

    /**
     * Handle payment completed event
     *
     * @param string $order_id The order ID
     */
    public static function onPaymentCompleted(string $order_id): void
    {
        // Get order data
        $order_data = self::getOrderData($order_id);

        if (!$order_data) {
            return;
        }

        // Find or create subscriber from order email
        $subscriber_id = self::findOrCreateSubscriber($order_data);

        if (!$subscriber_id) {
            return;
        }

        // Fire the trigger with order context
        TriggerRegistry::fire('wpdm_new_purchase', $subscriber_id, $order_data);
    }

    /**
     * Get order data from order ID
     *
     * @param string $order_id
     * @return array|null
     */
    private static function getOrderData(string $order_id): ?array
    {
        // Use WPDM's Order class if available
        if (class_exists('WPDMPP\Libs\Order')) {
            $order = new \WPDMPP\Libs\Order($order_id);

            if (!$order->oid) {
                return null;
            }

            // Get billing info
            $billing_info = [];
            if (isset($order->billing_info)) {
                $billing_info = is_array($order->billing_info)
                    ? $order->billing_info
                    : maybe_unserialize($order->billing_info);
            }

            // Get cart data for product info
            $cart_data = [];
            if (isset($order->cart_data)) {
                $cart_data = is_array($order->cart_data)
                    ? $order->cart_data
                    : maybe_unserialize($order->cart_data);
            }

            // Extract product IDs and names
            $product_ids = [];
            $product_names = [];

            foreach ($cart_data as $item) {
                if (isset($item['pid'])) {
                    $product_ids[] = (int) $item['pid'];
                    $product_names[] = $item['product_name'] ?? get_the_title($item['pid']);
                }
            }

            // Get customer email - try billing info first, then user
            $email = $billing_info['email'] ?? '';

            if (empty($email) && $order->uid > 0) {
                $user = get_user_by('id', $order->uid);
                if ($user) {
                    $email = $user->user_email;
                }
            }

            if (empty($email)) {
                return null;
            }

            // Get currency info
            $currency = 'USD';
            if (isset($order->currency)) {
                $currency_data = is_array($order->currency)
                    ? $order->currency
                    : maybe_unserialize($order->currency);
                $currency = $currency_data['code'] ?? 'USD';
            }

            return [
                'order_id'       => $order_id,
                'order_total'    => (float) ($order->total ?? 0),
                'subtotal'       => (float) ($order->subtotal ?? 0),
                'tax'            => (float) ($order->tax ?? 0),
                'discount'       => (float) ($order->coupon_discount ?? 0),
                'currency'       => $currency,
                'product_ids'    => $product_ids,
                'product_names'  => $product_names,
                'customer_email' => $email,
                'customer_name'  => trim(($billing_info['first_name'] ?? '') . ' ' . ($billing_info['last_name'] ?? '')),
                'first_name'     => $billing_info['first_name'] ?? '',
                'last_name'      => $billing_info['last_name'] ?? '',
                'user_id'        => (int) ($order->uid ?? 0),
                'payment_method' => $order->payment_method ?? '',
                'coupon_code'    => $order->coupon_code ?? '',
            ];
        }

        return null;
    }

    /**
     * Find existing subscriber or create new one from order data
     *
     * @param array $order_data
     * @return int|null Subscriber ID or null on failure
     */
    private static function findOrCreateSubscriber(array $order_data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $email = $order_data['customer_email'] ?? '';

        if (empty($email)) {
            return null;
        }

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, status FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            // Update name if missing
            if (!empty($order_data['first_name']) || !empty($order_data['last_name'])) {
                $wpdb->update(
                    $table,
                    [
                        'first_name' => $order_data['first_name'] ?: $subscriber->first_name ?? '',
                        'last_name'  => $order_data['last_name'] ?: $subscriber->last_name ?? '',
                    ],
                    ['id' => $subscriber->id]
                );
            }

            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'first_name' => $order_data['first_name'] ?? '',
            'last_name'  => $order_data['last_name'] ?? '',
            'status'     => 'active', // Customers are auto-confirmed
            'source'     => 'wpdm_purchase',
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        // Fire subscriber created action
        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $email,
            'first_name' => $order_data['first_name'] ?? '',
            'last_name'  => $order_data['last_name'] ?? '',
            'source'     => 'wpdm_purchase',
        ]);

        return $subscriber_id;
    }

    /**
     * Get all WPDM products for dropdown
     *
     * @return array
     */
    public static function getProducts(): array
    {
        $products = get_posts([
            'post_type'      => 'wpdmpro',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ]);

        $result = [];

        foreach ($products as $product) {
            $price = 0;
            if (function_exists('wpdmpp_effective_price')) {
                $price = wpdmpp_effective_price($product->ID);
            }

            // Only include products with a price (premium packages)
            if ($price > 0) {
                $result[] = [
                    'id'    => $product->ID,
                    'name'  => $product->post_title,
                    'price' => $price,
                ];
            }
        }

        return $result;
    }
}
