<?php

namespace WPOutreach\Integrations\Forms;

use WPOutreach\Automation\TriggerRegistry;

/**
 * WooCommerce Integration
 *
 * Provides automation triggers for WooCommerce checkout as a form submission,
 * as well as order-related triggers.
 *
 * @since 1.3.0
 */
class WooCommerceIntegration
{
    /**
     * Initialize the integration
     */
    public static function init(): void
    {
        // Always register triggers so users can see what's available
        add_action('wp_outreach_register_triggers', [self::class, 'registerTriggers']);

        // Only hook into WooCommerce events if it's active
        if (self::isActive()) {
            // Hook into checkout completion
            add_action('woocommerce_checkout_order_processed', [self::class, 'onCheckoutSubmit'], 10, 3);

            // Hook into payment completion for paid orders
            add_action('woocommerce_payment_complete', [self::class, 'onPaymentComplete'], 10, 1);

            // Hook into order status changes
            add_action('woocommerce_order_status_completed', [self::class, 'onOrderCompleted'], 10, 1);
        }
    }

    /**
     * Check if WooCommerce is active
     *
     * @return bool
     */
    public static function isActive(): bool
    {
        return class_exists('WooCommerce') || function_exists('WC');
    }

    /**
     * Register WooCommerce triggers
     *
     * @param TriggerRegistry $registry
     */
    public static function registerTriggers(TriggerRegistry $registry): void
    {
        $isActive = self::isActive();
        $requires = $isActive ? null : __('WooCommerce plugin', 'outreach');

        $registry->register([
            'id'          => 'wc_checkout_submitted',
            'name'        => __('WooCommerce Checkout', 'outreach'),
            'description' => __('Fires when a customer completes the checkout form', 'outreach'),
            'group'       => __('Forms', 'outreach'),
            'icon'        => 'shopping-cart',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $requires,
            'config_fields' => [],
        ]);

        $registry->register([
            'id'          => 'wc_order_paid',
            'name'        => __('WooCommerce Order Paid', 'outreach'),
            'description' => __('Fires when a WooCommerce order payment is completed', 'outreach'),
            'group'       => __('WooCommerce', 'outreach'),
            'icon'        => 'shopping-cart',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $requires,
            'config_fields' => $isActive ? [
                [
                    'key'         => 'product_id',
                    'label'       => __('Contains Product', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any product', 'outreach'),
                    'options'     => [], // Loaded dynamically
                ],
            ] : [],
            'matcher' => function ($config, $context) {
                if (empty($config['product_id'])) {
                    return true; // Any product
                }
                $product_ids = $context['product_ids'] ?? [];
                return in_array((int) $config['product_id'], $product_ids, true);
            },
        ]);

        $registry->register([
            'id'          => 'wc_order_completed',
            'name'        => __('WooCommerce Order Completed', 'outreach'),
            'description' => __('Fires when an order status changes to completed', 'outreach'),
            'group'       => __('WooCommerce', 'outreach'),
            'icon'        => 'shopping-cart',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $requires,
            'config_fields' => $isActive ? [
                [
                    'key'         => 'product_id',
                    'label'       => __('Contains Product', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any product', 'outreach'),
                    'options'     => [], // Loaded dynamically
                ],
            ] : [],
            'matcher' => function ($config, $context) {
                if (empty($config['product_id'])) {
                    return true; // Any product
                }
                $product_ids = $context['product_ids'] ?? [];
                return in_array((int) $config['product_id'], $product_ids, true);
            },
        ]);
    }

    /**
     * Handle checkout form submission
     *
     * @param int           $order_id
     * @param array         $posted_data
     * @param \WC_Order     $order
     */
    public static function onCheckoutSubmit($order_id, $posted_data, $order): void
    {
        $email = $order->get_billing_email();

        if (empty($email)) {
            return;
        }

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($order);

        if (!$subscriber_id) {
            return;
        }

        // Build context
        $context = self::buildOrderContext($order, 'checkout_submitted');
        $context['form_plugin'] = 'woocommerce';

        // Add form fields from checkout
        $context['fields'] = [
            'billing_first_name'  => $posted_data['billing_first_name'] ?? '',
            'billing_last_name'   => $posted_data['billing_last_name'] ?? '',
            'billing_email'       => $posted_data['billing_email'] ?? '',
            'billing_phone'       => $posted_data['billing_phone'] ?? '',
            'billing_company'     => $posted_data['billing_company'] ?? '',
            'billing_address_1'   => $posted_data['billing_address_1'] ?? '',
            'billing_city'        => $posted_data['billing_city'] ?? '',
            'billing_state'       => $posted_data['billing_state'] ?? '',
            'billing_postcode'    => $posted_data['billing_postcode'] ?? '',
            'billing_country'     => $posted_data['billing_country'] ?? '',
            'order_comments'      => $posted_data['order_comments'] ?? '',
        ];

        // Fire the checkout trigger
        TriggerRegistry::fire('wc_checkout_submitted', $subscriber_id, $context);
    }

    /**
     * Handle payment completion
     *
     * @param int $order_id
     */
    public static function onPaymentComplete($order_id): void
    {
        $order = wc_get_order($order_id);

        if (!$order) {
            return;
        }

        $email = $order->get_billing_email();

        if (empty($email)) {
            return;
        }

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($order);

        if (!$subscriber_id) {
            return;
        }

        // Build context
        $context = self::buildOrderContext($order, 'order_paid');

        // Fire the payment trigger
        TriggerRegistry::fire('wc_order_paid', $subscriber_id, $context);
    }

    /**
     * Handle order completed status
     *
     * @param int $order_id
     */
    public static function onOrderCompleted($order_id): void
    {
        $order = wc_get_order($order_id);

        if (!$order) {
            return;
        }

        $email = $order->get_billing_email();

        if (empty($email)) {
            return;
        }

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($order);

        if (!$subscriber_id) {
            return;
        }

        // Build context
        $context = self::buildOrderContext($order, 'order_completed');

        // Fire the completed trigger
        TriggerRegistry::fire('wc_order_completed', $subscriber_id, $context);
    }

    /**
     * Build order context data
     *
     * @param \WC_Order $order
     * @param string    $event
     * @return array
     */
    private static function buildOrderContext(\WC_Order $order, string $event): array
    {
        $product_ids = [];
        $product_names = [];
        $product_skus = [];

        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $product_ids[] = $product->get_id();
                $product_names[] = $product->get_name();
                $product_skus[] = $product->get_sku();
            }
        }

        $first_name = $order->get_billing_first_name();
        $last_name = $order->get_billing_last_name();

        return [
            'event'          => $event,
            'order_id'       => $order->get_id(),
            'order_number'   => $order->get_order_number(),
            'order_total'    => (float) $order->get_total(),
            'order_subtotal' => (float) $order->get_subtotal(),
            'order_tax'      => (float) $order->get_total_tax(),
            'order_discount' => (float) $order->get_total_discount(),
            'order_shipping' => (float) $order->get_shipping_total(),
            'currency'       => $order->get_currency(),
            'payment_method' => $order->get_payment_method_title(),
            'product_ids'    => $product_ids,
            'product_names'  => $product_names,
            'product_skus'   => $product_skus,
            'item_count'     => $order->get_item_count(),
            'email'          => $order->get_billing_email(),
            'first_name'     => $first_name,
            'last_name'      => $last_name,
            'customer_name'  => trim($first_name . ' ' . $last_name),
            'billing_phone'  => $order->get_billing_phone(),
            'billing_city'   => $order->get_billing_city(),
            'billing_state'  => $order->get_billing_state(),
            'billing_country' => $order->get_billing_country(),
            'user_id'        => $order->get_user_id(),
            'is_returning_customer' => self::isReturningCustomer($order),
            'subscriber'     => [
                'email'      => $order->get_billing_email(),
                'first_name' => $first_name,
                'last_name'  => $last_name,
            ],
        ];
    }

    /**
     * Check if customer is returning
     *
     * @param \WC_Order $order
     * @return bool
     */
    private static function isReturningCustomer(\WC_Order $order): bool
    {
        $email = $order->get_billing_email();

        $order_count = wc_get_customer_order_count(
            $order->get_user_id() > 0 ? $order->get_user_id() : $email
        );

        return $order_count > 1;
    }

    /**
     * Find or create subscriber from order
     *
     * @param \WC_Order $order
     * @return int|null
     */
    private static function findOrCreateSubscriber(\WC_Order $order): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $email = $order->get_billing_email();
        $first_name = $order->get_billing_first_name();
        $last_name = $order->get_billing_last_name();

        if (empty($email)) {
            return null;
        }

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, first_name, last_name FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            // Update name if currently empty
            $updates = [];
            if (empty($subscriber->first_name) && !empty($first_name)) {
                $updates['first_name'] = $first_name;
            }
            if (empty($subscriber->last_name) && !empty($last_name)) {
                $updates['last_name'] = $last_name;
            }

            if (!empty($updates)) {
                $wpdb->update($table, $updates, ['id' => $subscriber->id]);
            }

            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'status'     => 'active',
            'source'     => 'woocommerce',
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $email,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'source'     => 'woocommerce',
        ]);

        return $subscriber_id;
    }

    /**
     * Get all WooCommerce products for dropdown
     *
     * @return array
     */
    public static function getProductsForDropdown(): array
    {
        if (!self::isActive()) {
            return [];
        }

        $products = wc_get_products([
            'status' => 'publish',
            'limit'  => -1,
            'orderby' => 'title',
            'order'  => 'ASC',
        ]);

        $result = [];

        foreach ($products as $product) {
            $result[] = [
                'value' => $product->get_id(),
                'label' => $product->get_name(),
            ];
        }

        return $result;
    }
}
