<?php

namespace WPOutreach\Integrations\Forms;

use WPOutreach\Automation\TriggerRegistry;

/**
 * WPForms Integration
 *
 * Provides automation triggers for WPForms form submissions.
 *
 * @since 1.3.0
 */
class WPFormsIntegration
{
    /**
     * Initialize the integration
     */
    public static function init(): void
    {
        // Always register triggers so users can see what's available
        add_action('wp_outreach_register_triggers', [self::class, 'registerTriggers']);

        // Only hook into form submission if WPForms is active
        if (self::isActive()) {
            add_action('wpforms_process_complete', [self::class, 'onFormSubmit'], 10, 4);
        }
    }

    /**
     * Check if WPForms is active
     *
     * @return bool
     */
    public static function isActive(): bool
    {
        return defined('WPFORMS_VERSION') || function_exists('wpforms');
    }

    /**
     * Register WPForms triggers
     *
     * @param TriggerRegistry $registry
     */
    public static function registerTriggers(TriggerRegistry $registry): void
    {
        $isActive = self::isActive();

        $registry->register([
            'id'          => 'wpforms_form_submitted',
            'name'        => __('WPForms Submission', 'outreach'),
            'description' => __('Fires when a WPForms form is submitted', 'outreach'),
            'group'       => __('Forms', 'outreach'),
            'icon'        => 'mail',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $isActive ? null : __('WPForms plugin', 'outreach'),
            'config_fields' => $isActive ? [
                [
                    'key'         => 'form_id',
                    'label'       => __('Form', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any form', 'outreach'),
                    'options'     => self::getFormsForDropdown(),
                ],
            ] : [],
            'matcher' => function ($config, $context) {
                if (empty($config['form_id'])) {
                    return true; // Any form
                }
                return (int) $config['form_id'] === (int) ($context['form_id'] ?? 0);
            },
        ]);
    }

    /**
     * Handle form submission
     *
     * @param array $fields    Sanitized entry field values.
     * @param array $entry     Original $_POST entry.
     * @param array $form_data Form data and settings.
     * @param int   $entry_id  Entry ID.
     */
    public static function onFormSubmit($fields, $entry, $form_data, $entry_id): void
    {
        $form_id = $form_data['id'] ?? 0;
        $form_title = $form_data['settings']['form_title'] ?? '';

        // Extract email from form fields
        $email = self::extractEmail($fields);

        if (empty($email)) {
            return; // No email found, can't create subscriber
        }

        // Extract name fields
        $name_data = self::extractName($fields);

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($email, $name_data);

        if (!$subscriber_id) {
            return;
        }

        // Build simplified fields array for conditions
        $simple_fields = [];
        foreach ($fields as $field) {
            $field_name = sanitize_key($field['name'] ?? 'field_' . $field['id']);
            $simple_fields[$field_name] = $field['value'] ?? '';
        }

        // Build context for conditions and merge tags
        $context = [
            'form_id'     => $form_id,
            'form_name'   => $form_title,
            'form_plugin' => 'wpforms',
            'fields'      => $simple_fields,
            'raw_fields'  => $fields,
            'email'       => $email,
            'first_name'  => $name_data['first_name'] ?? '',
            'last_name'   => $name_data['last_name'] ?? '',
            'entry_id'    => $entry_id,
            'subscriber'  => [
                'email'      => $email,
                'first_name' => $name_data['first_name'] ?? '',
                'last_name'  => $name_data['last_name'] ?? '',
            ],
        ];

        // Fire the trigger
        TriggerRegistry::fire('wpforms_form_submitted', $subscriber_id, $context);
    }

    /**
     * Extract email from form fields
     *
     * @param array $fields
     * @return string|null
     */
    private static function extractEmail(array $fields): ?string
    {
        foreach ($fields as $field) {
            // WPForms uses 'type' => 'email' for email fields
            if (isset($field['type']) && $field['type'] === 'email') {
                if (!empty($field['value']) && is_email($field['value'])) {
                    return sanitize_email($field['value']);
                }
            }
        }

        // Fallback: search by field name
        foreach ($fields as $field) {
            $name = strtolower($field['name'] ?? '');
            if (strpos($name, 'email') !== false && !empty($field['value']) && is_email($field['value'])) {
                return sanitize_email($field['value']);
            }
        }

        return null;
    }

    /**
     * Extract name from form fields
     *
     * @param array $fields
     * @return array
     */
    private static function extractName(array $fields): array
    {
        $result = ['first_name' => '', 'last_name' => ''];

        foreach ($fields as $field) {
            $type = $field['type'] ?? '';
            $name = strtolower($field['name'] ?? '');

            // WPForms 'name' field type has 'first' and 'last' subfields
            if ($type === 'name') {
                if (!empty($field['first'])) {
                    $result['first_name'] = sanitize_text_field($field['first']);
                }
                if (!empty($field['last'])) {
                    $result['last_name'] = sanitize_text_field($field['last']);
                }
                break;
            }

            // Check for separate first/last name text fields
            if (strpos($name, 'first') !== false && strpos($name, 'name') !== false) {
                $result['first_name'] = sanitize_text_field($field['value'] ?? '');
            } elseif (strpos($name, 'last') !== false && strpos($name, 'name') !== false) {
                $result['last_name'] = sanitize_text_field($field['value'] ?? '');
            }
        }

        // Fallback: look for generic 'name' field
        if (empty($result['first_name']) && empty($result['last_name'])) {
            foreach ($fields as $field) {
                $name = strtolower($field['name'] ?? '');
                if ($name === 'name' || strpos($name, 'your name') !== false || strpos($name, 'full name') !== false) {
                    $parts = explode(' ', trim($field['value'] ?? ''), 2);
                    $result['first_name'] = sanitize_text_field($parts[0]);
                    $result['last_name'] = isset($parts[1]) ? sanitize_text_field($parts[1]) : '';
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Find or create subscriber
     *
     * @param string $email
     * @param array $name_data
     * @return int|null
     */
    private static function findOrCreateSubscriber(string $email, array $name_data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            // Update name if currently empty
            $current = $wpdb->get_row($wpdb->prepare(
                "SELECT first_name, last_name FROM {$table} WHERE id = %d",
                $subscriber->id
            ));

            $updates = [];
            if (empty($current->first_name) && !empty($name_data['first_name'])) {
                $updates['first_name'] = $name_data['first_name'];
            }
            if (empty($current->last_name) && !empty($name_data['last_name'])) {
                $updates['last_name'] = $name_data['last_name'];
            }

            if (!empty($updates)) {
                $wpdb->update($table, $updates, ['id' => $subscriber->id]);
            }

            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'status'     => 'active',
            'source'     => 'wpforms',
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'source'     => 'wpforms',
        ]);

        return $subscriber_id;
    }

    /**
     * Get all WPForms for dropdown
     *
     * @return array
     */
    public static function getFormsForDropdown(): array
    {
        if (!self::isActive()) {
            return [];
        }

        $forms = get_posts([
            'post_type'      => 'wpforms',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ]);

        $result = [];

        foreach ($forms as $form) {
            $result[] = [
                'value' => $form->ID,
                'label' => $form->post_title,
            ];
        }

        return $result;
    }
}
