<?php

namespace WPOutreach\Integrations\Forms;

use WPOutreach\Automation\TriggerRegistry;

/**
 * Gravity Forms Integration
 *
 * Provides automation triggers for Gravity Forms form submissions.
 *
 * @since 1.3.0
 */
class GravityFormsIntegration
{
    /**
     * Initialize the integration
     */
    public static function init(): void
    {
        // Always register triggers so users can see what's available
        add_action('wp_outreach_register_triggers', [self::class, 'registerTriggers']);

        // Only hook into form submission if Gravity Forms is active
        if (self::isActive()) {
            add_action('gform_after_submission', [self::class, 'onFormSubmit'], 10, 2);
        }
    }

    /**
     * Check if Gravity Forms is active
     *
     * @return bool
     */
    public static function isActive(): bool
    {
        return class_exists('GFForms') || class_exists('GFAPI');
    }

    /**
     * Register Gravity Forms triggers
     *
     * @param TriggerRegistry $registry
     */
    public static function registerTriggers(TriggerRegistry $registry): void
    {
        $isActive = self::isActive();

        $registry->register([
            'id'          => 'gf_form_submitted',
            'name'        => __('Gravity Forms Submission', 'outreach'),
            'description' => __('Fires when a Gravity Forms form is submitted', 'outreach'),
            'group'       => __('Forms', 'outreach'),
            'icon'        => 'mail',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $isActive ? null : __('Gravity Forms plugin', 'outreach'),
            'config_fields' => $isActive ? [
                [
                    'key'         => 'form_id',
                    'label'       => __('Form', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any form', 'outreach'),
                    'options'     => self::getFormsForDropdown(),
                ],
            ] : [],
            'matcher' => function ($config, $context) {
                if (empty($config['form_id'])) {
                    return true; // Any form
                }
                return (int) $config['form_id'] === (int) ($context['form_id'] ?? 0);
            },
        ]);
    }

    /**
     * Handle form submission
     *
     * @param array $entry The entry that was just created.
     * @param array $form  The form object.
     */
    public static function onFormSubmit($entry, $form): void
    {
        $form_id = $form['id'] ?? 0;
        $form_title = $form['title'] ?? '';

        // Extract email from form entry
        $email = self::extractEmail($entry, $form);

        if (empty($email)) {
            return; // No email found, can't create subscriber
        }

        // Extract name fields
        $name_data = self::extractName($entry, $form);

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($email, $name_data);

        if (!$subscriber_id) {
            return;
        }

        // Build simplified fields array for conditions
        $simple_fields = [];
        foreach ($form['fields'] as $field) {
            $field_id = $field->id;
            $label = sanitize_key($field->label ?? 'field_' . $field_id);
            $simple_fields[$label] = rgar($entry, $field_id);
        }

        // Build context for conditions and merge tags
        $context = [
            'form_id'     => $form_id,
            'form_name'   => $form_title,
            'form_plugin' => 'gravity_forms',
            'fields'      => $simple_fields,
            'entry'       => $entry,
            'email'       => $email,
            'first_name'  => $name_data['first_name'] ?? '',
            'last_name'   => $name_data['last_name'] ?? '',
            'entry_id'    => $entry['id'] ?? 0,
            'source_url'  => $entry['source_url'] ?? '',
            'subscriber'  => [
                'email'      => $email,
                'first_name' => $name_data['first_name'] ?? '',
                'last_name'  => $name_data['last_name'] ?? '',
            ],
        ];

        // Fire the trigger
        TriggerRegistry::fire('gf_form_submitted', $subscriber_id, $context);
    }

    /**
     * Extract email from form entry
     *
     * @param array $entry
     * @param array $form
     * @return string|null
     */
    private static function extractEmail(array $entry, array $form): ?string
    {
        foreach ($form['fields'] as $field) {
            // Check for email field type
            if ($field->type === 'email') {
                $value = rgar($entry, $field->id);
                if (!empty($value) && is_email($value)) {
                    return sanitize_email($value);
                }
            }
        }

        // Fallback: search by field label
        foreach ($form['fields'] as $field) {
            $label = strtolower($field->label ?? '');
            if (strpos($label, 'email') !== false) {
                $value = rgar($entry, $field->id);
                if (!empty($value) && is_email($value)) {
                    return sanitize_email($value);
                }
            }
        }

        return null;
    }

    /**
     * Extract name from form entry
     *
     * @param array $entry
     * @param array $form
     * @return array
     */
    private static function extractName(array $entry, array $form): array
    {
        $result = ['first_name' => '', 'last_name' => ''];

        foreach ($form['fields'] as $field) {
            $type = $field->type ?? '';
            $label = strtolower($field->label ?? '');

            // Gravity Forms 'name' field type has subfields
            if ($type === 'name') {
                // Standard name field has inputs like 3.3 (First) and 3.6 (Last)
                if (!empty($field->inputs)) {
                    foreach ($field->inputs as $input) {
                        $input_label = strtolower($input['label'] ?? '');
                        $value = rgar($entry, $input['id']);

                        if (strpos($input_label, 'first') !== false && !empty($value)) {
                            $result['first_name'] = sanitize_text_field($value);
                        } elseif (strpos($input_label, 'last') !== false && !empty($value)) {
                            $result['last_name'] = sanitize_text_field($value);
                        }
                    }
                }
                if (!empty($result['first_name']) || !empty($result['last_name'])) {
                    break;
                }
            }

            // Check for separate first/last name text fields
            if ($type === 'text') {
                if (strpos($label, 'first') !== false && strpos($label, 'name') !== false) {
                    $result['first_name'] = sanitize_text_field(rgar($entry, $field->id));
                } elseif (strpos($label, 'last') !== false && strpos($label, 'name') !== false) {
                    $result['last_name'] = sanitize_text_field(rgar($entry, $field->id));
                }
            }
        }

        // Fallback: look for generic 'name' field
        if (empty($result['first_name']) && empty($result['last_name'])) {
            foreach ($form['fields'] as $field) {
                $label = strtolower($field->label ?? '');
                if ($label === 'name' || strpos($label, 'your name') !== false || strpos($label, 'full name') !== false) {
                    $value = rgar($entry, $field->id);
                    $parts = explode(' ', trim($value), 2);
                    $result['first_name'] = sanitize_text_field($parts[0]);
                    $result['last_name'] = isset($parts[1]) ? sanitize_text_field($parts[1]) : '';
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Find or create subscriber
     *
     * @param string $email
     * @param array $name_data
     * @return int|null
     */
    private static function findOrCreateSubscriber(string $email, array $name_data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            // Update name if currently empty
            $current = $wpdb->get_row($wpdb->prepare(
                "SELECT first_name, last_name FROM {$table} WHERE id = %d",
                $subscriber->id
            ));

            $updates = [];
            if (empty($current->first_name) && !empty($name_data['first_name'])) {
                $updates['first_name'] = $name_data['first_name'];
            }
            if (empty($current->last_name) && !empty($name_data['last_name'])) {
                $updates['last_name'] = $name_data['last_name'];
            }

            if (!empty($updates)) {
                $wpdb->update($table, $updates, ['id' => $subscriber->id]);
            }

            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'status'     => 'active',
            'source'     => 'gravity_forms',
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'source'     => 'gravity_forms',
        ]);

        return $subscriber_id;
    }

    /**
     * Get all Gravity Forms for dropdown
     *
     * @return array
     */
    public static function getFormsForDropdown(): array
    {
        if (!self::isActive() || !class_exists('GFAPI')) {
            return [];
        }

        $forms = \GFAPI::get_forms();
        $result = [];

        foreach ($forms as $form) {
            $result[] = [
                'value' => $form['id'],
                'label' => $form['title'],
            ];
        }

        return $result;
    }
}
