<?php

namespace WPOutreach\Integrations\Forms;

use WPOutreach\Automation\TriggerRegistry;

/**
 * Contact Form 7 Integration
 *
 * Provides automation triggers for Contact Form 7 form submissions.
 *
 * @since 1.3.0
 */
class ContactForm7Integration
{
    /**
     * Initialize the integration
     */
    public static function init(): void
    {
        // Always register triggers so users can see what's available
        add_action('wp_outreach_register_triggers', [self::class, 'registerTriggers']);

        // Only hook into form submission if Contact Form 7 is active
        if (self::isActive()) {
            add_action('wpcf7_mail_sent', [self::class, 'onFormSubmit'], 10, 1);
        }
    }

    /**
     * Check if Contact Form 7 is active
     *
     * @return bool
     */
    public static function isActive(): bool
    {
        return defined('WPCF7_VERSION') || class_exists('WPCF7');
    }

    /**
     * Register Contact Form 7 triggers
     *
     * @param TriggerRegistry $registry
     */
    public static function registerTriggers(TriggerRegistry $registry): void
    {
        $isActive = self::isActive();

        $registry->register([
            'id'          => 'cf7_form_submitted',
            'name'        => __('Contact Form 7 Submission', 'outreach'),
            'description' => __('Fires when a Contact Form 7 form is submitted', 'outreach'),
            'group'       => __('Forms', 'outreach'),
            'icon'        => 'mail',
            'source'      => 'integration',
            'available'   => $isActive,
            'requires'    => $isActive ? null : __('Contact Form 7 plugin', 'outreach'),
            'config_fields' => $isActive ? [
                [
                    'key'         => 'form_id',
                    'label'       => __('Form', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any form', 'outreach'),
                    'options'     => self::getFormsForDropdown(),
                ],
            ] : [],
            'matcher' => function ($config, $context) {
                if (empty($config['form_id'])) {
                    return true; // Any form
                }
                return (int) $config['form_id'] === (int) ($context['form_id'] ?? 0);
            },
        ]);
    }

    /**
     * Handle form submission
     *
     * @param \WPCF7_ContactForm $contact_form
     */
    public static function onFormSubmit($contact_form): void
    {
        $submission = \WPCF7_Submission::get_instance();

        if (!$submission) {
            return;
        }

        $form_data = $submission->get_posted_data();
        $form_id = $contact_form->id();
        $form_title = $contact_form->title();

        // Extract email from form data
        $email = self::extractEmail($form_data);

        if (empty($email)) {
            return; // No email found, can't create subscriber
        }

        // Extract name fields
        $name_data = self::extractName($form_data);

        // Find or create subscriber
        $subscriber_id = self::findOrCreateSubscriber($email, $name_data);

        if (!$subscriber_id) {
            return;
        }

        // Build context for conditions and merge tags
        $context = [
            'form_id'     => $form_id,
            'form_name'   => $form_title,
            'form_plugin' => 'contact_form_7',
            'fields'      => $form_data,
            'email'       => $email,
            'first_name'  => $name_data['first_name'] ?? '',
            'last_name'   => $name_data['last_name'] ?? '',
            'page_url'    => $submission->get_meta('url') ?? '',
            'page_title'  => $submission->get_meta('page_title') ?? '',
            'subscriber'  => [
                'email'      => $email,
                'first_name' => $name_data['first_name'] ?? '',
                'last_name'  => $name_data['last_name'] ?? '',
            ],
        ];

        // Fire the trigger
        TriggerRegistry::fire('cf7_form_submitted', $subscriber_id, $context);
    }

    /**
     * Extract email from form data
     *
     * @param array $form_data
     * @return string|null
     */
    private static function extractEmail(array $form_data): ?string
    {
        // Common email field names
        $email_fields = ['your-email', 'email', 'your_email', 'Email', 'EMAIL', 'e-mail', 'user-email'];

        foreach ($email_fields as $field) {
            if (!empty($form_data[$field]) && is_email($form_data[$field])) {
                return sanitize_email($form_data[$field]);
            }
        }

        // Search for any field containing 'email'
        foreach ($form_data as $key => $value) {
            if (stripos($key, 'email') !== false && is_email($value)) {
                return sanitize_email($value);
            }
        }

        return null;
    }

    /**
     * Extract name from form data
     *
     * @param array $form_data
     * @return array
     */
    private static function extractName(array $form_data): array
    {
        $result = ['first_name' => '', 'last_name' => ''];

        // Check for separate first/last name fields
        $first_name_fields = ['first-name', 'first_name', 'firstname', 'FirstName', 'fname'];
        $last_name_fields = ['last-name', 'last_name', 'lastname', 'LastName', 'lname'];

        foreach ($first_name_fields as $field) {
            if (!empty($form_data[$field])) {
                $result['first_name'] = sanitize_text_field($form_data[$field]);
                break;
            }
        }

        foreach ($last_name_fields as $field) {
            if (!empty($form_data[$field])) {
                $result['last_name'] = sanitize_text_field($form_data[$field]);
                break;
            }
        }

        // If no separate fields, try combined name field
        if (empty($result['first_name']) && empty($result['last_name'])) {
            $name_fields = ['your-name', 'name', 'your_name', 'Name', 'full-name', 'fullname'];

            foreach ($name_fields as $field) {
                if (!empty($form_data[$field])) {
                    $parts = explode(' ', trim($form_data[$field]), 2);
                    $result['first_name'] = sanitize_text_field($parts[0]);
                    $result['last_name'] = isset($parts[1]) ? sanitize_text_field($parts[1]) : '';
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Find or create subscriber
     *
     * @param string $email
     * @param array $name_data
     * @return int|null
     */
    private static function findOrCreateSubscriber(string $email, array $name_data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Check if subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            // Update name if currently empty
            $current = $wpdb->get_row($wpdb->prepare(
                "SELECT first_name, last_name FROM {$table} WHERE id = %d",
                $subscriber->id
            ));

            $updates = [];
            if (empty($current->first_name) && !empty($name_data['first_name'])) {
                $updates['first_name'] = $name_data['first_name'];
            }
            if (empty($current->last_name) && !empty($name_data['last_name'])) {
                $updates['last_name'] = $name_data['last_name'];
            }

            if (!empty($updates)) {
                $wpdb->update($table, $updates, ['id' => $subscriber->id]);
            }

            return (int) $subscriber->id;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'status'     => 'active',
            'source'     => 'cf7_form',
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $email,
            'first_name' => $name_data['first_name'] ?? '',
            'last_name'  => $name_data['last_name'] ?? '',
            'source'     => 'cf7_form',
        ]);

        return $subscriber_id;
    }

    /**
     * Get all CF7 forms for dropdown
     *
     * @return array
     */
    public static function getFormsForDropdown(): array
    {
        if (!self::isActive()) {
            return [];
        }

        $forms = get_posts([
            'post_type'      => 'wpcf7_contact_form',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ]);

        $result = [];

        foreach ($forms as $form) {
            $result[] = [
                'value' => $form->ID,
                'label' => $form->post_title,
            ];
        }

        return $result;
    }
}
