<?php

namespace WPOutreach;

/**
 * Database installer
 */
class Installer {

    /**
     * Install database tables
     */
    public function install(): void {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        // Create all tables
        $this->create_subscribers_table($charset_collate);
        $this->create_lists_table($charset_collate);
        $this->create_subscriber_list_table($charset_collate);
        $this->create_tags_table($charset_collate);
        $this->create_subscriber_tag_table($charset_collate);
        $this->create_templates_table($charset_collate);
        $this->create_campaigns_table($charset_collate);
        $this->create_automations_table($charset_collate);
        $this->create_automation_queue_table($charset_collate);
        $this->create_queue_table($charset_collate);
        $this->create_logs_table($charset_collate);
        $this->create_links_table($charset_collate);
        $this->create_notifications_table($charset_collate);
        $this->create_post_subscriptions_table($charset_collate);

        // Update database version
        update_option('wp_outreach_db_version', WP_OUTREACH_VERSION);
    }

    /**
     * Create subscribers table
     */
    private function create_subscribers_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT UNSIGNED NULL,
            email VARCHAR(255) NOT NULL,
            first_name VARCHAR(100) NULL,
            last_name VARCHAR(100) NULL,
            status ENUM('pending','active','unsubscribed','bounced') DEFAULT 'pending',
            source VARCHAR(50) DEFAULT 'form',
            ip_address VARCHAR(45) NULL,
            token VARCHAR(64) NULL,
            custom_fields JSON NULL,
            confirmed_at DATETIME NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY email (email),
            KEY status (status),
            KEY user_id (user_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create lists table
     */
    private function create_lists_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            slug VARCHAR(255) NOT NULL,
            description TEXT NULL,
            is_public TINYINT(1) DEFAULT 1,
            double_optin TINYINT(1) DEFAULT 1,
            subscriber_count INT DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create subscriber-list pivot table
     */
    private function create_subscriber_list_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscriber_list';

        $sql = "CREATE TABLE {$table} (
            subscriber_id BIGINT UNSIGNED NOT NULL,
            list_id BIGINT UNSIGNED NOT NULL,
            subscribed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (subscriber_id, list_id),
            KEY list_id (list_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create tags table
     */
    private function create_tags_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_tags';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(100) NOT NULL,
            slug VARCHAR(100) NOT NULL,
            color VARCHAR(7) DEFAULT '#6366f1',
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create subscriber-tag pivot table
     */
    private function create_subscriber_tag_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscriber_tag';

        $sql = "CREATE TABLE {$table} (
            subscriber_id BIGINT UNSIGNED NOT NULL,
            tag_id BIGINT UNSIGNED NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (subscriber_id, tag_id),
            KEY tag_id (tag_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create templates table
     */
    private function create_templates_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            slug VARCHAR(255) NOT NULL,
            subject VARCHAR(500) NULL,
            content LONGTEXT NULL,
            content_json JSON NULL,
            thumbnail VARCHAR(500) NULL,
            is_system TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create campaigns table
     */
    private function create_campaigns_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            type ENUM('one_time','scheduled','recurring') DEFAULT 'one_time',
            status ENUM('draft','scheduled','sending','sent','paused') DEFAULT 'draft',
            from_name VARCHAR(255) NULL,
            from_email VARCHAR(255) NULL,
            subject VARCHAR(500) NOT NULL,
            preheader VARCHAR(255) NULL,
            content LONGTEXT NULL,
            content_json JSON NULL,
            list_ids JSON NULL,
            tag_ids JSON NULL,
            scheduled_at DATETIME NULL,
            recurring_config JSON NULL,
            last_sent_at DATETIME NULL,
            next_send_at DATETIME NULL,
            sent_at DATETIME NULL,
            stats JSON NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY scheduled_at (scheduled_at),
            KEY next_send_at (next_send_at)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create automations table
     */
    private function create_automations_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            status ENUM('draft','active','paused') DEFAULT 'draft',
            trigger_type VARCHAR(100) NOT NULL,
            trigger_config JSON NULL,
            steps JSON NOT NULL,
            stats JSON NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create automation queue table
     */
    private function create_automation_queue_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            automation_id BIGINT UNSIGNED NOT NULL,
            subscriber_id BIGINT UNSIGNED NOT NULL,
            current_step VARCHAR(100) DEFAULT '0',
            status ENUM('active','completed','cancelled','failed') DEFAULT 'active',
            attempts TINYINT UNSIGNED DEFAULT 0,
            last_error TEXT NULL,
            next_run_at DATETIME NULL,
            started_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            failed_at DATETIME NULL,
            context JSON NULL,
            PRIMARY KEY (id),
            KEY status (status),
            KEY next_run_at (next_run_at),
            KEY automation_id (automation_id),
            KEY subscriber_id (subscriber_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create email queue table
     */
    private function create_queue_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_queue';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            campaign_id BIGINT UNSIGNED NULL,
            automation_id BIGINT UNSIGNED NULL,
            subscriber_id BIGINT UNSIGNED NOT NULL,
            to_email VARCHAR(255) NOT NULL,
            subject VARCHAR(500) NOT NULL,
            content LONGTEXT NOT NULL,
            tracking_id VARCHAR(64) NULL,
            priority TINYINT DEFAULT 5,
            status ENUM('pending','processing','sent','failed') DEFAULT 'pending',
            attempts TINYINT DEFAULT 0,
            scheduled_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            sent_at DATETIME NULL,
            error TEXT NULL,
            PRIMARY KEY (id),
            KEY status (status),
            KEY scheduled_at (scheduled_at),
            KEY campaign_id (campaign_id),
            KEY tracking_id (tracking_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create email logs table
     */
    private function create_logs_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            type VARCHAR(20) DEFAULT 'campaign',
            campaign_id BIGINT UNSIGNED NULL,
            automation_id BIGINT UNSIGNED NULL,
            subscriber_id BIGINT UNSIGNED NULL,
            email VARCHAR(255) NOT NULL,
            subject VARCHAR(500) NULL,
            tracking_id VARCHAR(64) NULL,
            status ENUM('sent','opened','clicked','bounced','complained') DEFAULT 'sent',
            opens INT DEFAULT 0,
            clicks INT DEFAULT 0,
            clicked_links JSON NULL,
            first_opened_at DATETIME NULL,
            last_opened_at DATETIME NULL,
            error TEXT NULL,
            sent_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY type (type),
            KEY campaign_id (campaign_id),
            KEY subscriber_id (subscriber_id),
            KEY status (status),
            KEY tracking_id (tracking_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create links table for click tracking
     */
    private function create_links_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_links';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            campaign_id BIGINT UNSIGNED NULL,
            url TEXT NOT NULL,
            hash VARCHAR(32) NOT NULL,
            clicks INT DEFAULT 0,
            PRIMARY KEY (id),
            KEY hash (hash),
            KEY campaign_id (campaign_id)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create notification rules table
     */
    private function create_notifications_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_notifications';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            type ENUM('wpdm','woocommerce','post') DEFAULT 'post',
            status ENUM('active','inactive') DEFAULT 'active',
            post_types JSON NULL,
            list_ids JSON NULL,
            template_id BIGINT UNSIGNED NULL,
            subject VARCHAR(500) NULL,
            delay_minutes INT DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY type (type)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Create post subscriptions table
     * Allows subscribers to subscribe to post/page/CPT updates
     */
    private function create_post_subscriptions_table(string $charset_collate): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            subscriber_id BIGINT UNSIGNED NOT NULL,
            post_id BIGINT UNSIGNED DEFAULT 0,
            post_type VARCHAR(50) NOT NULL DEFAULT 'post',
            subscription_type ENUM('new','update','both') DEFAULT 'both',
            status TINYINT(1) DEFAULT 1,
            token VARCHAR(64) NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY subscriber_id (subscriber_id),
            KEY post_id (post_id),
            KEY post_type (post_type),
            KEY status (status),
            UNIQUE KEY unique_subscription (subscriber_id, post_id, post_type)
        ) {$charset_collate};";

        dbDelta($sql);
    }

    /**
     * Upgrade database schema for existing installations.
     * Adds new columns if they don't exist.
     */
    public function upgrade(): void {
        global $wpdb;

        // First ensure all base tables exist (dbDelta is idempotent)
        $this->install();

        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';

        // Add recurring_config column if it doesn't exist
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$campaigns_table} LIKE 'recurring_config'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$campaigns_table} ADD COLUMN recurring_config JSON NULL AFTER scheduled_at");
        }

        // Add last_sent_at column if it doesn't exist
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$campaigns_table} LIKE 'last_sent_at'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$campaigns_table} ADD COLUMN last_sent_at DATETIME NULL AFTER recurring_config");
        }

        // Add next_send_at column if it doesn't exist
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$campaigns_table} LIKE 'next_send_at'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$campaigns_table} ADD COLUMN next_send_at DATETIME NULL AFTER last_sent_at");
            $wpdb->query("ALTER TABLE {$campaigns_table} ADD KEY next_send_at (next_send_at)");
        }

        // Upgrade automation_queue table for error handling
        $this->upgrade_automation_queue_table();

        // Upgrade tracking tables
        $this->upgrade_tracking_tables();

        // Upgrade logs table with type column
        $this->upgrade_logs_table();

        // Create post_subscriptions table if it doesn't exist
        $this->upgrade_post_subscriptions_table();

        // Update database version
        update_option('wp_outreach_db_version', WP_OUTREACH_VERSION);
    }

    /**
     * Create post_subscriptions table if it doesn't exist (for upgrades)
     */
    private function upgrade_post_subscriptions_table(): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';

        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table}'");
        if (!$table_exists) {
            $charset_collate = $wpdb->get_charset_collate();
            $this->create_post_subscriptions_table($charset_collate);
        }
    }

    /**
     * Upgrade queue and logs tables for tracking
     */
    private function upgrade_tracking_tables(): void {
        global $wpdb;

        // Add tracking_id to queue table
        $queue_table = $wpdb->prefix . 'outreach_queue';
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$queue_table} LIKE 'tracking_id'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$queue_table} ADD COLUMN tracking_id VARCHAR(64) NULL AFTER content");
            $wpdb->query("ALTER TABLE {$queue_table} ADD INDEX tracking_id (tracking_id)");
        }

        // Add tracking_id and timestamp columns to logs table
        $logs_table = $wpdb->prefix . 'outreach_logs';

        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$logs_table} LIKE 'tracking_id'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$logs_table} ADD COLUMN tracking_id VARCHAR(64) NULL AFTER subject");
            $wpdb->query("ALTER TABLE {$logs_table} ADD INDEX tracking_id (tracking_id)");
        }

        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$logs_table} LIKE 'first_opened_at'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$logs_table} ADD COLUMN first_opened_at DATETIME NULL AFTER clicked_links");
        }

        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$logs_table} LIKE 'last_opened_at'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$logs_table} ADD COLUMN last_opened_at DATETIME NULL AFTER first_opened_at");
        }
    }

    /**
     * Upgrade logs table with type column for categorizing emails
     */
    private function upgrade_logs_table(): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_logs';

        // Add type column to distinguish email sources
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'type'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN type VARCHAR(20) DEFAULT 'campaign' AFTER id");
            $wpdb->query("ALTER TABLE {$table} ADD INDEX type (type)");

            // Update existing records based on their source
            $wpdb->query("UPDATE {$table} SET type = 'automation' WHERE automation_id IS NOT NULL AND campaign_id IS NULL");
            $wpdb->query("UPDATE {$table} SET type = 'campaign' WHERE campaign_id IS NOT NULL");
        }

        // Add error column to store bounce/failure reasons
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'error'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN error TEXT NULL AFTER last_opened_at");
        }

        // Make subscriber_id nullable for WordPress emails to non-subscribers
        $wpdb->query("ALTER TABLE {$table} MODIFY COLUMN subscriber_id BIGINT UNSIGNED NULL");

        // Add not_opened_triggered column for email_not_opened trigger tracking
        // This prevents the same email from triggering the automation multiple times
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'not_opened_triggered'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN not_opened_triggered DATETIME NULL AFTER error");
        }
    }

    /**
     * Upgrade automation_queue table with error handling columns
     */
    private function upgrade_automation_queue_table(): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        // Add attempts column
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'attempts'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN attempts TINYINT UNSIGNED DEFAULT 0 AFTER status");
        }

        // Add last_error column
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'last_error'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN last_error TEXT NULL AFTER attempts");
        }

        // Add failed_at column
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'failed_at'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN failed_at DATETIME NULL AFTER started_at");
        }

        // Add context column if missing
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table} LIKE 'context'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE {$table} ADD COLUMN context JSON NULL AFTER failed_at");
        }

        // Update status ENUM to include 'failed'
        $wpdb->query("ALTER TABLE {$table} MODIFY COLUMN status ENUM('active','completed','cancelled','failed') DEFAULT 'active'");
    }
}
