<?php

namespace WPOutreach\Helpers;

/**
 * Date Helper
 *
 * Handles timezone-aware date formatting for API responses.
 * Ensures dates stored in WordPress timezone are correctly formatted
 * for JavaScript clients in any timezone.
 *
 * @since 1.0.0
 */
class DateHelper
{
    /**
     * Convert a MySQL datetime string from WordPress timezone to ISO 8601 format.
     *
     * WordPress stores dates using current_time('mysql') which is in the site's timezone.
     * This method converts those dates to ISO 8601 format with timezone offset,
     * allowing JavaScript clients to correctly parse and display the date in their local timezone.
     *
     * @param string|null $mysql_date MySQL datetime string (Y-m-d H:i:s) in WordPress timezone
     * @return string|null ISO 8601 formatted date with timezone offset, or null if input is null/empty
     *
     * @example
     * // WordPress timezone: UTC+0
     * // Input: "2024-12-29 10:00:00" (stored as WP timezone)
     * // Output: "2024-12-29T10:00:00+00:00"
     *
     * // JavaScript can then correctly parse this:
     * // new Date("2024-12-29T10:00:00+00:00") -> shows 4:00 PM in UTC+6
     */
    public static function toIso8601(?string $mysql_date): ?string
    {
        if (empty($mysql_date) || $mysql_date === '0000-00-00 00:00:00') {
            return null;
        }

        try {
            // Get WordPress timezone
            $wp_timezone = wp_timezone();

            // Parse the date in WordPress timezone
            $date = new \DateTime($mysql_date, $wp_timezone);

            // Return ISO 8601 format with timezone offset
            return $date->format('c'); // 'c' = ISO 8601 format (e.g., 2024-12-29T10:00:00+00:00)
        } catch (\Exception $e) {
            // If parsing fails, return the original string
            return $mysql_date;
        }
    }

    /**
     * Convert multiple date fields in an array to ISO 8601 format.
     *
     * @param array $data The data array with date fields
     * @param array $date_fields List of field names to convert
     * @return array The data array with converted dates
     */
    public static function convertDates(array $data, array $date_fields): array
    {
        foreach ($date_fields as $field) {
            if (isset($data[$field]) && !empty($data[$field])) {
                $data[$field] = self::toIso8601($data[$field]);
            }
        }
        return $data;
    }

    /**
     * Get the WordPress timezone offset string.
     *
     * @return string Timezone offset (e.g., "+05:30", "-08:00", "+00:00")
     */
    public static function getWpTimezoneOffset(): string
    {
        $wp_timezone = wp_timezone();
        $now = new \DateTime('now', $wp_timezone);
        return $now->format('P'); // e.g., "+00:00", "+05:30"
    }

    /**
     * Get the WordPress timezone name.
     *
     * @return string Timezone identifier (e.g., "UTC", "America/New_York", "Asia/Dhaka")
     */
    public static function getWpTimezoneName(): string
    {
        return wp_timezone_string();
    }
}
