<?php

namespace WPOutreach\Forms;

/**
 * Public subscription form handler
 *
 * Provides shortcode and REST API endpoint for public email subscriptions.
 * Includes honeypot spam protection and rate limiting.
 *
 * @since 1.0.0
 */
class SubscriptionForm
{
    /**
     * Rate limit: max subscriptions per IP per hour
     */
    private const RATE_LIMIT = 5;

    /**
     * Rate limit window in seconds (1 hour)
     */
    private const RATE_WINDOW = 3600;

    /**
     * Register hooks
     */
    public static function init(): void
    {
        add_shortcode('wp_outreach_form', [self::class, 'renderShortcode']);
        add_action('wp_enqueue_scripts', [self::class, 'enqueueAssets']);
    }

    /**
     * Enqueue form assets when shortcode is present
     */
    public static function enqueueAssets(): void
    {
        global $post;

        if ($post && has_shortcode($post->post_content, 'wp_outreach_form')) {
            wp_enqueue_style(
                'outreach-form',
                WP_OUTREACH_URL . 'public/css/form.css',
                [],
                WP_OUTREACH_VERSION
            );

            wp_enqueue_script(
                'outreach-form',
                WP_OUTREACH_URL . 'public/js/form.js',
                [],
                WP_OUTREACH_VERSION,
                true
            );

            wp_localize_script('outreach-form', 'wpOutreachForm', [
                'apiUrl' => rest_url('outreach/v1/subscribe'),
                'nonce' => wp_create_nonce('wp_rest'),
            ]);
        }
    }

    /**
     * Render subscription form shortcode
     *
     * Shortcode: [wp_outreach_form]
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public static function renderShortcode(array $atts = []): string
    {
        $atts = shortcode_atts([
            'list' => '',                           // List ID or slug
            'button_text' => 'Subscribe',           // Submit button text
            'show_name' => 'true',                  // Show name fields
            'show_first_name' => 'true',            // Show first name field
            'show_last_name' => 'false',            // Show last name field
            'first_name_required' => 'false',       // Require first name
            'last_name_required' => 'false',        // Require last name
            'success_message' => 'Thanks for subscribing!',
            'placeholder_email' => 'Your email address',
            'placeholder_first_name' => 'First name',
            'placeholder_last_name' => 'Last name',
            'class' => '',                          // Additional CSS class
        ], $atts);

        // Convert string booleans
        $showName = filter_var($atts['show_name'], FILTER_VALIDATE_BOOLEAN);
        $showFirstName = filter_var($atts['show_first_name'], FILTER_VALIDATE_BOOLEAN);
        $showLastName = filter_var($atts['show_last_name'], FILTER_VALIDATE_BOOLEAN);
        $firstNameRequired = filter_var($atts['first_name_required'], FILTER_VALIDATE_BOOLEAN);
        $lastNameRequired = filter_var($atts['last_name_required'], FILTER_VALIDATE_BOOLEAN);

        // Resolve list ID from slug if needed
        $listId = self::resolveListId($atts['list']);

        // Generate unique form ID
        $formId = 'wpo-form-' . wp_rand(1000, 9999);

        ob_start();
        ?>
        <div id="<?php echo esc_attr($formId); ?>" class="wpo-subscribe-form <?php echo esc_attr($atts['class']); ?>">
            <form class="wpo-form" data-list="<?php echo esc_attr($listId); ?>" data-success-message="<?php echo esc_attr($atts['success_message']); ?>">
                <?php if ($showName && $showFirstName): ?>
                <div class="wpo-form-field">
                    <input
                        type="text"
                        name="first_name"
                        placeholder="<?php echo esc_attr($atts['placeholder_first_name']); ?>"
                        class="wpo-form-input"
                        <?php echo $firstNameRequired ? 'required' : ''; ?>
                    />
                </div>
                <?php endif; ?>

                <?php if ($showName && $showLastName): ?>
                <div class="wpo-form-field">
                    <input
                        type="text"
                        name="last_name"
                        placeholder="<?php echo esc_attr($atts['placeholder_last_name']); ?>"
                        class="wpo-form-input"
                        <?php echo $lastNameRequired ? 'required' : ''; ?>
                    />
                </div>
                <?php endif; ?>

                <div class="wpo-form-field">
                    <input
                        type="email"
                        name="email"
                        placeholder="<?php echo esc_attr($atts['placeholder_email']); ?>"
                        class="wpo-form-input"
                        required
                    />
                </div>

                <!-- Honeypot field (hidden) -->
                <div class="wpo-hp-field" aria-hidden="true">
                    <input type="text" name="website" tabindex="-1" autocomplete="off" />
                </div>

                <div class="wpo-form-field">
                    <button type="submit" class="wpo-form-button">
                        <span class="wpo-btn-text"><?php echo esc_html($atts['button_text']); ?></span>
                        <span class="wpo-btn-loading" style="display:none;">
                            <svg class="wpo-spinner" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10" stroke-dasharray="32" stroke-linecap="round"/>
                            </svg>
                        </span>
                    </button>
                </div>

                <div class="wpo-form-message" style="display:none;"></div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Resolve list ID from ID or slug
     */
    private static function resolveListId(string $list): int
    {
        if (empty($list)) {
            return 0;
        }

        // If numeric, use as ID
        if (is_numeric($list)) {
            return (int) $list;
        }

        // Otherwise, look up by slug
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';

        $id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE slug = %s",
            sanitize_title($list)
        ));

        return (int) $id;
    }

    /**
     * Process subscription request (called from REST API)
     *
     * @param array $data Request data
     * @return array{success: bool, message: string, requires_confirmation?: bool}
     */
    public static function processSubscription(array $data): array
    {
        global $wpdb;

        // Check honeypot
        if (!empty($data['website'])) {
            // Silently fail for bots
            return [
                'success' => true,
                'message' => 'Thanks for subscribing!',
            ];
        }

        // Validate email
        $email = sanitize_email($data['email'] ?? '');
        if (empty($email) || !is_email($email)) {
            return [
                'success' => false,
                'message' => 'Please enter a valid email address.',
            ];
        }

        // Rate limiting
        $ip = self::getClientIp();
        if (!self::checkRateLimit($ip)) {
            return [
                'success' => false,
                'message' => 'Too many subscription attempts. Please try again later.',
            ];
        }

        // Get subscriber data
        $firstName = sanitize_text_field($data['first_name'] ?? '');
        $lastName = sanitize_text_field($data['last_name'] ?? '');
        $listId = (int) ($data['list_id'] ?? 0);

        $subscribersTable = $wpdb->prefix . 'outreach_subscribers';
        $listTable = $wpdb->prefix . 'outreach_lists';
        $pivotTable = $wpdb->prefix . 'outreach_subscriber_list';

        // Check if list requires double opt-in
        $requiresConfirmation = false;
        if ($listId > 0) {
            $list = $wpdb->get_row($wpdb->prepare(
                "SELECT double_optin FROM {$listTable} WHERE id = %d",
                $listId
            ));
            $requiresConfirmation = $list && $list->double_optin;
        }

        // Check if subscriber already exists
        $existingSubscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, status FROM {$subscribersTable} WHERE email = %s",
            $email
        ));

        if ($existingSubscriber) {
            // Already subscribed and active
            if ($existingSubscriber->status === 'active') {
                // Add to list if not already
                if ($listId > 0) {
                    $wpdb->replace($pivotTable, [
                        'subscriber_id' => $existingSubscriber->id,
                        'list_id' => $listId,
                        'subscribed_at' => current_time('mysql'),
                    ]);
                }

                return [
                    'success' => true,
                    'message' => 'You are already subscribed!',
                ];
            }

            // Previously unsubscribed - reactivate
            if ($existingSubscriber->status === 'unsubscribed') {
                $status = $requiresConfirmation ? 'pending' : 'active';

                $wpdb->update(
                    $subscribersTable,
                    [
                        'first_name' => $firstName ?: $wpdb->get_var($wpdb->prepare("SELECT first_name FROM {$subscribersTable} WHERE id = %d", $existingSubscriber->id)),
                        'last_name' => $lastName ?: $wpdb->get_var($wpdb->prepare("SELECT last_name FROM {$subscribersTable} WHERE id = %d", $existingSubscriber->id)),
                        'status' => $status,
                        'updated_at' => current_time('mysql'),
                    ],
                    ['id' => $existingSubscriber->id]
                );

                // Add to list
                if ($listId > 0) {
                    $wpdb->replace($pivotTable, [
                        'subscriber_id' => $existingSubscriber->id,
                        'list_id' => $listId,
                        'subscribed_at' => current_time('mysql'),
                    ]);
                }

                // Fire hook
                do_action('wp_outreach_subscriber_resubscribed', $existingSubscriber->id, $listId);

                if ($requiresConfirmation) {
                    // Send confirmation email
                    DoubleOptIn::sendConfirmationEmail($existingSubscriber->id);
                    return [
                        'success' => true,
                        'message' => 'Please check your email to confirm your subscription.',
                        'requires_confirmation' => true,
                    ];
                }

                return [
                    'success' => true,
                    'message' => 'Welcome back! You have been re-subscribed.',
                ];
            }

            // Pending status - resend confirmation
            if ($existingSubscriber->status === 'pending') {
                // Resend confirmation email
                DoubleOptIn::sendConfirmationEmail($existingSubscriber->id);
                return [
                    'success' => true,
                    'message' => 'Please check your email to confirm your subscription.',
                    'requires_confirmation' => true,
                ];
            }
        }

        // Create new subscriber
        $status = $requiresConfirmation ? 'pending' : 'active';
        $token = wp_generate_password(32, false);

        $result = $wpdb->insert($subscribersTable, [
            'email' => $email,
            'first_name' => $firstName,
            'last_name' => $lastName,
            'status' => $status,
            'source' => 'form',
            'ip_address' => $ip,
            'token' => $token,
            'created_at' => current_time('mysql'),
        ]);

        if (!$result) {
            return [
                'success' => false,
                'message' => 'An error occurred. Please try again.',
            ];
        }

        $subscriberId = $wpdb->insert_id;

        // Add to list
        if ($listId > 0) {
            $wpdb->insert($pivotTable, [
                'subscriber_id' => $subscriberId,
                'list_id' => $listId,
                'subscribed_at' => current_time('mysql'),
            ]);
        }

        // Record rate limit
        self::recordSubscription($ip);

        // Fire hook
        do_action('wp_outreach_subscriber_created', $subscriberId, [
            'email' => $email,
            'first_name' => $firstName,
            'last_name' => $lastName,
            'list_id' => $listId,
            'source' => 'form',
        ]);

        if ($requiresConfirmation) {
            // Send confirmation email
            DoubleOptIn::sendConfirmationEmail($subscriberId);
            return [
                'success' => true,
                'message' => 'Please check your email to confirm your subscription.',
                'requires_confirmation' => true,
            ];
        }

        return [
            'success' => true,
            'message' => 'Thanks for subscribing!',
        ];
    }

    /**
     * Get client IP address
     */
    private static function getClientIp(): string
    {
        $ip = '';

        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }

        return sanitize_text_field(trim($ip));
    }

    /**
     * Check rate limit for IP
     */
    private static function checkRateLimit(string $ip): bool
    {
        $transientKey = 'wpo_subscribe_' . md5($ip);
        $count = (int) get_transient($transientKey);

        return $count < self::RATE_LIMIT;
    }

    /**
     * Record subscription for rate limiting
     */
    private static function recordSubscription(string $ip): void
    {
        $transientKey = 'wpo_subscribe_' . md5($ip);
        $count = (int) get_transient($transientKey);

        set_transient($transientKey, $count + 1, self::RATE_WINDOW);
    }
}
