<?php

namespace WPOutreach\Forms;

use WPOutreach\Subscriptions\PostSubscriptionHandler;

/**
 * Post Subscription Form Shortcode
 *
 * Provides shortcode for subscribing to post/page/CPT updates.
 * Users can subscribe to specific post updates or all posts of a type.
 *
 * @since 1.0.0
 */
class PostSubscriptionForm
{
    /**
     * Rate limit: max subscriptions per IP per hour
     */
    private const RATE_LIMIT = 5;

    /**
     * Rate limit window in seconds (1 hour)
     */
    private const RATE_WINDOW = 3600;

    /**
     * Register hooks
     */
    public static function init(): void
    {
        add_shortcode('wp_outreach_post_subscribe', [self::class, 'renderShortcode']);
        add_action('wp_enqueue_scripts', [self::class, 'enqueueAssets']);
    }

    /**
     * Enqueue form assets when shortcode is present
     */
    public static function enqueueAssets(): void
    {
        global $post;

        $hasShortcode = $post && has_shortcode($post->post_content, 'wp_outreach_post_subscribe');

        if ($hasShortcode) {
            wp_enqueue_style(
                'outreach-form',
                WP_OUTREACH_URL . 'public/css/form.css',
                [],
                WP_OUTREACH_VERSION
            );

            wp_enqueue_script(
                'outreach-post-subscribe',
                WP_OUTREACH_URL . 'public/js/post-subscribe.js',
                [],
                WP_OUTREACH_VERSION,
                true
            );

            wp_localize_script('outreach-post-subscribe', 'wpOutreachPostSubscribe', [
                'apiUrl' => rest_url('outreach/v1/post-subscribe'),
                'nonce' => wp_create_nonce('wp_rest'),
            ]);
        }
    }

    /**
     * Render post subscription form shortcode
     *
     * Shortcode: [wp_outreach_post_subscribe]
     *
     * @param array $atts {
     *     Shortcode attributes.
     *
     *     @type int    $post_id          Post ID to subscribe to (0 = current post, -1 = all posts of type)
     *     @type string $post_type        Post type (default: current post type or 'post')
     *     @type string $subscription_type Subscription type: 'new', 'update', 'both' (default: 'both')
     *     @type string $button_text      Submit button text
     *     @type string $success_message  Message shown after subscription
     *     @type string $placeholder_email Email placeholder text
     *     @type string $show_type_choice Show radio buttons for subscription type
     *     @type string $show_scope_choice Show radio buttons for post vs post type
     *     @type string $class            Additional CSS class
     * }
     * @return string HTML output
     */
    public static function renderShortcode(array $atts = []): string
    {
        global $post;

        $atts = shortcode_atts([
            'post_id' => '0',                    // 0 = current post, -1 = all posts of type
            'post_type' => '',                   // Empty = detect from current post
            'subscription_type' => 'both',       // 'new', 'update', or 'both'
            'button_text' => 'Notify Me',
            'success_message' => 'You will be notified when this content is updated!',
            'placeholder_email' => 'Your email address',
            'show_type_choice' => 'false',       // Show new/update/both radio buttons
            'show_scope_choice' => 'false',      // Show this post / all posts choice
            'class' => '',
        ], $atts);

        // Determine post ID
        $postId = (int) $atts['post_id'];
        if ($postId === 0 && $post) {
            $postId = $post->ID;
        }

        // Determine post type
        $postType = $atts['post_type'];
        if (empty($postType)) {
            if ($postId > 0) {
                $postType = get_post_type($postId);
            } elseif ($post) {
                $postType = $post->post_type;
            } else {
                $postType = 'post';
            }
        }

        // Get post type label for UI
        $postTypeObj = get_post_type_object($postType);
        $postTypeLabel = $postTypeObj ? $postTypeObj->labels->singular_name : $postType;
        $postTypeLabelPlural = $postTypeObj ? $postTypeObj->labels->name : $postType . 's';

        // Convert string booleans
        $showTypeChoice = filter_var($atts['show_type_choice'], FILTER_VALIDATE_BOOLEAN);
        $showScopeChoice = filter_var($atts['show_scope_choice'], FILTER_VALIDATE_BOOLEAN);

        // Get post title for display
        $postTitle = $postId > 0 ? get_the_title($postId) : '';

        // Generate unique form ID
        $formId = 'wpo-post-subscribe-' . wp_rand(1000, 9999);

        ob_start();
        ?>
        <div id="<?php echo esc_attr($formId); ?>" class="wpo-subscribe-form wpo-post-subscribe-form <?php echo esc_attr($atts['class']); ?>">
            <form class="wpo-form" data-success-message="<?php echo esc_attr($atts['success_message']); ?>">
                <input type="hidden" name="post_id" value="<?php echo esc_attr($postId); ?>" />
                <input type="hidden" name="post_type" value="<?php echo esc_attr($postType); ?>" />

                <?php if ($showScopeChoice && $postId > 0): ?>
                <div class="wpo-form-field wpo-radio-group">
                    <label class="wpo-radio-label">
                        <input type="radio" name="scope" value="post" checked />
                        <span>Notify me about this <?php echo esc_html(strtolower($postTypeLabel)); ?> only</span>
                    </label>
                    <label class="wpo-radio-label">
                        <input type="radio" name="scope" value="type" />
                        <span>Notify me about all <?php echo esc_html(strtolower($postTypeLabelPlural)); ?></span>
                    </label>
                </div>
                <?php endif; ?>

                <?php if ($showTypeChoice): ?>
                <div class="wpo-form-field wpo-radio-group">
                    <label class="wpo-radio-label">
                        <input type="radio" name="subscription_type" value="both" <?php checked($atts['subscription_type'], 'both'); ?> />
                        <span>All updates (new and changes)</span>
                    </label>
                    <label class="wpo-radio-label">
                        <input type="radio" name="subscription_type" value="new" <?php checked($atts['subscription_type'], 'new'); ?> />
                        <span>New <?php echo esc_html(strtolower($postTypeLabelPlural)); ?> only</span>
                    </label>
                    <label class="wpo-radio-label">
                        <input type="radio" name="subscription_type" value="update" <?php checked($atts['subscription_type'], 'update'); ?> />
                        <span>Changes to existing content only</span>
                    </label>
                </div>
                <?php else: ?>
                <input type="hidden" name="subscription_type" value="<?php echo esc_attr($atts['subscription_type']); ?>" />
                <?php endif; ?>

                <div class="wpo-form-field">
                    <input
                        type="email"
                        name="email"
                        placeholder="<?php echo esc_attr($atts['placeholder_email']); ?>"
                        class="wpo-form-input"
                        required
                    />
                </div>

                <!-- Honeypot field (hidden) -->
                <div class="wpo-hp-field" aria-hidden="true">
                    <input type="text" name="website" tabindex="-1" autocomplete="off" />
                </div>

                <div class="wpo-form-field">
                    <button type="submit" class="wpo-form-button">
                        <span class="wpo-btn-text"><?php echo esc_html($atts['button_text']); ?></span>
                        <span class="wpo-btn-loading" style="display:none;">
                            <svg class="wpo-spinner" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10" stroke-dasharray="32" stroke-linecap="round"/>
                            </svg>
                        </span>
                    </button>
                </div>

                <div class="wpo-form-message" style="display:none;"></div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Process post subscription request (called from REST API)
     *
     * @param array $data Request data
     * @return array{success: bool, message: string}
     */
    public static function processSubscription(array $data): array
    {
        global $wpdb;

        // Check honeypot
        if (!empty($data['website'])) {
            // Silently fail for bots
            return [
                'success' => true,
                'message' => 'You will be notified when this content is updated!',
            ];
        }

        // Validate email
        $email = sanitize_email($data['email'] ?? '');
        if (empty($email) || !is_email($email)) {
            return [
                'success' => false,
                'message' => 'Please enter a valid email address.',
            ];
        }

        // Rate limiting
        $ip = self::getClientIp();
        if (!self::checkRateLimit($ip)) {
            return [
                'success' => false,
                'message' => 'Too many subscription attempts. Please try again later.',
            ];
        }

        // Get subscription parameters
        $postId = (int) ($data['post_id'] ?? 0);
        $postType = sanitize_key($data['post_type'] ?? 'post');
        $subscriptionType = sanitize_key($data['subscription_type'] ?? 'both');
        $scope = sanitize_key($data['scope'] ?? 'post');

        // Validate subscription type
        if (!in_array($subscriptionType, ['new', 'update', 'both'], true)) {
            $subscriptionType = 'both';
        }

        // If scope is 'type', set post_id to 0 (subscribe to all posts of type)
        if ($scope === 'type') {
            $postId = 0;
        }

        // Validate post exists if subscribing to specific post
        if ($postId > 0 && !get_post($postId)) {
            return [
                'success' => false,
                'message' => 'Invalid post.',
            ];
        }

        $subscribersTable = $wpdb->prefix . 'outreach_subscribers';

        // Find or create subscriber
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, status FROM {$subscribersTable} WHERE email = %s",
            $email
        ));

        if ($subscriber) {
            $subscriberId = (int) $subscriber->id;

            // Reactivate if unsubscribed
            if ($subscriber->status === 'unsubscribed') {
                $wpdb->update(
                    $subscribersTable,
                    ['status' => 'active', 'updated_at' => current_time('mysql')],
                    ['id' => $subscriberId]
                );
            }
        } else {
            // Create new subscriber
            $token = wp_generate_password(32, false);
            $result = $wpdb->insert($subscribersTable, [
                'email' => $email,
                'status' => 'active',
                'source' => 'post_subscription',
                'ip_address' => $ip,
                'token' => $token,
                'created_at' => current_time('mysql'),
            ]);

            if (!$result) {
                return [
                    'success' => false,
                    'message' => 'An error occurred. Please try again.',
                ];
            }

            $subscriberId = $wpdb->insert_id;

            // Fire subscriber created hook
            do_action('wp_outreach_subscriber_created', $subscriberId, [
                'email' => $email,
                'source' => 'post_subscription',
            ]);
        }

        // Create post subscription
        $handler = new PostSubscriptionHandler();
        $subscriptionId = $handler->subscribe($subscriberId, $postId, $postType, $subscriptionType);

        if ($subscriptionId === false) {
            return [
                'success' => false,
                'message' => 'An error occurred. Please try again.',
            ];
        }

        // Add to default list if configured
        $settings = get_option('wp_outreach_post_subscriptions', []);
        $defaultListId = $settings['default_list_id'] ?? null;
        if ($defaultListId) {
            $pivotTable = $wpdb->prefix . 'outreach_subscriber_list';
            // Check if already in list
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT subscriber_id FROM {$pivotTable} WHERE subscriber_id = %d AND list_id = %d",
                $subscriberId,
                $defaultListId
            ));
            if (!$exists) {
                $wpdb->insert($pivotTable, [
                    'subscriber_id' => $subscriberId,
                    'list_id' => $defaultListId,
                ]);
            }
        }

        // Fire post subscription created action for automations
        do_action('wp_outreach_post_subscription_created', $subscriberId, [
            'subscription_id' => $subscriptionId,
            'post_id' => $postId,
            'post_type' => $postType,
            'post_title' => $postId > 0 ? get_the_title($postId) : '',
            'post_url' => $postId > 0 ? get_permalink($postId) : '',
            'subscription_type' => $subscriptionType,
            'email' => $email,
        ]);

        // Record rate limit
        self::recordSubscription($ip);

        // Get appropriate success message
        if ($postId > 0) {
            $postTitle = get_the_title($postId);
            $message = sprintf('You will be notified when "%s" is updated!', $postTitle);
        } else {
            $postTypeObj = get_post_type_object($postType);
            $label = $postTypeObj ? strtolower($postTypeObj->labels->name) : $postType . 's';
            $message = sprintf('You will be notified about new %s!', $label);
        }

        return [
            'success' => true,
            'message' => $message,
        ];
    }

    /**
     * Get client IP address
     */
    private static function getClientIp(): string
    {
        $ip = '';

        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }

        return sanitize_text_field(trim($ip));
    }

    /**
     * Check rate limit for IP
     */
    private static function checkRateLimit(string $ip): bool
    {
        $transientKey = 'wpo_post_sub_' . md5($ip);
        $count = (int) get_transient($transientKey);

        return $count < self::RATE_LIMIT;
    }

    /**
     * Record subscription for rate limiting
     */
    private static function recordSubscription(string $ip): void
    {
        $transientKey = 'wpo_post_sub_' . md5($ip);
        $count = (int) get_transient($transientKey);

        set_transient($transientKey, $count + 1, self::RATE_WINDOW);
    }
}
