<?php

namespace WPOutreach\Forms;

/**
 * Double Opt-In Handler
 *
 * Handles confirmation email sending and token validation for
 * subscriber email verification.
 *
 * @since 1.0.0
 */
class DoubleOptIn
{
    /**
     * Token expiry in seconds (48 hours)
     */
    private const TOKEN_EXPIRY = 172800;

    /**
     * Send confirmation email to a pending subscriber
     *
     * @param int $subscriberId Subscriber ID
     * @return bool Whether email was sent successfully
     */
    public static function sendConfirmationEmail(int $subscriberId): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $subscriberId
        ));

        if (!$subscriber) {
            return false;
        }

        if ($subscriber->status !== 'pending') {
            return false;
        }

        $confirmUrl = self::getConfirmationUrl($subscriberId, $subscriber->token);
        $siteName = get_bloginfo('name');

        // Get general settings for from name/email
        $generalSettings = get_option('wp_outreach_general', []);
        $fromName = $generalSettings['from_name'] ?? $siteName;
        $fromEmail = $generalSettings['from_email'] ?? get_option('admin_email');

        $subject = sprintf(
            __('Please confirm your subscription to %s', 'outreach'),
            $siteName
        );

        $content = self::buildConfirmationEmail($subscriber, $confirmUrl, $siteName);

        // Set headers
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            sprintf('From: %s <%s>', $fromName, $fromEmail),
        ];

        // Send email
        $sent = wp_mail($subscriber->email, $subject, $content, $headers);

        // Log the attempt
        if ($sent) {
            $wpdb->update(
                $table,
                ['updated_at' => current_time('mysql')],
                ['id' => $subscriberId]
            );
        }

        return $sent;
    }

    /**
     * Generate confirmation URL
     *
     * @param int    $subscriberId Subscriber ID
     * @param string $token        Subscriber token
     * @return string Confirmation URL
     */
    public static function getConfirmationUrl(int $subscriberId, string $token): string
    {
        $data = base64_encode($subscriberId . ':' . $token);
        return rest_url('outreach/v1/confirm') . '?token=' . urlencode($data);
    }

    /**
     * Process confirmation request
     *
     * @param string $encodedToken Base64 encoded token from URL
     * @return array{success: bool, message: string, subscriber_id?: int}
     */
    public static function processConfirmation(string $encodedToken): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Decode token
        $decoded = base64_decode($encodedToken, true);
        if ($decoded === false) {
            return [
                'success' => false,
                'message' => __('Invalid confirmation link.', 'outreach'),
            ];
        }

        $parts = explode(':', $decoded, 2);
        if (count($parts) !== 2) {
            return [
                'success' => false,
                'message' => __('Invalid confirmation link.', 'outreach'),
            ];
        }

        [$subscriberId, $token] = $parts;
        $subscriberId = (int) $subscriberId;

        if ($subscriberId <= 0 || empty($token)) {
            return [
                'success' => false,
                'message' => __('Invalid confirmation link.', 'outreach'),
            ];
        }

        // Get subscriber
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $subscriberId
        ));

        if (!$subscriber) {
            return [
                'success' => false,
                'message' => __('Subscriber not found.', 'outreach'),
            ];
        }

        // Validate token using timing-safe comparison
        if (!hash_equals($subscriber->token, $token)) {
            return [
                'success' => false,
                'message' => __('Invalid confirmation link.', 'outreach'),
            ];
        }

        // Check if already confirmed
        if ($subscriber->status === 'active') {
            return [
                'success' => true,
                'message' => __('Your email has already been confirmed.', 'outreach'),
                'subscriber_id' => $subscriberId,
            ];
        }

        // Check token expiry (based on created_at)
        $createdAt = strtotime($subscriber->created_at);
        if (time() - $createdAt > self::TOKEN_EXPIRY) {
            return [
                'success' => false,
                'message' => __('This confirmation link has expired. Please subscribe again.', 'outreach'),
            ];
        }

        // Activate subscriber
        $wpdb->update(
            $table,
            [
                'status' => 'active',
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $subscriberId]
        );

        // Fire hook
        do_action('wp_outreach_subscriber_confirmed', $subscriberId, [
            'email' => $subscriber->email,
            'first_name' => $subscriber->first_name,
            'last_name' => $subscriber->last_name,
        ]);

        return [
            'success' => true,
            'message' => __('Thank you! Your email has been confirmed.', 'outreach'),
            'subscriber_id' => $subscriberId,
        ];
    }

    /**
     * Render confirmation result page
     *
     * @param array $result Result from processConfirmation
     * @return string HTML content
     */
    public static function renderConfirmationPage(array $result): string
    {
        $siteName = esc_html(get_bloginfo('name'));
        $siteUrl = esc_url(home_url('/'));
        $success = $result['success'];
        $message = esc_html($result['message']);

        $iconColor = $success ? '#10b981' : '#ef4444';
        $iconPath = $success
            ? 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'
            : 'M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z';
        $subMessage = esc_html(self::getSubMessage($success));

        ob_start();
        include WP_OUTREACH_PATH . 'templates/pages/confirmation.php';
        return ob_get_clean();
    }

    /**
     * Get sub-message based on result
     */
    private static function getSubMessage(bool $success): string
    {
        if ($success) {
            return __('You are now subscribed and will receive our updates. We\'re excited to have you!', 'outreach');
        }
        return __('We were unable to confirm your email address. The link may have expired or is invalid.', 'outreach');
    }

    /**
     * Build confirmation email HTML content
     *
     * @param object $subscriber Subscriber data
     * @param string $confirmUrl Confirmation URL
     * @param string $siteName   Site name
     * @return string HTML email content
     */
    private static function buildConfirmationEmail(object $subscriber, string $confirmUrl, string $siteName): string
    {
        $siteName = esc_html($siteName);
        $firstName = esc_html($subscriber->first_name ?: 'there');
        $confirmUrl = esc_url($confirmUrl);
        $siteUrl = esc_url(home_url('/'));
        $year = date('Y');

        ob_start();
        include WP_OUTREACH_PATH . 'templates/emails/confirmation.php';
        return ob_get_clean();
    }

    /**
     * Resend confirmation email
     *
     * @param string $email Subscriber email
     * @return array{success: bool, message: string}
     */
    public static function resendConfirmation(string $email): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE email = %s AND status = 'pending'",
            $email
        ));

        if (!$subscriber) {
            // Don't reveal if email exists or not
            return [
                'success' => true,
                'message' => __('If you have a pending subscription, you will receive a new confirmation email.', 'outreach'),
            ];
        }

        // Regenerate token and reset created_at for fresh expiry
        $newToken = wp_generate_password(32, false);
        $wpdb->update(
            $table,
            [
                'token' => $newToken,
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $subscriber->id]
        );

        // Update subscriber object for email sending
        $subscriber->token = $newToken;

        $confirmUrl = self::getConfirmationUrl($subscriber->id, $newToken);
        $siteName = get_bloginfo('name');
        $generalSettings = get_option('wp_outreach_general', []);
        $fromName = $generalSettings['from_name'] ?? $siteName;
        $fromEmail = $generalSettings['from_email'] ?? get_option('admin_email');

        $subject = sprintf(
            __('Please confirm your subscription to %s', 'outreach'),
            $siteName
        );

        $content = self::buildConfirmationEmail($subscriber, $confirmUrl, $siteName);

        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            sprintf('From: %s <%s>', $fromName, $fromEmail),
        ];

        wp_mail($subscriber->email, $subject, $content, $headers);

        return [
            'success' => true,
            'message' => __('If you have a pending subscription, you will receive a new confirmation email.', 'outreach'),
        ];
    }
}
