<?php

namespace WPOutreach;

use WPOutreach\Admin\AdminPage;
use WPOutreach\Admin\Assets;
// UpdateChecker removed for WordPress.org hosting (auto-updates handled by WP.org)
use WPOutreach\API\RestController;
use WPOutreach\Automation\AutomationEngine;
use WPOutreach\Automation\ContentTriggerHandler;
use WPOutreach\Automation\TriggerRegistry;
use WPOutreach\Forms\SubscriptionForm;
use WPOutreach\Forms\PostSubscriptionForm;
use WPOutreach\Integrations\WPDMIntegration;
use WPOutreach\Integrations\Forms\FormIntegrationsLoader;
use WPOutreach\Mailer\Queue\QueueWorker;
use WPOutreach\Mailer\WPMailOverride;
use WPOutreach\Subscriptions\PostSubscriptionHandler;
use WPOutreach\Subscriptions\PostSubscriptionInjector;
use WPOutreach\Subscriptions\PostSubscriptionMetaBox;

/**
 * Main plugin bootstrap class
 */
class Bootstrap {

    /**
     * Initialize the plugin
     */
    public function init(): void {
        $this->maybe_upgrade_database();
        // Note: load_textdomain() removed for WordPress.org compliance (auto-loaded since WP 4.6)
        $this->register_cron_schedules();
        $this->init_cron_handlers();
        $this->init_admin();
        $this->init_api();
        $this->init_frontend();
        $this->init_mail_override();

        // Register user_register hook early (plugins_loaded) to catch all registration sources
        // The handler queues the trigger fire until TriggerRegistry is ready
        add_action('user_register', [$this, 'handle_user_registered'], 10, 2);

        // Register wp_login hook early (plugins_loaded) to catch all login sources
        // Some plugins/themes may trigger login before 'init' action
        add_action('wp_login', [$this, 'handle_user_login'], 10, 2);

        // Defer automation initialization to 'init' to avoid early translation loading
        // This fixes the "_load_textdomain_just_in_time" notice in WordPress 6.7+
        add_action('init', [$this, 'init_automations']);
    }

    /**
     * Initialize WordPress mail override if enabled
     */
    private function init_mail_override(): void {
        WPMailOverride::init();
    }

    /**
     * Initialize automation engine
     *
     * Called on 'init' action to ensure translations are loaded first.
     * This fixes the "_load_textdomain_just_in_time" notice in WordPress 6.7+.
     */
    public function init_automations(): void {
        // Initialize integrations BEFORE AutomationEngine so triggers are registered
        // before do_action('wp_outreach_register_triggers') is fired
        WPDMIntegration::init();

        // Initialize form integrations (CF7, WPForms, Gravity Forms, WooCommerce)
        FormIntegrationsLoader::init();

        AutomationEngine::init();

        // Initialize content-based triggers (post_published, post_updated for automations)
        ContentTriggerHandler::init();

        // Register trigger hooks for post subscriptions
        $this->register_trigger_hooks();
    }

    /**
     * Register hooks that fire automation triggers
     *
     * This method connects WordPress hooks to WP Outreach automation triggers.
     * Each hook handler:
     * 1. Collects relevant context data from the event
     * 2. Finds or creates a subscriber (if needed)
     * 3. Fires the corresponding trigger via TriggerRegistry::fire()
     *
     * Hook → Trigger mapping:
     * - wp_outreach_post_subscription_created → post_subscription_created
     * - user_register                         → wp_user_registered
     * - wp_login                              → wp_user_login
     *
     * Note: Some triggers are fired from other places:
     * - subscriber_created      : Fired in RestController when subscriber is created
     * - subscriber_joins_list   : Fired in RestController when list is assigned
     * - subscriber_added_tag    : Fired in RestController when tag is added
     * - post_published          : Fired in ContentTriggerHandler
     * - post_updated            : Fired in ContentTriggerHandler
     *
     * @see TriggerRegistry for full trigger documentation
     * @return void
     */
    private function register_trigger_hooks(): void {
        /**
         * Hook: wp_outreach_post_subscription_created
         * Trigger: post_subscription_created
         *
         * Fires when someone subscribes to a post/page for content updates.
         * The subscriber_id and data are passed from PostSubscriptionHandler.
         */
        add_action('wp_outreach_post_subscription_created', function (int $subscriber_id, array $data) {
            TriggerRegistry::fire('post_subscription_created', $subscriber_id, $data);
        }, 10, 2);

        // Note: user_register hook is registered earlier in init() to catch all registration sources

        // Note: wp_login hook is registered earlier in init() to catch all login sources

        /**
         * Hook: wp_outreach_email_opened
         * Trigger: email_opened
         *
         * Fires when a subscriber opens an email (first unique open only).
         * The log data is passed from OpenTracker::recordOpen().
         *
         * @see handle_email_opened()
         */
        add_action('wp_outreach_email_opened', [$this, 'handle_email_opened'], 10, 1);

        /**
         * Hook: wp_outreach_email_clicked
         * Trigger: email_clicked
         *
         * Fires when a subscriber clicks a link in an email (first click only).
         * The log data and clicked URL are passed from ClickTracker::recordClick().
         *
         * @see handle_email_clicked()
         */
        add_action('wp_outreach_email_clicked', [$this, 'handle_email_clicked'], 10, 2);
    }

    /**
     * Handle WordPress user registration and fire automation trigger
     *
     * This handler is called when a new WordPress user is created via:
     * - Registration form
     * - Admin creating a user
     * - Plugin creating a user programmatically
     *
     * What it does:
     * 1. Gets the full user object and extracts name/email
     * 2. Creates a subscriber record (or finds existing by email)
     * 3. Fires the 'wp_user_registered' trigger with context data
     *
     * Context data provided:
     * - user_id      : WordPress user ID
     * - user_email   : User's email address
     * - user_login   : Username
     * - display_name : Display name
     * - first_name   : First name (from user meta or display_name)
     * - last_name    : Last name (from user meta or display_name)
     * - user_role    : Primary role (first role in roles array)
     * - user_roles   : Array of all assigned roles
     *
     * The subscriber is created with:
     * - status: 'active' (WP users are already confirmed)
     * - source: 'wp_user'
     *
     * @param int   $user_id  The new user's ID
     * @param array $userdata The user data passed to wp_insert_user() (WP 5.8+)
     *
     * @see TriggerRegistry::fire() for how triggers are processed
     * @see find_or_create_subscriber_from_user() for subscriber creation
     */
    public function handle_user_registered(int $user_id, array $userdata = []): void {
        // Get the full user object
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }

        // Get user's primary role
        $user_role = !empty($user->roles) ? $user->roles[0] : 'subscriber';

        // Get user meta for first/last name
        $first_name = get_user_meta($user_id, 'first_name', true);
        $last_name = get_user_meta($user_id, 'last_name', true);

        // If first_name is empty, try to extract from display_name
        if (empty($first_name) && !empty($user->display_name)) {
            $name_parts = explode(' ', $user->display_name, 2);
            $first_name = $name_parts[0];
            $last_name = isset($name_parts[1]) ? $name_parts[1] : $last_name;
        }

        // Build context data for the trigger
        $context = [
            'user_id'      => $user_id,
            'user_email'   => $user->user_email,
            'user_login'   => $user->user_login,
            'display_name' => $user->display_name,
            'first_name'   => $first_name,
            'last_name'    => $last_name,
            'user_role'    => $user_role,
            'user_roles'   => $user->roles,
        ];

        // Find or create subscriber from WP user
        $subscriber_id = $this->find_or_create_subscriber_from_user($user, $first_name, $last_name);

        if ($subscriber_id) {
            // Fire the trigger
            TriggerRegistry::fire('wp_user_registered', $subscriber_id, $context);

            // Allow other plugins to hook in
            do_action('wp_outreach_user_registered', $subscriber_id, $user_id, $context);
        }
    }

    /**
     * Find or create a subscriber from WordPress user data
     *
     * @param \WP_User $user       The WordPress user object
     * @param string   $first_name First name
     * @param string   $last_name  Last name
     * @return int|null Subscriber ID or null on failure
     */
    private function find_or_create_subscriber_from_user(\WP_User $user, string $first_name, string $last_name): ?int {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Check if subscriber already exists with this email
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s",
            $user->user_email
        ));

        if ($existing) {
            return (int) $existing;
        }

        // Create new subscriber
        $result = $wpdb->insert($table, [
            'email'      => $user->user_email,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'status'     => 'active', // WP users are already confirmed
            'source'     => 'wp_user',
            'created_at' => current_time('mysql'),
        ]);

        if ($result === false) {
            return null;
        }

        $subscriber_id = $wpdb->insert_id;

        // Fire subscriber created hook
        do_action('wp_outreach_subscriber_created', $subscriber_id, [
            'email'      => $user->user_email,
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'source'     => 'wp_user',
        ]);

        return $subscriber_id;
    }

    /**
     * Handle WordPress user login and fire automation trigger
     *
     * This handler is called every time a WordPress user logs in.
     * It tracks the last login time and calculates inactivity period.
     *
     * What it does:
     * 1. Gets the last login timestamp from user meta (wp_outreach_last_login)
     * 2. Calculates days since last login
     * 3. Updates the last login timestamp (BEFORE firing trigger)
     * 4. Creates/finds subscriber from WP user
     * 5. Fires the 'wp_user_login' trigger with context data
     *
     * Login Tracking:
     * - Stores timestamp in user meta: wp_outreach_last_login
     * - Updates BEFORE firing trigger so next login has correct data
     * - First login ever: days_since_last_login = PHP_INT_MAX (passes any min_days check)
     *
     * Context data provided:
     * - user_id               : WordPress user ID
     * - user_email            : User's email address
     * - user_login            : Username
     * - display_name          : Display name
     * - first_name            : First name
     * - last_name             : Last name
     * - user_role             : Primary role
     * - user_roles            : Array of all roles
     * - days_since_last_login : Days since previous login (PHP_INT_MAX for first login)
     * - last_login_at         : Previous login datetime string (null if first login)
     *
     * Inactivity Filter:
     * The trigger has a 'min_days_since_last_login' config option.
     * When set, the trigger only fires if days_since_last_login >= min_days.
     * This prevents spamming users who log in frequently.
     *
     * Example use cases:
     * - "Welcome back!" email after 30+ days of inactivity
     * - Re-engagement campaign for users inactive 90+ days
     * - Tag users on each login for engagement tracking
     *
     * @param string   $user_login Username (passed by wp_login hook)
     * @param \WP_User $user       The logged-in user object
     *
     * @see TriggerRegistry for trigger configuration options
     */
    public function handle_user_login(string $user_login, \WP_User $user): void {
        // Get last login timestamp from user meta
        $last_login = get_user_meta($user->ID, 'wp_outreach_last_login', true);
        $last_login_timestamp = $last_login ? (int) $last_login : 0;

        // Calculate days since last login
        $days_since_last_login = 0;
        if ($last_login_timestamp > 0) {
            $days_since_last_login = floor((time() - $last_login_timestamp) / DAY_IN_SECONDS);
        } else {
            // First login ever - treat as infinite days (will pass any min_days check)
            $days_since_last_login = PHP_INT_MAX;
        }

        // Update last login timestamp BEFORE firing trigger
        update_user_meta($user->ID, 'wp_outreach_last_login', time());

        // Get user's primary role
        $user_role = !empty($user->roles) ? $user->roles[0] : 'subscriber';

        // Get user meta for first/last name
        $first_name = get_user_meta($user->ID, 'first_name', true);
        $last_name = get_user_meta($user->ID, 'last_name', true);

        // If first_name is empty, try to extract from display_name
        if (empty($first_name) && !empty($user->display_name)) {
            $name_parts = explode(' ', $user->display_name, 2);
            $first_name = $name_parts[0];
            $last_name = isset($name_parts[1]) ? $name_parts[1] : $last_name;
        }

        // Build context data for the trigger
        $context = [
            'user_id'              => $user->ID,
            'user_email'           => $user->user_email,
            'email'                => $user->user_email, // Alias for condition compatibility
            'user_login'           => $user->user_login,
            'display_name'         => $user->display_name,
            'first_name'           => $first_name,
            'last_name'            => $last_name,
            'user_role'            => $user_role,
            'user_roles'           => $user->roles,
            'days_since_last_login' => $days_since_last_login,
            'last_login_at'        => $last_login_timestamp > 0 ? date('Y-m-d H:i:s', $last_login_timestamp) : null,
        ];

        // Find or create subscriber from WP user
        $subscriber_id = $this->find_or_create_subscriber_from_user($user, $first_name, $last_name);

        if ($subscriber_id) {
            // Fire the trigger
            TriggerRegistry::fire('wp_user_login', $subscriber_id, $context);

            // Allow other plugins to hook in
            do_action('wp_outreach_user_login', $subscriber_id, $user->ID, $context);
        }
    }

    /**
     * Handle email opened event and fire automation trigger
     *
     * This handler is called when a subscriber opens an email for the first time.
     * The OpenTracker fires this hook with the full log entry data.
     *
     * What it does:
     * 1. Validates the log entry has a subscriber_id
     * 2. Fetches subscriber data for context
     * 3. Fires the 'email_opened' trigger with context data
     *
     * Context data provided:
     * - campaign_id    : Campaign ID (if from campaign)
     * - automation_id  : Automation ID (if from automation)
     * - email          : Subscriber email
     * - subject        : Email subject line
     * - tracking_id    : Unique tracking ID
     * - first_name     : Subscriber first name
     * - last_name      : Subscriber last name
     * - opened_at      : When the email was opened
     * - sent_at        : When the email was originally sent
     *
     * @param object $log The log entry from wp_outreach_logs table
     *
     * @see OpenTracker::recordOpen() for hook source
     * @see TriggerRegistry for trigger configuration
     */
    public function handle_email_opened(object $log): void {
        // Validate we have a subscriber
        if (empty($log->subscriber_id)) {
            return;
        }

        // Get subscriber data for context
        global $wpdb;
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_subscribers WHERE id = %d",
            $log->subscriber_id
        ));

        if (!$subscriber) {
            return;
        }

        // Build context data
        $context = [
            'campaign_id'   => $log->campaign_id ?? null,
            'automation_id' => $log->automation_id ?? null,
            'email'         => $log->email,
            'subject'       => $log->subject ?? '',
            'tracking_id'   => $log->tracking_id ?? '',
            'first_name'    => $subscriber->first_name ?? '',
            'last_name'     => $subscriber->last_name ?? '',
            'opened_at'     => $log->first_opened_at ?? current_time('mysql'),
            'sent_at'       => $log->sent_at ?? '',
        ];

        // Fire the trigger
        TriggerRegistry::fire('email_opened', (int) $log->subscriber_id, $context);
    }

    /**
     * Handle email clicked event and fire automation trigger
     *
     * This handler is called when a subscriber clicks a link in an email
     * for the first time. The ClickTracker fires this hook with log entry
     * and the clicked URL.
     *
     * What it does:
     * 1. Validates the log entry has a subscriber_id
     * 2. Fetches subscriber data for context
     * 3. Fires the 'email_clicked' trigger with context data
     *
     * Context data provided:
     * - campaign_id    : Campaign ID (if from campaign)
     * - automation_id  : Automation ID (if from automation)
     * - email          : Subscriber email
     * - subject        : Email subject line
     * - tracking_id    : Unique tracking ID
     * - first_name     : Subscriber first name
     * - last_name      : Subscriber last name
     * - clicked_url    : The URL that was clicked
     * - clicked_at     : When the link was clicked
     * - sent_at        : When the email was originally sent
     *
     * @param object $log        The log entry from wp_outreach_logs table
     * @param string $clicked_url The URL that was clicked
     *
     * @see ClickTracker::recordClick() for hook source
     * @see TriggerRegistry for trigger configuration
     */
    public function handle_email_clicked(object $log, string $clicked_url): void {
        // Validate we have a subscriber
        if (empty($log->subscriber_id)) {
            return;
        }

        // Get subscriber data for context
        global $wpdb;
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_subscribers WHERE id = %d",
            $log->subscriber_id
        ));

        if (!$subscriber) {
            return;
        }

        // Build context data
        $context = [
            'campaign_id'   => $log->campaign_id ?? null,
            'automation_id' => $log->automation_id ?? null,
            'email'         => $log->email,
            'subject'       => $log->subject ?? '',
            'tracking_id'   => $log->tracking_id ?? '',
            'first_name'    => $subscriber->first_name ?? '',
            'last_name'     => $subscriber->last_name ?? '',
            'clicked_url'   => $clicked_url,
            'clicked_at'    => current_time('mysql'),
            'sent_at'       => $log->sent_at ?? '',
        ];

        // Fire the trigger
        TriggerRegistry::fire('email_clicked', (int) $log->subscriber_id, $context);
    }

    /**
     * Check if database needs upgrading and run migrations
     */
    private function maybe_upgrade_database(): void {
        $db_version = get_option('wp_outreach_db_version', '0');
        if (version_compare($db_version, WP_OUTREACH_VERSION, '<')) {
            $installer = new Installer();
            $installer->upgrade();
        }
    }

    /**
     * Register custom cron schedules
     */
    private function register_cron_schedules(): void {
        add_filter('cron_schedules', function ($schedules) {
            $schedules['wp_outreach_minute'] = [
                'interval' => 60,
                'display' => __('Every Minute (WP Outreach)', 'outreach'),
            ];
            return $schedules;
        });
    }

    /**
     * Initialize cron job handlers - processes ALL queue types
     *
     * IMPORTANT: This method MUST stay in sync with RestController::run_cron()
     * Both process the same 4 tasks in the same order:
     *
     * 1. Scheduled Campaigns  - One-time campaigns scheduled for future delivery
     * 2. Recurring Campaigns  - Campaigns that repeat on a schedule (daily/weekly/monthly)
     * 3. Automation Queue     - Automation workflow steps (wp_outreach_automation_queue table)
     * 4. Email Queue          - Pending emails waiting to be sent (wp_outreach_queue table)
     *
     * Queue Processing Flow:
     * ┌─────────────────────────────────────────────────────────────────────┐
     * │ TRIGGERS (create queue entries)                                      │
     * │ ├── Subscriber events → AutomationEngine → wp_outreach_automation_queue │
     * │ ├── Post published    → AutomationEngine → wp_outreach_automation_queue │
     * │ ├── Campaign send     → QueueManager     → wp_outreach_queue            │
     * │ └── Scheduled campaign→ This method      → wp_outreach_queue            │
     * ├─────────────────────────────────────────────────────────────────────┤
     * │ PROCESSING (this method / External Cron)                             │
     * │ ├── AutomationEngine::processQueue() → Executes automation steps     │
     * │ │   └── May add emails to wp_outreach_queue via send_email action    │
     * │ └── QueueWorker::processQueue()      → Sends emails via mailer       │
     * └─────────────────────────────────────────────────────────────────────┘
     *
     * @see RestController::run_cron() - External cron version (must match this)
     * @see AutomationEngine::processQueue() - Processes automation steps
     * @see QueueWorker::processQueue() - Sends queued emails
     */
    private function init_cron_handlers(): void {
        $cron_settings = get_option('wp_outreach_cron', []);
        $cron_mode = $cron_settings['mode'] ?? 'wp_cron';

        // Only register WP Cron handlers if mode is 'wp_cron' or 'both'
        if ($cron_mode === 'wp_cron' || $cron_mode === 'both') {
            // Main cron processing - 4 steps (must match RestController::run_cron)
            add_action('wp_outreach_process_queue', function () {
                // =====================================================================
                // CRON PROCESSING - 4 STEPS (must match RestController::run_cron)
                // =====================================================================

                // STEP 1: Scheduled Campaigns
                // - Finds campaigns with type='scheduled', status='scheduled', scheduled_at <= now
                // - Queues emails to wp_outreach_queue for each subscriber
                $this->process_scheduled_campaigns();

                // STEP 2: Recurring Campaigns
                // - Finds campaigns with type='recurring', status='scheduled', next_send_at <= now
                // - Queues emails and calculates next_send_at
                $this->process_recurring_campaigns();

                // STEP 3: Automation Queue (CRITICAL - don't forget this!)
                // - Processes wp_outreach_automation_queue table
                // - Executes automation steps (send_email, wait, add_tag, webhook, etc.)
                // - Triggered by: subscriber_created, tag_added, post_published, etc.
                AutomationEngine::processQueue();

                // STEP 4: Email Queue
                // - Processes wp_outreach_queue table (entries from campaigns + automations)
                // - Sends emails via configured mailer (WP Mail, SMTP, or SES)
                if (!QueueWorker::isPaused()) {
                    QueueWorker::processQueue();
                }

                // STEP 5: Unopened Email Triggers
                // - Checks for emails that haven't been opened after X days
                // - Fires email_not_opened trigger for each matching email
                $this->process_unopened_email_triggers();
            });

            // Daily cleanup
            add_action('wp_outreach_cleanup_queue', [QueueWorker::class, 'cleanup']);
        }
    }

    /**
     * Process scheduled campaigns that are due
     */
    private function process_scheduled_campaigns(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $now = current_time('mysql');

        // Find scheduled campaigns that are due
        $campaigns = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table}
             WHERE type = 'scheduled'
             AND status = 'scheduled'
             AND scheduled_at <= %s",
            $now
        ));

        $processed = 0;

        foreach ($campaigns as $campaign) {
            // Get subscribers from lists
            $list_ids = json_decode($campaign->list_ids, true) ?: [];

            if (empty($list_ids)) {
                $wpdb->update($table, ['status' => 'draft'], ['id' => $campaign->id]);
                continue;
            }

            $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

            if (empty($subscriber_ids)) {
                $wpdb->update($table, ['status' => 'sent', 'sent_at' => $now], ['id' => $campaign->id]);
                continue;
            }

            // Update status to 'sending'
            $wpdb->update($table, ['status' => 'sending'], ['id' => $campaign->id]);

            // Queue emails
            $queue = new \WPOutreach\Mailer\Queue\QueueManager();
            $queue->queueCampaign($campaign->id, $subscriber_ids, $campaign->subject, $campaign->content);

            $processed++;
        }

        return $processed;
    }

    /**
     * Process recurring campaigns that are due
     */
    private function process_recurring_campaigns(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $now = current_time('mysql');
        $now_timestamp = current_time('timestamp');

        // Find recurring campaigns that are due
        $campaigns = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table}
             WHERE type = 'recurring'
             AND status = 'scheduled'
             AND next_send_at IS NOT NULL
             AND next_send_at <= %s",
            $now
        ));

        $processed = 0;

        foreach ($campaigns as $campaign) {
            $recurring_config = json_decode($campaign->recurring_config, true);

            if (!$recurring_config) {
                continue;
            }

            // Check start_date constraint
            if (!empty($recurring_config['start_date'])) {
                $start_date = strtotime($recurring_config['start_date']);
                if ($start_date > $now_timestamp) {
                    continue; // Not started yet
                }
            }

            // Check end_date constraint
            if (!empty($recurring_config['end_date'])) {
                $end_date = strtotime($recurring_config['end_date'] . ' 23:59:59');
                if ($end_date < $now_timestamp) {
                    // Campaign has ended - mark as sent
                    $wpdb->update($table, ['status' => 'sent'], ['id' => $campaign->id]);
                    continue;
                }
            }

            // Get subscribers from lists
            $list_ids = json_decode($campaign->list_ids, true) ?: [];

            if (empty($list_ids)) {
                continue;
            }

            $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

            if (empty($subscriber_ids)) {
                // No subscribers, just update next_send_at
                $next_send_at = $this->calculate_next_send_time($recurring_config);
                $wpdb->update($table, [
                    'last_sent_at' => $now,
                    'next_send_at' => $next_send_at,
                ], ['id' => $campaign->id]);
                continue;
            }

            // Update status to 'sending'
            $wpdb->update($table, ['status' => 'sending'], ['id' => $campaign->id]);

            // Queue emails
            $queue = new \WPOutreach\Mailer\Queue\QueueManager();
            $queue->queueCampaign($campaign->id, $subscriber_ids, $campaign->subject, $campaign->content);

            // Calculate next send time and update campaign
            $next_send_at = $this->calculate_next_send_time($recurring_config);

            // Check if next send is after end_date
            $campaign_ended = false;
            if (!empty($recurring_config['end_date'])) {
                $end_date = strtotime($recurring_config['end_date'] . ' 23:59:59');
                $next_timestamp = strtotime($next_send_at);
                if ($next_timestamp > $end_date) {
                    $campaign_ended = true;
                }
            }

            // Update last_sent_at and next_send_at (status will be updated when sending completes)
            $update_data = [
                'last_sent_at' => $now,
                'next_send_at' => $campaign_ended ? null : $next_send_at,
            ];

            $wpdb->update($table, $update_data, ['id' => $campaign->id]);

            $processed++;
        }

        return $processed;
    }

    /**
     * Calculate next send time for recurring campaign
     */
    private function calculate_next_send_time(array $config): ?string {
        $frequency = $config['frequency'] ?? 'weekly';
        $time = $config['time'] ?? '09:00';

        // Use WordPress timezone for consistent calculations
        $wp_timezone = wp_timezone();
        $now = new \DateTime('now', $wp_timezone);

        switch ($frequency) {
            case 'daily':
                $next = new \DateTime("today {$time}", $wp_timezone);
                if ($next <= $now) {
                    $next = new \DateTime("tomorrow {$time}", $wp_timezone);
                }
                break;

            case 'weekly':
                $day = (int) ($config['day_of_week'] ?? 1);
                $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                $day_name = $days[$day] ?? 'Monday';
                $next = new \DateTime("this {$day_name} {$time}", $wp_timezone);
                if ($next <= $now) {
                    $next = new \DateTime("next {$day_name} {$time}", $wp_timezone);
                }
                break;

            case 'monthly':
                $day_config = $config['day_of_month'] ?? 1;

                if ($day_config === 'last') {
                    // Last day of current month
                    $next = new \DateTime('last day of this month ' . $time, $wp_timezone);
                    if ($next <= $now) {
                        // Last day of next month
                        $next = new \DateTime('last day of next month ' . $time, $wp_timezone);
                    }
                } else {
                    $day = (int) $day_config;
                    $day = max(1, min($day, 31)); // Clamp to valid range 1-31

                    // Get the number of days in current month
                    $days_in_current_month = (int) $now->format('t');
                    $effective_day = min($day, $days_in_current_month);

                    $next = new \DateTime($now->format('Y-m-') . sprintf('%02d', $effective_day) . " {$time}", $wp_timezone);
                    if ($next <= $now) {
                        // Move to next month
                        $next->modify('first day of next month');
                        $days_in_next_month = (int) $next->format('t');
                        $effective_day = min($day, $days_in_next_month);
                        $next->setDate((int)$next->format('Y'), (int)$next->format('m'), $effective_day);
                        $next = new \DateTime($next->format('Y-m-d') . " {$time}", $wp_timezone);
                    }
                }
                break;

            default:
                return null;
        }

        // Return in MySQL datetime format (WordPress local time)
        return $next->format('Y-m-d H:i:s');
    }

    /**
     * Process unopened email triggers
     *
     * Checks for emails that haven't been opened after X days and fires
     * the email_not_opened trigger for each matching email.
     *
     * Processing Logic:
     * 1. Find all active automations using email_not_opened trigger
     * 2. For each automation, extract the configured days threshold
     * 3. Find log entries where:
     *    - opens = 0 (never opened)
     *    - sent_at is older than the threshold
     *    - not_opened_triggered IS NULL (not already triggered)
     * 4. Fire the trigger for each matching log entry
     * 5. Mark the log entry as triggered to prevent duplicates
     *
     * Performance Notes:
     * - Uses a batch limit (100 emails per run) to avoid long-running queries
     * - Groups by days threshold to minimize database queries
     * - Only processes automations that are 'active' status
     *
     * @return int Number of triggers fired
     */
    private function process_unopened_email_triggers(): int {
        global $wpdb;

        $automations_table = $wpdb->prefix . 'outreach_automations';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        // Find all active automations using email_not_opened trigger
        $automations = $wpdb->get_results(
            "SELECT id, trigger_config FROM {$automations_table}
             WHERE status = 'active'
             AND trigger_type = 'email_not_opened'"
        );

        if (empty($automations)) {
            return 0;
        }

        // Group automations by days threshold to minimize queries
        $thresholds = [];
        foreach ($automations as $automation) {
            $config = json_decode($automation->trigger_config, true) ?: [];
            $days = (int) ($config['days'] ?? 0);
            $campaign_id = $config['campaign_id'] ?? null;

            if ($days <= 0) {
                continue;
            }

            $key = $days . '_' . ($campaign_id ?? 'any');
            if (!isset($thresholds[$key])) {
                $thresholds[$key] = [
                    'days' => $days,
                    'campaign_id' => $campaign_id,
                    'automation_ids' => [],
                ];
            }
            $thresholds[$key]['automation_ids'][] = $automation->id;
        }

        if (empty($thresholds)) {
            return 0;
        }

        $triggered = 0;

        foreach ($thresholds as $threshold) {
            $days = $threshold['days'];
            $campaign_id = $threshold['campaign_id'];

            // Calculate the cutoff date (emails sent before this date are eligible)
            $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));

            // Build query to find unopened emails
            $query = "SELECT l.*, s.first_name, s.last_name
                      FROM {$logs_table} l
                      LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
                      WHERE l.opens = 0
                      AND l.sent_at <= %s
                      AND l.not_opened_triggered IS NULL
                      AND l.subscriber_id IS NOT NULL";

            $params = [$cutoff_date];

            // Filter by campaign if specified
            if (!empty($campaign_id)) {
                $query .= " AND l.campaign_id = %d";
                $params[] = $campaign_id;
            }

            $query .= " LIMIT 100"; // Batch limit to avoid long-running queries

            $logs = $wpdb->get_results($wpdb->prepare($query, ...$params));

            foreach ($logs as $log) {
                // Calculate exact days since sent
                $sent_timestamp = strtotime($log->sent_at);
                $days_unopened = floor((time() - $sent_timestamp) / DAY_IN_SECONDS);

                // Build context data
                $context = [
                    'campaign_id'   => $log->campaign_id ?? null,
                    'automation_id' => $log->automation_id ?? null,
                    'email'         => $log->email,
                    'subject'       => $log->subject ?? '',
                    'tracking_id'   => $log->tracking_id ?? '',
                    'first_name'    => $log->first_name ?? '',
                    'last_name'     => $log->last_name ?? '',
                    'sent_at'       => $log->sent_at,
                    'days_unopened' => $days_unopened,
                ];

                // Fire the trigger
                $result = TriggerRegistry::fire('email_not_opened', (int) $log->subscriber_id, $context);

                if ($result > 0) {
                    $triggered++;
                }

                // Mark this log entry as triggered to prevent duplicate triggers
                $wpdb->update(
                    $logs_table,
                    ['not_opened_triggered' => current_time('mysql')],
                    ['id' => $log->id]
                );
            }
        }

        return $triggered;
    }

    /**
     * Get subscribers for given list IDs
     */
    private function get_subscribers_for_lists(array $list_ids): array {
        global $wpdb;

        if (empty($list_ids)) {
            return [];
        }

        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $placeholders = implode(',', array_fill(0, count($list_ids), '%d'));

        return $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT s.id
             FROM {$subscribers_table} s
             INNER JOIN {$pivot_table} sl ON s.id = sl.subscriber_id
             WHERE sl.list_id IN ({$placeholders})
             AND s.status = 'active'",
            ...$list_ids
        ));
    }

    /**
     * Initialize admin components
     */
    private function init_admin(): void {
        if (!is_admin()) {
            return;
        }

        $admin_page = new AdminPage();
        $admin_page->init();

        $assets = new Assets();
        $assets->init();

        // Post subscription meta box (notify checkbox in publish panel)
        PostSubscriptionMetaBox::init();
    }

    /**
     * Initialize REST API
     */
    private function init_api(): void {
        add_action('rest_api_init', function() {
            $controller = new RestController();
            $controller->register_routes();
        });
    }

    /**
     * Initialize frontend components
     */
    private function init_frontend(): void {
        // Subscription forms and shortcodes
        SubscriptionForm::init();

        // Post subscription form shortcode
        PostSubscriptionForm::init();

        // Post subscription handlers (for content update notifications)
        PostSubscriptionHandler::init();

        // Post subscription auto-injector (settings-driven form injection)
        PostSubscriptionInjector::init();

        // Tracking endpoints
        add_action('init', [$this, 'register_rewrite_rules']);
        add_action('template_redirect', [$this, 'handle_tracking']);
    }

    /**
     * Register custom rewrite rules for tracking
     */
    public function register_rewrite_rules(): void {
        add_rewrite_rule(
            '^outreach/t/o/([a-zA-Z0-9]+)/?$',
            'index.php?outreach_track=open&outreach_hash=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^outreach/t/c/([a-zA-Z0-9]+)/?$',
            'index.php?outreach_track=click&outreach_hash=$matches[1]',
            'top'
        );

        add_rewrite_tag('%outreach_track%', '([^&]+)');
        add_rewrite_tag('%outreach_hash%', '([^&]+)');
    }

    /**
     * Handle tracking requests
     */
    public function handle_tracking(): void {
        $track_type = get_query_var('outreach_track');
        $hash = get_query_var('outreach_hash');

        if (!$track_type || !$hash) {
            return;
        }

        // Tracking will be implemented in Phase 7
        if ($track_type === 'open') {
            // Return 1x1 transparent GIF
            header('Content-Type: image/gif');
            header('Cache-Control: no-cache, no-store, must-revalidate');
            echo base64_decode('R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7');
            exit;
        }

        if ($track_type === 'click') {
            // Redirect to original URL (to be implemented)
            wp_redirect(home_url());
            exit;
        }
    }
}
