<?php

namespace WPOutreach\Automation;

/**
 * Trigger Registry - Core Automation Trigger Management
 *
 * This class manages all automation triggers that can start a workflow.
 * Triggers are events that, when they occur, can initiate an automation
 * sequence (e.g., send email, add tag, wait, etc.).
 *
 * @since 1.0.0
 * @package WPOutreach\Automation
 *
 * ============================================================================
 * TRIGGER SYSTEM OVERVIEW
 * ============================================================================
 *
 * How Triggers Work:
 * 1. A trigger is registered with an ID, name, and optional configuration fields
 * 2. When the corresponding WordPress hook fires, TriggerRegistry::fire() is called
 * 3. The system finds all active automations using that trigger
 * 4. For each automation, the "matcher" function checks if the trigger config matches
 * 5. If matched, the subscriber is enqueued into the automation workflow
 *
 * Trigger Flow:
 * ```
 * WordPress Hook (e.g., user_register)
 *     ↓
 * Bootstrap.php handler (e.g., handle_user_registered)
 *     ↓
 * TriggerRegistry::fire('wp_user_registered', $subscriber_id, $context)
 *     ↓
 * Find active automations with trigger_type = 'wp_user_registered'
 *     ↓
 * For each automation: run matcher($config, $context)
 *     ↓
 * If matched: AutomationEngine::enqueueSubscriber()
 *     ↓
 * Cron processes queue and executes automation steps
 * ```
 *
 * ============================================================================
 * AVAILABLE TRIGGERS (CORE)
 * ============================================================================
 *
 * SUBSCRIBER GROUP:
 * - subscriber_created       : New subscriber added to the system
 * - subscriber_joins_list    : Subscriber added to a specific list
 * - subscriber_leaves_list   : Subscriber removed from a list
 * - subscriber_added_tag     : Tag added to a subscriber
 * - subscriber_removed_tag   : Tag removed from a subscriber
 *
 * POST SUBSCRIPTIONS GROUP:
 * - post_subscription_created : Someone subscribes to post/page updates
 *
 * WORDPRESS GROUP:
 * - wp_user_registered       : New WordPress user signs up
 * - wp_user_login            : WordPress user logs in (with inactivity filter)
 *
 * CONTENT GROUP:
 * - post_published           : New post/page/CPT is published
 * - post_updated             : Existing post/page/CPT is updated
 *
 * THIRD-PARTY (registered via wp_outreach_register_triggers hook):
 * - wpdm_email_lock_submitted : WPDM email lock form submitted
 * - (any custom trigger from plugins)
 *
 * ============================================================================
 * CONFIGURATION FIELDS
 * ============================================================================
 *
 * Each trigger can have config_fields that allow users to customize when
 * the trigger should fire. Common field types:
 *
 * - select           : Dropdown with single selection
 * - multi_select     : Dropdown with multiple selections
 * - text             : Text input
 * - number           : Numeric input
 * - checkbox         : Boolean toggle
 *
 * Field properties:
 * - key              : Unique identifier for the field
 * - label            : Display label in UI
 * - type             : Field type (select, multi_select, text, etc.)
 * - placeholder      : Placeholder text (e.g., "Any list")
 * - help_text        : Additional help text below the field
 * - options          : Static array of options for select fields
 * - options_endpoint : REST API endpoint to fetch dynamic options
 * - default          : Default value
 * - required         : Whether the field is required
 *
 * ============================================================================
 * MATCHER FUNCTIONS
 * ============================================================================
 *
 * Each trigger can have a custom "matcher" function that determines whether
 * the trigger configuration matches the event context.
 *
 * Matcher signature:
 * ```php
 * function($config, $context) : bool
 * ```
 *
 * - $config : The automation's trigger configuration (user settings)
 * - $context : Event context data passed when trigger is fired
 * - Return true to fire the automation, false to skip
 *
 * Example:
 * ```php
 * 'matcher' => function ($config, $context) {
 *     // Only fire if the list_id in config matches the event's list_id
 *     if (empty($config['list_id'])) {
 *         return true; // No filter = match any list
 *     }
 *     return (int) $config['list_id'] === (int) ($context['list_id'] ?? 0);
 * },
 * ```
 *
 * ============================================================================
 * CONTEXT DATA
 * ============================================================================
 *
 * Context is the data passed when a trigger fires. It contains:
 * - Event-specific data (e.g., list_id, tag_id, post_id)
 * - Subscriber data (email, first_name, last_name)
 * - Additional metadata for template variables
 *
 * Context data is available in automation actions as merge tags:
 * - {{first_name}}, {{last_name}}, {{email}}
 * - {{post_title}}, {{post_url}} (for content triggers)
 * - Any custom context keys passed
 *
 * ============================================================================
 * THIRD-PARTY INTEGRATION
 * ============================================================================
 *
 * To register a custom trigger from your plugin:
 *
 * ```php
 * add_action('wp_outreach_register_triggers', function($registry) {
 *     $registry->register([
 *         'id'          => 'my_plugin_custom_event',
 *         'name'        => 'Custom Event Triggered',
 *         'description' => 'Fires when something happens in my plugin',
 *         'group'       => 'My Plugin',
 *         'icon'        => 'star', // Lucide icon name
 *         'config_fields' => [
 *             [
 *                 'key'   => 'event_type',
 *                 'label' => 'Event Type',
 *                 'type'  => 'select',
 *                 'options' => [
 *                     ['value' => 'created', 'label' => 'Created'],
 *                     ['value' => 'updated', 'label' => 'Updated'],
 *                 ],
 *             ],
 *         ],
 *         'matcher' => function ($config, $context) {
 *             if (empty($config['event_type'])) return true;
 *             return $config['event_type'] === $context['event_type'];
 *         },
 *     ]);
 * });
 * ```
 *
 * Then fire the trigger when your event occurs:
 *
 * ```php
 * // First, ensure you have a subscriber ID (or create one)
 * $subscriber_id = wp_outreach_get_or_create_subscriber($email, [
 *     'first_name' => 'John',
 *     'last_name' => 'Doe',
 * ]);
 *
 * // Fire the trigger with context data
 * do_action('wp_outreach_trigger', 'my_plugin_custom_event', $subscriber_id, [
 *     'event_type'  => 'created',
 *     'custom_data' => $your_data,
 *     'first_name'  => 'John', // Available as {{first_name}} in emails
 * ]);
 * ```
 *
 * ============================================================================
 * WORDPRESS HOOKS
 * ============================================================================
 *
 * Actions:
 * - wp_outreach_register_triggers     : Register custom triggers
 * - wp_outreach_core_triggers_registered : After core triggers registered
 * - wp_outreach_trigger               : Fire a trigger (shorthand)
 * - wp_outreach_trigger_not_found     : When trigger ID doesn't exist
 * - wp_outreach_automation_triggered  : After automation is triggered
 *
 * Usage:
 * ```php
 * // Fire trigger via action hook
 * do_action('wp_outreach_trigger', 'trigger_id', $subscriber_id, $context);
 *
 * // Or call directly
 * TriggerRegistry::fire('trigger_id', $subscriber_id, $context);
 * ```
 */
class TriggerRegistry
{
    /**
     * Registered triggers
     *
     * Stored as associative array: trigger_id => trigger_config
     *
     * @var array<string, array>
     */
    private static array $triggers = [];

    /**
     * Whether triggers have been initialized
     *
     * @var bool
     */
    private static bool $initialized = false;

    /**
     * Initialize the registry and register core triggers
     *
     * This method is called automatically when accessing triggers.
     * It registers all core triggers and fires the hook for third-party
     * plugins to register their own triggers.
     *
     * @return void
     */
    public static function init(): void
    {
        if (self::$initialized) {
            return;
        }

        self::$initialized = true;

        // Register core triggers
        self::registerCoreTriggers();

        // Allow third-party plugins to register their triggers
        // Plugins should hook into this to add custom triggers
        do_action('wp_outreach_register_triggers', new self());
    }

    /**
     * Register a new trigger
     *
     * @param array $trigger {
     *     Trigger configuration array.
     *
     *     @type string   $id            Required. Unique trigger identifier (e.g., 'wpdm_package_updated').
     *                                   Should be snake_case and prefixed with plugin name for third-party.
     *     @type string   $name          Required. Human-readable trigger name shown in UI.
     *     @type string   $description   Optional. Detailed description explaining when trigger fires.
     *     @type string   $group         Optional. Group name for UI organization. Default: 'General'.
     *     @type string   $icon          Optional. Lucide icon name (e.g., 'user-plus', 'mail', 'tag').
     *                                   See https://lucide.dev/icons for available icons. Default: 'zap'.
     *     @type array    $config_fields Optional. Configuration fields array. See class docs for structure.
     *     @type callable $matcher       Optional. Custom matcher function: fn($config, $context) => bool.
     *                                   If not provided, uses default key matching.
     *     @type string   $source        Optional. 'core' for built-in, 'third-party' for plugins.
     *     @type bool     $available     Optional. Whether trigger is currently available. Default: true.
     *     @type string   $requires      Optional. Plugin/feature required (shown if unavailable).
     *     @type bool     $is_content_trigger Optional. True for post_published/post_updated to show
     *                                   recipient selection options in send_email action.
     * }
     * @return bool True on success, false if trigger ID is empty or already exists.
     *
     * @example
     * ```php
     * $registry->register([
     *     'id'          => 'my_custom_trigger',
     *     'name'        => 'My Custom Event',
     *     'description' => 'Fires when...',
     *     'group'       => 'My Plugin',
     *     'icon'        => 'star',
     *     'config_fields' => [...],
     *     'matcher'     => function($config, $context) { return true; },
     * ]);
     * ```
     */
    public function register(array $trigger): bool
    {
        if (empty($trigger['id'])) {
            return false;
        }

        $id = sanitize_key($trigger['id']);

        if (isset(self::$triggers[$id])) {
            return false; // Already registered
        }

        self::$triggers[$id] = [
            'id'            => $id,
            'name'          => $trigger['name'] ?? $id,
            'description'   => $trigger['description'] ?? '',
            'group'         => $trigger['group'] ?? 'General',
            'icon'          => $trigger['icon'] ?? 'zap',
            'config_fields' => $trigger['config_fields'] ?? [],
            'matcher'       => $trigger['matcher'] ?? null,
            'source'        => $trigger['source'] ?? 'third-party',
            'available'     => $trigger['available'] ?? true,
            'requires'      => $trigger['requires'] ?? null,
        ];

        return true;
    }

    /**
     * Unregister a trigger
     *
     * @param string $id Trigger ID to remove
     * @return bool True if removed, false if not found.
     */
    public function unregister(string $id): bool
    {
        if (!isset(self::$triggers[$id])) {
            return false;
        }

        unset(self::$triggers[$id]);
        return true;
    }

    /**
     * Get all registered triggers
     *
     * @return array<string, array> All triggers keyed by ID
     */
    public static function getAll(): array
    {
        self::init();
        return self::$triggers;
    }

    /**
     * Get triggers grouped by their group name
     *
     * @return array<string, array> Triggers grouped by group name
     */
    public static function getGrouped(): array
    {
        self::init();
        $grouped = [];

        foreach (self::$triggers as $trigger) {
            $group = $trigger['group'];
            if (!isset($grouped[$group])) {
                $grouped[$group] = [];
            }
            $grouped[$group][] = $trigger;
        }

        return $grouped;
    }

    /**
     * Get a single trigger by ID
     *
     * @param string $id Trigger ID
     * @return array|null Trigger data or null if not found.
     */
    public static function get(string $id): ?array
    {
        self::init();
        return self::$triggers[$id] ?? null;
    }

    /**
     * Check if a trigger exists
     *
     * @param string $id Trigger ID
     * @return bool
     */
    public static function exists(string $id): bool
    {
        self::init();
        return isset(self::$triggers[$id]);
    }

    /**
     * Get triggers formatted for REST API response
     *
     * Returns triggers grouped and formatted for the frontend UI.
     *
     * @return array API-ready trigger data
     */
    public static function forApi(): array
    {
        self::init();
        $grouped = self::getGrouped();
        $result = [];

        foreach ($grouped as $group => $triggers) {
            $result[] = [
                'group' => $group,
                'triggers' => array_map(function ($trigger) {
                    // Check license for trigger availability
                    $available = \WPOutreach\License\FeatureManager::canUseTrigger($trigger['id']);

                    return [
                        'id'                 => $trigger['id'],
                        'name'               => $trigger['name'],
                        'description'        => $trigger['description'],
                        'icon'               => $trigger['icon'],
                        'config_fields'      => $trigger['config_fields'],
                        'available'          => $available,
                        'requires'           => $trigger['requires'] ?? null,
                        'is_content_trigger' => $trigger['is_content_trigger'] ?? false,
                        'is_pro'             => !$available && !\WPOutreach\License\FeatureManager::hasValidLicense(),
                    ];
                }, $triggers),
            ];
        }

        return $result;
    }

    /**
     * Register core WP Outreach triggers
     *
     * This method registers all built-in triggers. Each trigger is documented
     * with its purpose, configuration options, context data, and example use cases.
     *
     * @return void
     */
    private static function registerCoreTriggers(): void
    {
        $registry = new self();

        // ====================================================================
        // SUBSCRIBER GROUP - Triggers based on subscriber actions
        // ====================================================================

        /**
         * TRIGGER: subscriber_created
         * ============================
         *
         * Fires when a new subscriber is added to the system through any method:
         * - Public subscription form
         * - Admin manual add
         * - CSV import
         * - API creation
         * - WordPress user registration (auto-creates subscriber)
         *
         * Configuration Options: None
         *
         * Context Data:
         * - email      : Subscriber email address
         * - first_name : First name (may be empty)
         * - last_name  : Last name (may be empty)
         * - source     : How they subscribed ('form', 'admin', 'import', 'api', 'wp_user')
         * - list_ids   : Array of list IDs they were added to
         *
         * WordPress Hook: wp_outreach_subscriber_created
         *
         * Example Use Cases:
         * - Welcome email sequence
         * - Add "new-subscriber" tag
         * - Notify admin of new signup
         *
         * Example:
         * ```php
         * do_action('wp_outreach_subscriber_created', $subscriber_id, [
         *     'email' => 'john@example.com',
         *     'first_name' => 'John',
         *     'source' => 'form',
         * ]);
         * ```
         */
        $registry->register([
            'id'          => 'subscriber_created',
            'name'        => __('Subscriber Created', 'outreach'),
            'description' => __('Fires when a new subscriber is added', 'outreach'),
            'group'       => __('Subscriber', 'outreach'),
            'icon'        => 'user-plus',
            'source'      => 'core',
            // No config_fields - triggers for ALL new subscribers
            // No matcher needed - always matches
        ]);

        /**
         * TRIGGER: subscriber_joins_list
         * ================================
         *
         * Fires when a subscriber is added to a specific list.
         * Can be configured to fire for a specific list or any list.
         *
         * Configuration Options:
         * - list_id : Select a specific list, or leave empty for "any list"
         *
         * Context Data:
         * - list_id   : The list ID they were added to
         * - list_name : The list name
         * - email     : Subscriber email
         * - first_name, last_name : Subscriber name
         *
         * WordPress Hook: wp_outreach_subscriber_added_to_list
         *
         * Matcher Logic:
         * - If list_id is empty in config → matches any list
         * - If list_id is set → only matches that specific list
         *
         * Example Use Cases:
         * - Send specific welcome for "VIP" list
         * - Add tag when joining "Product Updates" list
         * - Different onboarding for different lists
         *
         * Example:
         * ```php
         * do_action('wp_outreach_subscriber_added_to_list', $subscriber_id, [
         *     'list_id' => 5,
         *     'list_name' => 'Newsletter',
         * ]);
         * ```
         */
        $registry->register([
            'id'          => 'subscriber_joins_list',
            'name'        => __('Subscriber Joins List', 'outreach'),
            'description' => __('Fires when a subscriber is added to a list', 'outreach'),
            'group'       => __('Subscriber', 'outreach'),
            'icon'        => 'users',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'list_id',
                    'label'       => __('List', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any list', 'outreach'),
                    'options_endpoint' => '/lists', // Fetches from /wp-json/wp-outreach/v1/lists
                ],
            ],
            'matcher' => function ($config, $context) {
                // If no list_id configured, match any list
                if (empty($config['list_id'])) {
                    return true;
                }
                // Otherwise, match only the configured list
                return (int) $config['list_id'] === (int) ($context['list_id'] ?? 0);
            },
        ]);

        /**
         * TRIGGER: subscriber_added_tag
         * ===============================
         *
         * Fires when a tag is added to a subscriber.
         * Tags are used for segmentation and can trigger specific workflows.
         *
         * Configuration Options:
         * - tag_id : Select a specific tag, or leave empty for "any tag"
         *
         * Context Data:
         * - tag_id   : The tag ID that was added
         * - tag_name : The tag name
         * - email    : Subscriber email
         * - first_name, last_name : Subscriber name
         *
         * WordPress Hook: wp_outreach_tag_added_to_subscriber
         *
         * Matcher Logic:
         * - If tag_id is empty in config → matches any tag
         * - If tag_id is set → only matches that specific tag
         *
         * Example Use Cases:
         * - Send specific email when "vip" tag is added
         * - Start re-engagement when "inactive" tag is added
         * - Send product info when "interested-in-product-x" tag is added
         *
         * Example:
         * ```php
         * do_action('wp_outreach_tag_added_to_subscriber', $subscriber_id, $tag_id, [
         *     'tag_id' => 3,
         *     'tag_name' => 'VIP',
         * ]);
         * ```
         */
        $registry->register([
            'id'          => 'subscriber_added_tag',
            'name'        => __('Tag Added to Subscriber', 'outreach'),
            'description' => __('Fires when a tag is added to a subscriber', 'outreach'),
            'group'       => __('Subscriber', 'outreach'),
            'icon'        => 'tag',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'tag_id',
                    'label'       => __('Tag', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any tag', 'outreach'),
                    'options_endpoint' => '/tags', // Fetches from /wp-json/wp-outreach/v1/tags
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['tag_id'])) {
                    return true; // Any tag
                }
                return (int) $config['tag_id'] === (int) ($context['tag_id'] ?? 0);
            },
        ]);

        /**
         * TRIGGER: subscriber_removed_tag
         * =================================
         *
         * Fires when a tag is removed from a subscriber.
         * Useful for cleanup workflows or status change notifications.
         *
         * Configuration Options:
         * - tag_id : Select a specific tag, or leave empty for "any tag"
         *
         * Context Data:
         * - tag_id   : The tag ID that was removed
         * - tag_name : The tag name
         * - email    : Subscriber email
         * - first_name, last_name : Subscriber name
         *
         * WordPress Hook: wp_outreach_tag_removed_from_subscriber
         *
         * Example Use Cases:
         * - Send "sorry to see you go" when "active-customer" tag removed
         * - Trigger win-back campaign when "engaged" tag removed
         * - Notify admin when "vip" status is removed
         */
        $registry->register([
            'id'          => 'subscriber_removed_tag',
            'name'        => __('Tag Removed from Subscriber', 'outreach'),
            'description' => __('Fires when a tag is removed from a subscriber', 'outreach'),
            'group'       => __('Subscriber', 'outreach'),
            'icon'        => 'tag',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'tag_id',
                    'label'       => __('Tag', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any tag', 'outreach'),
                    'options_endpoint' => '/tags',
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['tag_id'])) {
                    return true; // Any tag
                }
                return (int) $config['tag_id'] === (int) ($context['tag_id'] ?? 0);
            },
        ]);

        /**
         * TRIGGER: subscriber_leaves_list
         * =================================
         *
         * Fires when a subscriber is removed from a list.
         * This happens via admin action, unsubscribe, or API.
         *
         * Configuration Options:
         * - list_id : Select a specific list, or leave empty for "any list"
         *
         * Context Data:
         * - list_id   : The list ID they left
         * - list_name : The list name
         * - email     : Subscriber email
         * - reason    : Why they left ('admin', 'unsubscribe', 'api')
         *
         * WordPress Hook: wp_outreach_subscriber_removed_from_list
         *
         * Example Use Cases:
         * - Send "we'll miss you" email
         * - Remove related tags
         * - Trigger win-back sequence after delay
         */
        $registry->register([
            'id'          => 'subscriber_leaves_list',
            'name'        => __('Subscriber Leaves List', 'outreach'),
            'description' => __('Fires when a subscriber is removed from a list', 'outreach'),
            'group'       => __('Subscriber', 'outreach'),
            'icon'        => 'user-minus',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'list_id',
                    'label'       => __('List', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any list', 'outreach'),
                    'options_endpoint' => '/lists',
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['list_id'])) {
                    return true;
                }
                return (int) $config['list_id'] === (int) ($context['list_id'] ?? 0);
            },
        ]);

        // ====================================================================
        // POST SUBSCRIPTIONS GROUP - Triggers for post/page subscription system
        // ====================================================================

        /**
         * TRIGGER: post_subscription_created
         * ====================================
         *
         * Fires when someone subscribes to receive updates for a specific post or page.
         * This is different from newsletter subscription - it's content-specific.
         *
         * Configuration Options:
         * - post_type : Filter by post type (post, page, or custom post type)
         *
         * Context Data:
         * - post_id         : The post ID they subscribed to
         * - post_type       : The post type (post, page, etc.)
         * - post_title      : The post title
         * - post_url        : URL to the post
         * - subscription_type : 'new_only', 'updates_only', or 'both'
         * - email           : Subscriber email
         * - first_name, last_name : Subscriber name
         *
         * WordPress Hook: wp_outreach_post_subscription_created
         *
         * Example Use Cases:
         * - Send confirmation of subscription
         * - Add subscriber to related newsletter list
         * - Tag subscriber with post category
         *
         * Note: Post update notifications are handled separately by
         * PostSubscriptionHandler which can auto-notify or use automations.
         */
        $registry->register([
            'id'          => 'post_subscription_created',
            'name'        => __('Post Subscription Created', 'outreach'),
            'description' => __('Fires when someone subscribes to a post/page for updates', 'outreach'),
            'group'       => __('Post Subscriptions', 'outreach'),
            'icon'        => 'bell',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'post_type',
                    'label'       => __('Post Type', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any post type', 'outreach'),
                    'options_endpoint' => '/settings/post-types', // Enabled post types
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['post_type'])) {
                    return true; // Any post type
                }
                return $config['post_type'] === ($context['post_type'] ?? '');
            },
        ]);

        // ====================================================================
        // WORDPRESS GROUP - Triggers based on WordPress user actions
        // ====================================================================

        /**
         * TRIGGER: wp_user_registered
         * =============================
         *
         * Fires when a new WordPress user account is created.
         * Automatically creates a subscriber from the WP user data.
         *
         * Configuration Options:
         * - user_roles : Multi-select to filter by user role(s)
         *                Leave empty to trigger for any role
         *
         * Context Data:
         * - user_id      : WordPress user ID
         * - user_email   : User's email address
         * - user_login   : Username
         * - display_name : Display name
         * - first_name   : First name from user meta
         * - last_name    : Last name from user meta
         * - user_role    : Primary role (first role in array)
         * - user_roles   : Array of all user roles
         *
         * WordPress Hook: user_register
         * Handler: Bootstrap::handle_user_registered()
         *
         * Matcher Logic:
         * - If user_roles is empty in config → matches any role
         * - If user_roles is set → only matches if user has one of those roles
         *
         * Example Use Cases:
         * - Welcome email for new members
         * - Add to "Members" list based on role
         * - Send admin notification for new registrations
         * - Different onboarding for different roles (customer vs author)
         *
         * Example:
         * ```php
         * // Fired automatically by Bootstrap when user registers
         * TriggerRegistry::fire('wp_user_registered', $subscriber_id, [
         *     'user_id' => 123,
         *     'user_email' => 'john@example.com',
         *     'user_role' => 'subscriber',
         *     'first_name' => 'John',
         * ]);
         * ```
         */
        $registry->register([
            'id'          => 'wp_user_registered',
            'name'        => __('WordPress User Registered', 'outreach'),
            'description' => __('Fires when a new WordPress user signs up', 'outreach'),
            'group'       => __('WordPress', 'outreach'),
            'icon'        => 'user-plus',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'user_roles',
                    'label'       => __('User Roles', 'outreach'),
                    'type'        => 'multi_select',
                    'placeholder' => __('Any role', 'outreach'),
                    'help_text'   => __('Leave empty to trigger for any user role', 'outreach'),
                    'options_endpoint' => '/user-roles', // Fetches WP roles
                ],
            ],
            'matcher' => function ($config, $context) {
                // Check user role filter
                if (!empty($config['user_roles'])) {
                    $allowed_roles = is_array($config['user_roles']) ? $config['user_roles'] : [$config['user_roles']];
                    $user_role = $context['user_role'] ?? '';
                    if (!in_array($user_role, $allowed_roles)) {
                        return false;
                    }
                }
                return true;
            },
        ]);

        /**
         * TRIGGER: wp_user_login
         * =======================
         *
         * Fires when a WordPress user logs into the site.
         * Includes an "inactivity filter" to only trigger for users who
         * haven't logged in for a specified number of days.
         *
         * This is useful for re-engagement campaigns - you don't want to
         * email users every time they log in, only when they return after
         * a period of inactivity.
         *
         * Configuration Options:
         * - min_days_since_last_login : Only trigger if user hasn't logged in
         *                               for this many days (7, 14, 30, 60, 90)
         *                               Leave empty to trigger on every login
         * - user_roles : Multi-select to filter by user role(s)
         *
         * Context Data:
         * - user_id               : WordPress user ID
         * - user_email            : User's email address
         * - user_login            : Username
         * - display_name          : Display name
         * - first_name            : First name
         * - last_name             : Last name
         * - user_role             : Primary role
         * - user_roles            : Array of all roles
         * - days_since_last_login : Number of days since previous login
         *                           (PHP_INT_MAX for first login ever)
         * - last_login_at         : Previous login timestamp (null if first login)
         *
         * WordPress Hook: wp_login
         * Handler: Bootstrap::handle_user_login()
         *
         * Login Tracking:
         * - Stores last login time in user meta: wp_outreach_last_login
         * - Updates BEFORE firing trigger so next login has correct data
         *
         * Matcher Logic:
         * 1. If min_days_since_last_login is set:
         *    - Calculate days since last login
         *    - Only fire if days >= min_days
         * 2. If user_roles is set:
         *    - Only fire if user has one of the allowed roles
         *
         * Example Use Cases:
         * - "Welcome back!" email after 30+ days of inactivity
         * - Re-engagement campaign for users inactive 90+ days
         * - Track active users by tagging on each login
         * - Different messaging for returning customers vs admins
         *
         * Example:
         * ```php
         * // Fired automatically by Bootstrap on wp_login hook
         * TriggerRegistry::fire('wp_user_login', $subscriber_id, [
         *     'user_id' => 123,
         *     'user_email' => 'john@example.com',
         *     'user_role' => 'customer',
         *     'days_since_last_login' => 45,
         *     'last_login_at' => '2024-11-15 10:30:00',
         * ]);
         * ```
         */
        $registry->register([
            'id'          => 'wp_user_login',
            'name'        => __('WordPress User Login', 'outreach'),
            'description' => __('Fires when a WordPress user logs in (with inactivity filter)', 'outreach'),
            'group'       => __('WordPress', 'outreach'),
            'icon'        => 'log-in',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'min_days_since_last_login',
                    'label'       => __('Minimum Days Since Last Login', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Always trigger (no minimum)', 'outreach'),
                    'help_text'   => __('Only trigger if user hasn\'t logged in for this many days', 'outreach'),
                    'options'     => [
                        ['value' => '7', 'label' => '7 days'],
                        ['value' => '14', 'label' => '14 days'],
                        ['value' => '30', 'label' => '30 days'],
                        ['value' => '60', 'label' => '60 days'],
                        ['value' => '90', 'label' => '90 days'],
                    ],
                ],
                [
                    'key'         => 'user_roles',
                    'label'       => __('User Roles', 'outreach'),
                    'type'        => 'multi_select',
                    'placeholder' => __('Any role', 'outreach'),
                    'help_text'   => __('Leave empty to trigger for any user role', 'outreach'),
                    'options_endpoint' => '/user-roles',
                ],
            ],
            'matcher' => function ($config, $context) {
                // Check minimum days since last login filter
                if (!empty($config['min_days_since_last_login'])) {
                    $min_days = (int) $config['min_days_since_last_login'];
                    $days_since_last_login = $context['days_since_last_login'] ?? 0;
                    if ($days_since_last_login < $min_days) {
                        return false; // User logged in too recently
                    }
                }

                // Check user role filter
                if (!empty($config['user_roles'])) {
                    $allowed_roles = is_array($config['user_roles']) ? $config['user_roles'] : [$config['user_roles']];
                    $user_role = $context['user_role'] ?? '';
                    if (!in_array($user_role, $allowed_roles)) {
                        return false; // User doesn't have required role
                    }
                }
                return true;
            },
        ]);

        // ====================================================================
        // CONTENT GROUP - Triggers based on post/page content changes
        // ====================================================================

        /**
         * TRIGGER: post_published
         * =========================
         *
         * Fires when a NEW post, page, or custom post type is published.
         * Only fires on initial publication, not on updates.
         *
         * This is a "content trigger" - it's not tied to a single subscriber.
         * Instead, you configure recipient selection in the send_email action:
         * - All subscribers in selected lists
         * - Post subscribers (for post subscription system)
         *
         * Configuration Options:
         * - post_type : Filter by post type (post, page, custom)
         *               Leave empty for any post type
         *
         * Context Data:
         * - post_id        : The published post ID
         * - post_type      : Post type (post, page, product, etc.)
         * - post_title     : Post title
         * - post_url       : URL to the post
         * - post_excerpt   : Post excerpt or auto-generated summary
         * - post_author_id : Author's user ID
         * - post_author    : Author's display name
         * - post_date      : Publication date
         * - post_categories : Array of category names (if applicable)
         * - post_tags      : Array of tag names (if applicable)
         * - featured_image : Featured image URL (if set)
         *
         * WordPress Hook: transition_post_status (when status → 'publish')
         * Handler: ContentTriggerHandler::handlePostPublished()
         *
         * Special Flag:
         * is_content_trigger: true
         * - Tells frontend to show recipient selection options in send_email
         *
         * Example Use Cases:
         * - Notify newsletter subscribers of new blog posts
         * - Alert customers of new products (WooCommerce)
         * - Send to category-specific lists
         *
         * Note: The subscriber_id passed is 0 for content triggers.
         * The actual recipients are determined by the send_email action config.
         */
        $registry->register([
            'id'          => 'post_published',
            'name'        => __('Post Published', 'outreach'),
            'description' => __('Fires when a new post/page/CPT is published', 'outreach'),
            'group'       => __('Content', 'outreach'),
            'icon'        => 'file-text',
            'source'      => 'core',
            'is_content_trigger' => true, // Flag for frontend recipient selection
            'config_fields' => [
                [
                    'key'         => 'post_type',
                    'label'       => __('Post Type', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any post type', 'outreach'),
                    'options_endpoint' => '/post-types', // All public post types
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['post_type'])) {
                    return true; // Any post type
                }
                return $config['post_type'] === ($context['post_type'] ?? '');
            },
        ]);

        /**
         * TRIGGER: post_updated
         * =======================
         *
         * Fires when an EXISTING published post is updated.
         * Only fires when an already-published post is saved again.
         *
         * This is a "content trigger" with the same recipient selection
         * options as post_published.
         *
         * Configuration Options:
         * - post_type : Filter by post type
         *
         * Context Data:
         * - post_id        : The updated post ID
         * - post_type      : Post type
         * - post_title     : Post title (may have changed)
         * - post_url       : URL to the post
         * - post_excerpt   : Post excerpt
         * - post_author_id : Author's user ID
         * - post_author    : Author's display name
         * - post_modified  : Last modified date
         * - featured_image : Featured image URL
         *
         * WordPress Hook: post_updated
         * Handler: ContentTriggerHandler::handlePostUpdated()
         *
         * Example Use Cases:
         * - Notify post subscribers of content updates
         * - Alert team when documentation is updated
         * - Re-engage readers with "we've updated this post"
         *
         * Tip: Use with "wait" action to avoid spamming on multiple quick edits:
         * 1. post_updated trigger
         * 2. Wait 1 hour
         * 3. Send email (if post still matches)
         */
        $registry->register([
            'id'          => 'post_updated',
            'name'        => __('Post Updated', 'outreach'),
            'description' => __('Fires when a post/page/CPT is updated', 'outreach'),
            'group'       => __('Content', 'outreach'),
            'icon'        => 'edit',
            'source'      => 'core',
            'is_content_trigger' => true,
            'config_fields' => [
                [
                    'key'         => 'post_type',
                    'label'       => __('Post Type', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any post type', 'outreach'),
                    'options_endpoint' => '/post-types',
                ],
            ],
            'matcher' => function ($config, $context) {
                if (empty($config['post_type'])) {
                    return true; // Any post type
                }
                return $config['post_type'] === ($context['post_type'] ?? '');
            },
        ]);

        // ====================================================================
        // EMAIL ENGAGEMENT GROUP - Triggers based on email interactions
        // ====================================================================

        /**
         * TRIGGER: email_opened
         * =======================
         *
         * Fires when a subscriber opens an email (first unique open only).
         * This trigger requires open tracking to be enabled in settings.
         *
         * Configuration Options:
         * - campaign_id : Filter by specific campaign (optional)
         *                 Leave empty to trigger for any email opened
         *
         * Context Data:
         * - campaign_id    : The campaign ID (if from a campaign)
         * - automation_id  : The automation ID (if from an automation)
         * - email          : Subscriber's email address
         * - subject        : Email subject line
         * - tracking_id    : Unique tracking ID
         * - first_name     : Subscriber first name
         * - last_name      : Subscriber last name
         * - opened_at      : When the email was opened
         * - sent_at        : When the email was originally sent
         *
         * WordPress Hook: wp_outreach_email_opened
         *
         * Note: Only fires on the FIRST unique open, not subsequent opens.
         * Total opens are tracked separately in the logs table.
         *
         * Example Use Cases:
         * - Tag engaged subscribers
         * - Trigger follow-up sequence after open
         * - Move subscriber to "engaged" list
         * - Score subscribers based on engagement
         */
        $registry->register([
            'id'          => 'email_opened',
            'name'        => __('Email Opened', 'outreach'),
            'description' => __('Fires when a subscriber opens an email', 'outreach'),
            'group'       => __('Email Engagement', 'outreach'),
            'icon'        => 'mail-open',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'campaign_id',
                    'label'       => __('Campaign', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any email', 'outreach'),
                    'help_text'   => __('Trigger only when emails from this campaign are opened', 'outreach'),
                    'options_endpoint' => '/campaigns?status=sent', // Only sent campaigns
                ],
            ],
            'matcher' => function ($config, $context) {
                // If no campaign_id configured, match any email
                if (empty($config['campaign_id'])) {
                    return true;
                }
                // Match only the configured campaign
                return (int) $config['campaign_id'] === (int) ($context['campaign_id'] ?? 0);
            },
        ]);

        /**
         * TRIGGER: email_clicked
         * ========================
         *
         * Fires when a subscriber clicks a link in an email (first click only).
         * This trigger requires click tracking to be enabled in settings.
         *
         * Configuration Options:
         * - campaign_id : Filter by specific campaign (optional)
         *                 Leave empty to trigger for any email click
         *
         * Context Data:
         * - campaign_id    : The campaign ID (if from a campaign)
         * - automation_id  : The automation ID (if from an automation)
         * - email          : Subscriber's email address
         * - subject        : Email subject line
         * - tracking_id    : Unique tracking ID
         * - first_name     : Subscriber first name
         * - last_name      : Subscriber last name
         * - clicked_url    : The URL that was clicked
         * - clicked_at     : When the link was clicked
         * - sent_at        : When the email was originally sent
         *
         * WordPress Hook: wp_outreach_email_clicked
         *
         * Note: Only fires on the FIRST click by the subscriber, not subsequent clicks.
         * This is to avoid triggering automations multiple times per email.
         *
         * Example Use Cases:
         * - Tag subscribers interested in specific products
         * - Trigger product-specific follow-up sequence
         * - Move to "hot lead" list based on CTA clicks
         * - Score engagement based on clicked content
         */
        $registry->register([
            'id'          => 'email_clicked',
            'name'        => __('Email Clicked', 'outreach'),
            'description' => __('Fires when a subscriber clicks a link in an email', 'outreach'),
            'group'       => __('Email Engagement', 'outreach'),
            'icon'        => 'mouse-pointer-click',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'campaign_id',
                    'label'       => __('Campaign', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any email', 'outreach'),
                    'help_text'   => __('Trigger only when links in emails from this campaign are clicked', 'outreach'),
                    'options_endpoint' => '/campaigns?status=sent',
                ],
            ],
            'matcher' => function ($config, $context) {
                // If no campaign_id configured, match any click
                if (empty($config['campaign_id'])) {
                    return true;
                }
                // Match only the configured campaign
                return (int) $config['campaign_id'] === (int) ($context['campaign_id'] ?? 0);
            },
        ]);

        /**
         * TRIGGER: email_not_opened
         * ===========================
         *
         * Fires when a subscriber has NOT opened an email after a specified
         * number of days. This is useful for re-engagement and win-back campaigns.
         *
         * This trigger is processed by cron, not by a real-time hook. The system
         * checks for unopened emails during each cron run and fires the trigger
         * for emails that have exceeded the configured wait period.
         *
         * Configuration Options:
         * - days         : Number of days to wait before triggering (required)
         *                  Options: 1, 2, 3, 5, 7, 14, 30 days
         * - campaign_id  : Filter by specific campaign (optional)
         *
         * Context Data:
         * - campaign_id    : The campaign ID (if from a campaign)
         * - automation_id  : The automation ID (if from an automation)
         * - email          : Subscriber's email address
         * - subject        : Email subject line
         * - tracking_id    : Unique tracking ID
         * - first_name     : Subscriber first name
         * - last_name      : Subscriber last name
         * - sent_at        : When the email was originally sent
         * - days_unopened  : Number of days since email was sent
         *
         * Processing Logic:
         * 1. Cron job runs check_unopened_emails() every hour
         * 2. Finds log entries where:
         *    - opens = 0 (never opened)
         *    - sent_at >= X days ago
         *    - Not already triggered (checked via meta flag)
         * 3. For each match, fires this trigger and sets a flag to prevent re-triggering
         *
         * Example Use Cases:
         * - Resend email with different subject after 3 days
         * - Add "unengaged" tag after 7 days of no opens
         * - Start re-engagement automation after 14 days
         * - Remove from list after 30 days of inactivity
         *
         * Note: Each email can only trigger this once. The system marks
         * the log entry after triggering to prevent duplicate automations.
         */
        $registry->register([
            'id'          => 'email_not_opened',
            'name'        => __('Email Not Opened', 'outreach'),
            'description' => __('Fires when a subscriber hasn\'t opened an email after X days', 'outreach'),
            'group'       => __('Email Engagement', 'outreach'),
            'icon'        => 'mail-x',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'days',
                    'label'       => __('Days Without Opening', 'outreach'),
                    'type'        => 'select',
                    'required'    => true,
                    'placeholder' => __('Select days', 'outreach'),
                    'help_text'   => __('Trigger after this many days without opening the email', 'outreach'),
                    'options'     => [
                        ['value' => '1', 'label' => '1 day'],
                        ['value' => '2', 'label' => '2 days'],
                        ['value' => '3', 'label' => '3 days'],
                        ['value' => '5', 'label' => '5 days'],
                        ['value' => '7', 'label' => '7 days'],
                        ['value' => '14', 'label' => '14 days'],
                        ['value' => '30', 'label' => '30 days'],
                    ],
                ],
                [
                    'key'         => 'campaign_id',
                    'label'       => __('Campaign', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Any email', 'outreach'),
                    'help_text'   => __('Only check emails from this campaign', 'outreach'),
                    'options_endpoint' => '/campaigns?status=sent',
                ],
            ],
            'matcher' => function ($config, $context) {
                // Days is required for this trigger
                if (empty($config['days'])) {
                    return false;
                }

                // Check if the email has been unopened for enough days
                $required_days = (int) $config['days'];
                $days_unopened = (int) ($context['days_unopened'] ?? 0);

                if ($days_unopened < $required_days) {
                    return false;
                }

                // If campaign_id is set, check it matches
                if (!empty($config['campaign_id'])) {
                    return (int) $config['campaign_id'] === (int) ($context['campaign_id'] ?? 0);
                }

                return true;
            },
        ]);

        // ====================================================================
        // INCOMING WEBHOOKS GROUP - External system integration
        // ====================================================================

        /**
         * TRIGGER: incoming_webhook
         * ===========================
         *
         * Fires when an external system sends an HTTP POST request to the
         * automation's webhook endpoint. This allows integration with any
         * external service like Zapier, Make (Integromat), custom apps, etc.
         *
         * Endpoint:
         * POST /wp-json/outreach/v1/webhooks/incoming/{webhook_key}
         *
         * Note: Uses a unique random alphanumeric key instead of automation ID
         * for security (prevents enumeration). The key is auto-generated when
         * the trigger is selected and shown immediately in the configuration.
         *
         * Configuration Options:
         * - webhook_key             : Auto-generated unique URL key (16 chars)
         * - secret_key              : Auto-generated HMAC key for signature validation
         * - subscriber_email_field  : JSON path to email in payload (default: "email")
         * - create_if_not_found     : Auto-create subscriber if email not in system
         * - rate_limit              : Max requests per minute (default: 100)
         *
         * Security:
         * - Uses unique key in URL instead of sequential IDs (non-guessable)
         * - Validates HMAC-SHA256 signature from X-Webhook-Signature header
         * - Signature format: sha256=<hex_hmac>
         * - Rate limiting per automation
         * - Request size limit (1MB)
         *
         * Context Data:
         * - All payload fields are passed as context
         * - email       : Subscriber's email (extracted from payload)
         * - first_name  : First name if in payload
         * - last_name   : Last name if in payload
         * - webhook_*   : Any additional payload fields prefixed with webhook_
         *
         * WordPress Hook: wp_outreach_webhook_received (custom)
         *
         * Example Request:
         * ```
         * curl -X POST \
         *   -H "Content-Type: application/json" \
         *   -H "X-Webhook-Signature: sha256=<computed_hmac>" \
         *   -d '{"email": "user@example.com", "first_name": "John", "source": "zapier"}' \
         *   https://example.com/wp-json/outreach/v1/webhooks/incoming/AbC123xYz456qRsT
         * ```
         *
         * Example Use Cases:
         * - Trigger automation when Zapier sends data from another app
         * - Integrate with custom CRM or e-commerce system
         * - Receive events from external platforms
         * - Create subscribers from external form submissions
         */
        $registry->register([
            'id'          => 'incoming_webhook',
            'name'        => __('Incoming Webhook', 'outreach'),
            'description' => __('Fires when external system sends data via HTTP POST', 'outreach'),
            'group'       => __('Webhooks', 'outreach'),
            'icon'        => 'webhook',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'webhook_url',
                    'label'       => __('Webhook URL', 'outreach'),
                    'type'        => 'readonly',
                    'help_text'   => __('Send POST requests to this URL to trigger the automation', 'outreach'),
                ],
                [
                    'key'         => 'secret_key',
                    'label'       => __('Secret Key', 'outreach'),
                    'type'        => 'readonly',
                    'help_text'   => __('Use this key to sign your webhook requests (HMAC-SHA256)', 'outreach'),
                ],
                [
                    'key'         => 'require_signature',
                    'label'       => __('Require Signature Validation', 'outreach'),
                    'type'        => 'checkbox',
                    'default'     => false,
                    'help_text'   => __('When enabled, incoming requests must include a valid HMAC-SHA256 signature using the secret key above', 'outreach'),
                ],
                [
                    'key'         => 'signature_header',
                    'label'       => __('Signature Header Name', 'outreach'),
                    'type'        => 'text',
                    'placeholder' => 'X-Webhook-Signature',
                    'default'     => 'X-Webhook-Signature',
                    'help_text'   => __('HTTP header name containing the signature (e.g., X-Webhook-Signature, X-Wordfence-Signature)', 'outreach'),
                ],
                [
                    'key'         => 'subscriber_email_field',
                    'label'       => __('Email Field', 'outreach'),
                    'type'        => 'text',
                    'placeholder' => 'email',
                    'default'     => 'email',
                    'help_text'   => __('JSON path to the email address in payload (e.g., "email", "user.email", "data.contact.email")', 'outreach'),
                ],
                [
                    'key'         => 'create_if_not_found',
                    'label'       => __('Auto-create Subscriber', 'outreach'),
                    'type'        => 'checkbox',
                    'default'     => true,
                    'help_text'   => __('Automatically create a subscriber if email is not found in the system', 'outreach'),
                ],
                [
                    'key'         => 'rate_limit',
                    'label'       => __('Rate Limit', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('100 per minute', 'outreach'),
                    'default'     => '100',
                    'help_text'   => __('Maximum requests per minute for this webhook', 'outreach'),
                    'options'     => [
                        ['value' => '10', 'label' => '10 per minute'],
                        ['value' => '50', 'label' => '50 per minute'],
                        ['value' => '100', 'label' => '100 per minute'],
                        ['value' => '500', 'label' => '500 per minute'],
                        ['value' => '0', 'label' => 'Unlimited'],
                    ],
                ],
            ],
            // No matcher needed - webhook endpoint directly fires the trigger
            // for the specific automation it's configured for
        ]);

        // ====================================================================
        // Hook for additional core triggers
        // ====================================================================

        /**
         * Allow other parts of WP Outreach to register triggers after core.
         * This is used by integrations like WPDM to add their triggers.
         *
         * @param TriggerRegistry $registry The registry instance
         */
        do_action('wp_outreach_core_triggers_registered', $registry);
    }

    /**
     * Fire a trigger and enqueue matching automations
     *
     * This is the main method to fire a trigger. When called:
     * 1. Validates the trigger exists
     * 2. Finds all active automations using this trigger
     * 3. For each automation, runs the matcher function
     * 4. If matched, enqueues the subscriber into the automation
     *
     * Third-party plugins can use this directly or via the action hook:
     * ```php
     * // Direct call
     * TriggerRegistry::fire('my_trigger', $subscriber_id, $context);
     *
     * // Via action hook
     * do_action('wp_outreach_trigger', 'my_trigger', $subscriber_id, $context);
     * ```
     *
     * @param string $trigger_id   The trigger ID (e.g., 'subscriber_created').
     * @param int    $subscriber_id The subscriber ID to enqueue.
     *                              Use 0 for content triggers where recipients
     *                              are determined by the action configuration.
     * @param array  $context      Context data for trigger matching and template variables.
     *                              Include all data that might be needed for:
     *                              - Matcher function to filter automations
     *                              - Email merge tags ({{first_name}}, etc.)
     *                              - Webhook payloads
     * @return int Number of automations that were triggered.
     *
     * @example
     * ```php
     * // Simple trigger with subscriber
     * TriggerRegistry::fire('subscriber_created', $subscriber_id, [
     *     'email' => 'john@example.com',
     *     'first_name' => 'John',
     *     'source' => 'form',
     * ]);
     *
     * // Content trigger (subscriber_id = 0)
     * TriggerRegistry::fire('post_published', 0, [
     *     'post_id' => 123,
     *     'post_title' => 'New Blog Post',
     *     'post_url' => 'https://example.com/new-post',
     * ]);
     * ```
     */
    public static function fire(string $trigger_id, int $subscriber_id, array $context = []): int
    {
        self::init();

        // Validate trigger exists
        if (!self::exists($trigger_id)) {
            /**
             * Fires when an unregistered trigger is fired.
             *
             * @param string $trigger_id   The trigger ID that wasn't found
             * @param int    $subscriber_id The subscriber ID
             * @param array  $context      The context data
             */
            do_action('wp_outreach_trigger_not_found', $trigger_id, $subscriber_id, $context);
            return 0;
        }

        $trigger = self::get($trigger_id);
        $automations = self::getActiveAutomationsForTrigger($trigger_id);
        $count = 0;

        foreach ($automations as $automation) {
            // Parse the automation's trigger configuration
            $config = json_decode($automation->trigger_config, true) ?: [];

            // Determine if this automation should fire for this event
            $matches = true;
            if (is_callable($trigger['matcher'])) {
                // Use custom matcher function
                $matches = call_user_func($trigger['matcher'], $config, $context);
            } else {
                // Default matching: check all config keys against context
                // This is a fallback for simple triggers without custom matchers
                foreach ($config as $key => $value) {
                    // Skip name fields (list_name, tag_name, etc.) - they're for display only
                    if (str_ends_with($key, '_name') || str_ends_with($key, '_names')) {
                        continue;
                    }
                    if (!empty($value) && isset($context[$key]) && $value != $context[$key]) {
                        $matches = false;
                        break;
                    }
                }
            }

            // If matched, enqueue the subscriber into this automation
            if ($matches) {
                $result = AutomationEngine::enqueueSubscriber($automation->id, $subscriber_id, $context);
                if ($result) {
                    $count++;

                    /**
                     * Fires after an automation is successfully triggered.
                     *
                     * @param array $data {
                     *     @type int    $automation_id  The automation ID
                     *     @type int    $subscriber_id  The subscriber ID
                     *     @type string $trigger_id     The trigger that fired
                     *     @type array  $context        The event context
                     * }
                     */
                    do_action('wp_outreach_automation_triggered', [
                        'automation_id'  => $automation->id,
                        'subscriber_id'  => $subscriber_id,
                        'trigger_id'     => $trigger_id,
                        'context'        => $context,
                    ]);
                }
            }
        }

        return $count;
    }

    /**
     * Get active automations for a specific trigger type
     *
     * @param string $trigger_id The trigger ID to find automations for
     * @return array Array of automation objects from the database
     */
    private static function getActiveAutomationsForTrigger(string $trigger_id): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table} WHERE status = 'active' AND trigger_type = %s",
            $trigger_id
        ));
    }
}
