<?php

namespace WPOutreach\Automation;

/**
 * Content Trigger Handler
 *
 * Handles content-based automation triggers (post_published, post_updated)
 * independently of the post subscription system.
 *
 * This allows automations to fire when any post is published/updated,
 * targeting all active subscribers or those in specific lists.
 *
 * @since 2.6.3
 */
class ContentTriggerHandler
{
    /**
     * Excluded post types that shouldn't trigger automations
     */
    private static array $excluded_post_types = [
        'revision',
        'nav_menu_item',
        'custom_css',
        'customize_changeset',
        'oembed_cache',
        'user_request',
        'wp_block',
        'wp_template',
        'wp_template_part',
        'wp_global_styles',
        'wp_navigation',
    ];

    /**
     * Initialize WordPress hooks
     */
    public static function init(): void
    {
        // Hook into post status transitions (for publish events)
        add_action('transition_post_status', [self::class, 'onPostStatusChange'], 20, 3);

        // Hook into post updates (for content changes on published posts)
        add_action('post_updated', [self::class, 'onPostUpdated'], 20, 3);
    }

    /**
     * Handle post status transitions
     *
     * @param string   $new_status New post status
     * @param string   $old_status Old post status
     * @param \WP_Post $post       Post object
     */
    public static function onPostStatusChange(string $new_status, string $old_status, \WP_Post $post): void
    {
        // Skip excluded post types
        if (in_array($post->post_type, self::$excluded_post_types, true)) {
            return;
        }

        // Only trigger when transitioning TO publish from non-publish
        if ($new_status === 'publish' && $old_status !== 'publish') {
            self::fireContentTrigger('post_published', $post);
        }
    }

    /**
     * Handle post content updates
     *
     * @param int      $post_id     Post ID
     * @param \WP_Post $post_after  Post object after update
     * @param \WP_Post $post_before Post object before update
     */
    public static function onPostUpdated(int $post_id, \WP_Post $post_after, \WP_Post $post_before): void
    {
        // Skip excluded post types
        if (in_array($post_after->post_type, self::$excluded_post_types, true)) {
            return;
        }

        // Only trigger if post was already published and content actually changed
        if ($post_before->post_status === 'publish' && $post_after->post_status === 'publish') {
            $contentChanged = $post_before->post_content !== $post_after->post_content;
            $titleChanged = $post_before->post_title !== $post_after->post_title;

            if ($contentChanged || $titleChanged) {
                self::fireContentTrigger('post_updated', $post_after);
            }
        }
    }

    /**
     * Fire a content-based trigger for all matching automations
     *
     * @param string   $trigger_id Trigger ID ('post_published' or 'post_updated')
     * @param \WP_Post $post       Post object
     */
    private static function fireContentTrigger(string $trigger_id, \WP_Post $post): void
    {
        global $wpdb;

        // Get active automations with this trigger type
        $automations = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_automations
             WHERE status = 'active' AND trigger_type = %s",
            $trigger_id
        ));

        if (empty($automations)) {
            return;
        }

        // Build post context
        $context = self::buildPostContext($post);

        foreach ($automations as $automation) {
            // Check if this automation's trigger config matches this post
            $trigger_config = json_decode($automation->trigger_config, true) ?: [];

            // If post_type is configured, check it matches
            if (!empty($trigger_config['post_type']) && $trigger_config['post_type'] !== $post->post_type) {
                continue;
            }

            // Get subscribers to target for this automation
            // Combines: 1) Post-specific subscribers AND 2) List subscribers (if configured)
            $subscriber_ids = self::getSubscribersForTrigger($automation, $post->ID, $trigger_id);

            if (empty($subscriber_ids)) {
                continue;
            }

            // Fire the trigger for each subscriber
            foreach ($subscriber_ids as $subscriber_id) {
                // Use AutomationEngine::enqueueSubscriber directly to avoid double-triggering
                // from TriggerRegistry which was already called from PostSubscriptionHandler
                $result = AutomationEngine::enqueueSubscriber(
                    (int) $automation->id,
                    (int) $subscriber_id,
                    $context
                );

                if ($result) {
                    do_action('wp_outreach_content_automation_triggered', [
                        'automation_id' => $automation->id,
                        'subscriber_id' => $subscriber_id,
                        'trigger_id' => $trigger_id,
                        'post_id' => $post->ID,
                        'context' => $context,
                    ]);
                }
            }
        }
    }

    /**
     * Get all subscribers for a content trigger
     *
     * Combines two sources based on send_email step configuration:
     * 1. Post-specific subscribers (from wp_outreach_post_subscriptions)
     * 2. List subscribers (if send_email step has list_ids configured)
     *
     * The recipient_type in send_email step config determines who receives the email:
     * - 'auto' or 'post_subscribers': Only post subscribers (backward compatible)
     * - 'lists': Only subscribers from specified lists
     * - 'both': Both post subscribers and list subscribers
     *
     * @param object $automation  Automation row
     * @param int    $post_id     The post ID that was updated/published
     * @param string $trigger_id  The trigger type ('post_updated' or 'post_published')
     * @return array Unique subscriber IDs from both sources
     */
    private static function getSubscribersForTrigger(object $automation, int $post_id, string $trigger_id): array
    {
        $subscriber_ids = [];

        // Parse automation steps to find send_email configuration
        $steps = json_decode($automation->steps, true) ?: [];
        $recipient_config = self::getSendEmailRecipientConfig($steps);

        $recipient_type = $recipient_config['recipient_type'] ?? 'auto';
        $list_ids = $recipient_config['list_ids'] ?? [];

        // Normalize 'auto' to 'post_subscribers' for backward compatibility
        if ($recipient_type === 'auto') {
            $recipient_type = 'post_subscribers';
        }

        // Determine which subscribers to include based on recipient_type
        $include_post_subscribers = in_array($recipient_type, ['post_subscribers', 'both'], true);
        $include_list_subscribers = in_array($recipient_type, ['lists', 'both'], true);

        // 1. Get post-specific subscribers (people who subscribed to THIS post)
        if ($include_post_subscribers) {
            $post_subscribers = self::getPostSubscribers($post_id, $trigger_id);
            $subscriber_ids = array_merge($subscriber_ids, $post_subscribers);
        }

        // 2. Get list subscribers (if send_email step has list_ids configured)
        if ($include_list_subscribers && !empty($list_ids)) {
            $list_subscribers = self::getListSubscribers($list_ids);
            $subscriber_ids = array_merge($subscriber_ids, $list_subscribers);
        }

        // Remove duplicates and return
        return array_unique($subscriber_ids);
    }

    /**
     * Extract recipient configuration from send_email steps
     *
     * Searches through automation steps (including nested branch steps)
     * to find the first send_email step with recipient configuration.
     *
     * @param array $steps Automation steps array
     * @return array Recipient config with 'recipient_type' and 'list_ids'
     */
    private static function getSendEmailRecipientConfig(array $steps): array
    {
        foreach ($steps as $step) {
            if (($step['type'] ?? '') === 'send_email') {
                $config = $step['config'] ?? [];
                if (!empty($config['recipient_type'])) {
                    return [
                        'recipient_type' => $config['recipient_type'],
                        'list_ids' => $config['list_ids'] ?? [],
                    ];
                }
            }

            // Check condition branch steps (yes_steps and no_steps)
            if (($step['type'] ?? '') === 'condition') {
                $config = $step['config'] ?? [];

                // Check yes_steps
                if (!empty($config['yes_steps'])) {
                    $result = self::getSendEmailRecipientConfig($config['yes_steps']);
                    if (!empty($result['recipient_type'])) {
                        return $result;
                    }
                }

                // Check no_steps
                if (!empty($config['no_steps'])) {
                    $result = self::getSendEmailRecipientConfig($config['no_steps']);
                    if (!empty($result['recipient_type'])) {
                        return $result;
                    }
                }
            }
        }

        // Default: auto (behaves like post_subscribers for backward compatibility)
        return [
            'recipient_type' => 'auto',
            'list_ids' => [],
        ];
    }

    /**
     * Get subscribers from specific lists
     *
     * @param array $list_ids List IDs to get subscribers from
     * @return array Subscriber IDs
     */
    private static function getListSubscribers(array $list_ids): array
    {
        global $wpdb;

        if (empty($list_ids)) {
            return [];
        }

        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $placeholders = implode(',', array_fill(0, count($list_ids), '%d'));

        return $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT s.id
             FROM {$subscribers_table} s
             INNER JOIN {$pivot_table} sl ON s.id = sl.subscriber_id
             WHERE sl.list_id IN ({$placeholders})
             AND s.status = 'active'",
            ...$list_ids
        ));
    }

    /**
     * Get subscribers who subscribed to a specific post
     *
     * Queries wp_outreach_post_subscriptions table to find subscribers
     * who want updates for this specific post.
     *
     * @param int    $post_id    The post ID that was updated/published
     * @param string $trigger_id The trigger type ('post_updated' or 'post_published')
     * @return array Subscriber IDs
     */
    private static function getPostSubscribers(int $post_id, string $trigger_id): array
    {
        global $wpdb;

        $post_subs_table = $wpdb->prefix . 'outreach_post_subscriptions';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        // Determine which subscription types to include based on trigger
        // post_updated -> 'update' or 'both'
        // post_published -> 'new' or 'both' (for new posts in a category/type they subscribed to)
        if ($trigger_id === 'post_updated') {
            $subscription_types = ['update', 'both'];
        } else {
            // post_published - for now, also use 'update' and 'both' since user subscribed to this post
            $subscription_types = ['update', 'both'];
        }

        $type_placeholders = implode(',', array_fill(0, count($subscription_types), '%s'));

        // Get subscribers who subscribed to this specific post
        $query = $wpdb->prepare(
            "SELECT DISTINCT ps.subscriber_id
             FROM {$post_subs_table} ps
             INNER JOIN {$subscribers_table} s ON ps.subscriber_id = s.id
             WHERE ps.post_id = %d
             AND ps.subscription_type IN ({$type_placeholders})
             AND ps.status = 1
             AND s.status = 'active'",
            $post_id,
            ...$subscription_types
        );

        return $wpdb->get_col($query);
    }

    /**
     * Build context data from post for merge tags
     *
     * @param \WP_Post $post Post object
     * @return array Context data
     */
    private static function buildPostContext(\WP_Post $post): array
    {
        $author = get_userdata($post->post_author);
        $thumbnail_id = get_post_thumbnail_id($post->ID);

        return [
            'post_id' => $post->ID,
            'post_type' => $post->post_type,
            'post_title' => $post->post_title,
            'post_url' => get_permalink($post->ID),
            'post_excerpt' => wp_trim_words($post->post_excerpt ?: $post->post_content, 55, '...'),
            'post_content' => $post->post_content,
            'post_author' => $author ? $author->display_name : '',
            'post_date' => get_the_date('', $post),
            'post_thumbnail' => $thumbnail_id ? wp_get_attachment_image_url($thumbnail_id, 'large') : '',
            'post_type_label' => get_post_type_object($post->post_type)->labels->singular_name ?? $post->post_type,
            'site_name' => get_bloginfo('name'),
            'site_url' => get_site_url(),
            'current_date' => current_time('Y-m-d'),
            'current_year' => current_time('Y'),
        ];
    }
}
