<?php

namespace WPOutreach\Automation;

/**
 * Condition Evaluator
 *
 * Evaluates conditions for automation branching logic.
 * Supports various operators and dot notation for nested field access.
 * Provides trigger-specific field definitions with proper types.
 *
 * @since 1.3.0
 */
class ConditionEvaluator
{
    /**
     * Field types and their applicable operators
     */
    public const FIELD_TYPES = [
        'string' => [
            'operators' => ['equals', 'not_equals', 'contains', 'not_contains', 'starts_with', 'ends_with', 'is_empty', 'is_not_empty', 'matches_regex'],
            'input' => 'text',
        ],
        'number' => [
            'operators' => ['equals', 'not_equals', 'greater_than', 'less_than', 'greater_than_or_equals', 'less_than_or_equals', 'is_empty', 'is_not_empty'],
            'input' => 'number',
        ],
        'boolean' => [
            'operators' => ['equals', 'not_equals'],
            'input' => 'boolean',
        ],
        'select' => [
            'operators' => ['equals', 'not_equals', 'is_empty', 'is_not_empty'],
            'input' => 'select',
        ],
        'multiselect' => [
            'operators' => ['contains', 'not_contains', 'is_empty', 'is_not_empty'],
            'input' => 'multiselect',
        ],
        'array' => [
            'operators' => ['contains', 'not_contains', 'is_empty', 'is_not_empty', 'in_list', 'not_in_list'],
            'input' => 'text',
        ],
        'date' => [
            'operators' => ['equals', 'not_equals', 'greater_than', 'less_than', 'is_empty', 'is_not_empty'],
            'input' => 'date',
        ],
        'email' => [
            'operators' => ['equals', 'not_equals', 'contains', 'not_contains', 'ends_with', 'is_empty', 'is_not_empty'],
            'input' => 'text',
        ],
    ];

    /**
     * Available operators and their descriptions
     */
    public const OPERATORS = [
        // String/General
        'equals' => 'equals',
        'not_equals' => 'does not equal',
        'contains' => 'contains',
        'not_contains' => 'does not contain',
        'starts_with' => 'starts with',
        'ends_with' => 'ends with',
        'is_empty' => 'is empty',
        'is_not_empty' => 'is not empty',

        // Numeric
        'greater_than' => 'is greater than',
        'less_than' => 'is less than',
        'greater_than_or_equals' => 'is at least',
        'less_than_or_equals' => 'is at most',

        // Pattern
        'matches_regex' => 'matches pattern',

        // Array/List
        'in_list' => 'is one of',
        'not_in_list' => 'is not one of',
    ];

    /**
     * Operators that don't require a comparison value
     */
    public const UNARY_OPERATORS = ['is_empty', 'is_not_empty'];

    /**
     * Get available fields for a specific trigger
     *
     * @param string|null $triggerType The trigger type ID
     * @return array Grouped fields with metadata
     */
    public static function getFieldsForTrigger(?string $triggerType = null): array
    {
        $fields = [];

        // Base subscriber fields (always available)
        $fields['subscriber'] = [
            'label' => __('Subscriber', 'outreach'),
            'icon' => 'user',
            'fields' => [
                [
                    'value' => 'subscriber.email',
                    'label' => __('Email Address', 'outreach'),
                    'type' => 'email',
                    'description' => __('Subscriber\'s email address', 'outreach'),
                ],
                [
                    'value' => 'subscriber.first_name',
                    'label' => __('First Name', 'outreach'),
                    'type' => 'string',
                    'description' => __('Subscriber\'s first name', 'outreach'),
                ],
                [
                    'value' => 'subscriber.last_name',
                    'label' => __('Last Name', 'outreach'),
                    'type' => 'string',
                    'description' => __('Subscriber\'s last name', 'outreach'),
                ],
                [
                    'value' => 'subscriber.status',
                    'label' => __('Status', 'outreach'),
                    'type' => 'select',
                    'description' => __('Current subscription status', 'outreach'),
                    'options' => [
                        ['value' => 'active', 'label' => __('Active', 'outreach')],
                        ['value' => 'pending', 'label' => __('Pending', 'outreach')],
                        ['value' => 'unsubscribed', 'label' => __('Unsubscribed', 'outreach')],
                        ['value' => 'bounced', 'label' => __('Bounced', 'outreach')],
                    ],
                ],
                [
                    'value' => 'subscriber.source',
                    'label' => __('Source', 'outreach'),
                    'type' => 'string',
                    'description' => __('How the subscriber was added', 'outreach'),
                ],
            ],
        ];

        // Add trigger-specific fields
        switch ($triggerType) {
            case 'wpdm_new_purchase':
                $fields = array_merge($fields, self::getWPDMPurchaseFields());
                break;

            case 'wc_checkout_submitted':
            case 'wc_order_paid':
            case 'wc_order_completed':
                $fields = array_merge($fields, self::getWooCommerceFields());
                break;

            case 'cf7_form_submitted':
            case 'wpforms_form_submitted':
            case 'gf_form_submitted':
                $fields = array_merge($fields, self::getFormSubmissionFields($triggerType));
                break;

            case 'subscriber_joins_list':
            case 'subscriber_leaves_list':
                $fields = array_merge($fields, self::getListFields());
                break;

            case 'subscriber_added_tag':
            case 'subscriber_removed_tag':
                $fields = array_merge($fields, self::getTagFields());
                break;

            case 'post_published':
            case 'post_updated':
            case 'post_subscription_created':
                $fields = array_merge($fields, self::getPostFields());
                break;

            case 'incoming_webhook':
                $fields = array_merge($fields, self::getWebhookFields());
                break;
        }

        // Allow third-party extensions
        $fields = apply_filters('wp_outreach_condition_fields', $fields, $triggerType);

        return $fields;
    }

    /**
     * Get WPDM Purchase specific fields
     */
    private static function getWPDMPurchaseFields(): array
    {
        $products = [];
        if (class_exists('\\WPOutreach\\Integrations\\WPDMIntegration')) {
            $products = \WPOutreach\Integrations\WPDMIntegration::getProducts();
        }

        $productOptions = array_map(function ($p) {
            return ['value' => (string) $p['id'], 'label' => $p['name']];
        }, $products);

        return [
            'order' => [
                'label' => __('Order Details', 'outreach'),
                'icon' => 'shopping-cart',
                'fields' => [
                    [
                        'value' => 'order_total',
                        'label' => __('Order Total', 'outreach'),
                        'type' => 'number',
                        'description' => __('Total amount paid including tax', 'outreach'),
                        'suffix' => '$',
                    ],
                    [
                        'value' => 'subtotal',
                        'label' => __('Subtotal', 'outreach'),
                        'type' => 'number',
                        'description' => __('Order total before tax and discounts', 'outreach'),
                        'suffix' => '$',
                    ],
                    [
                        'value' => 'tax',
                        'label' => __('Tax Amount', 'outreach'),
                        'type' => 'number',
                        'description' => __('Tax charged on the order', 'outreach'),
                    ],
                    [
                        'value' => 'discount',
                        'label' => __('Discount Amount', 'outreach'),
                        'type' => 'number',
                        'description' => __('Discount applied to the order', 'outreach'),
                    ],
                    [
                        'value' => 'currency',
                        'label' => __('Currency', 'outreach'),
                        'type' => 'select',
                        'description' => __('Currency used for the purchase', 'outreach'),
                        'options' => [
                            ['value' => 'USD', 'label' => 'USD - US Dollar'],
                            ['value' => 'EUR', 'label' => 'EUR - Euro'],
                            ['value' => 'GBP', 'label' => 'GBP - British Pound'],
                            ['value' => 'CAD', 'label' => 'CAD - Canadian Dollar'],
                            ['value' => 'AUD', 'label' => 'AUD - Australian Dollar'],
                        ],
                    ],
                    [
                        'value' => 'coupon_code',
                        'label' => __('Coupon Code', 'outreach'),
                        'type' => 'string',
                        'description' => __('Coupon code used (if any)', 'outreach'),
                    ],
                    [
                        'value' => 'payment_method',
                        'label' => __('Payment Method', 'outreach'),
                        'type' => 'string',
                        'description' => __('Payment gateway used', 'outreach'),
                    ],
                ],
            ],
            'products' => [
                'label' => __('Products', 'outreach'),
                'icon' => 'package',
                'fields' => [
                    [
                        'value' => 'product_ids',
                        'label' => __('Purchased Products', 'outreach'),
                        'type' => 'multiselect',
                        'description' => __('Check if specific products were purchased', 'outreach'),
                        'options' => $productOptions,
                        'placeholder' => __('Select products...', 'outreach'),
                    ],
                    [
                        'value' => 'product_count',
                        'label' => __('Number of Products', 'outreach'),
                        'type' => 'number',
                        'description' => __('How many products were in the order', 'outreach'),
                        'computed' => true,
                    ],
                ],
            ],
            'customer' => [
                'label' => __('Customer', 'outreach'),
                'icon' => 'user',
                'fields' => [
                    [
                        'value' => 'customer_email',
                        'label' => __('Customer Email', 'outreach'),
                        'type' => 'email',
                        'description' => __('Email used for the order', 'outreach'),
                    ],
                    [
                        'value' => 'customer_name',
                        'label' => __('Customer Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Full name from billing info', 'outreach'),
                    ],
                    [
                        'value' => 'first_name',
                        'label' => __('First Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer\'s first name', 'outreach'),
                    ],
                    [
                        'value' => 'last_name',
                        'label' => __('Last Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer\'s last name', 'outreach'),
                    ],
                    [
                        'value' => 'user_id',
                        'label' => __('WordPress User', 'outreach'),
                        'type' => 'number',
                        'description' => __('WordPress user ID (0 if guest)', 'outreach'),
                    ],
                    [
                        'value' => 'is_registered',
                        'label' => __('Is Registered User', 'outreach'),
                        'type' => 'boolean',
                        'description' => __('Whether the customer has a WordPress account', 'outreach'),
                        'computed' => true,
                    ],
                ],
            ],
        ];
    }

    /**
     * Get WooCommerce specific fields
     */
    private static function getWooCommerceFields(): array
    {
        $productOptions = [];
        if (class_exists('WooCommerce')) {
            $products = wc_get_products(['limit' => 100, 'status' => 'publish']);
            foreach ($products as $product) {
                $productOptions[] = ['value' => (string) $product->get_id(), 'label' => $product->get_name()];
            }
        }

        return [
            'order' => [
                'label' => __('Order Details', 'outreach'),
                'icon' => 'shopping-cart',
                'fields' => [
                    [
                        'value' => 'order_total',
                        'label' => __('Order Total', 'outreach'),
                        'type' => 'number',
                        'description' => __('Total order amount', 'outreach'),
                    ],
                    [
                        'value' => 'order_subtotal',
                        'label' => __('Subtotal', 'outreach'),
                        'type' => 'number',
                        'description' => __('Order subtotal before tax/shipping', 'outreach'),
                    ],
                    [
                        'value' => 'order_tax',
                        'label' => __('Tax', 'outreach'),
                        'type' => 'number',
                        'description' => __('Tax amount', 'outreach'),
                    ],
                    [
                        'value' => 'order_shipping',
                        'label' => __('Shipping', 'outreach'),
                        'type' => 'number',
                        'description' => __('Shipping cost', 'outreach'),
                    ],
                    [
                        'value' => 'currency',
                        'label' => __('Currency', 'outreach'),
                        'type' => 'string',
                        'description' => __('Order currency', 'outreach'),
                    ],
                    [
                        'value' => 'payment_method',
                        'label' => __('Payment Method', 'outreach'),
                        'type' => 'string',
                        'description' => __('Payment method used', 'outreach'),
                    ],
                    [
                        'value' => 'item_count',
                        'label' => __('Item Count', 'outreach'),
                        'type' => 'number',
                        'description' => __('Number of items in order', 'outreach'),
                    ],
                ],
            ],
            'products' => [
                'label' => __('Products', 'outreach'),
                'icon' => 'package',
                'fields' => [
                    [
                        'value' => 'product_ids',
                        'label' => __('Products in Order', 'outreach'),
                        'type' => 'multiselect',
                        'description' => __('Check if specific products were ordered', 'outreach'),
                        'options' => $productOptions,
                    ],
                ],
            ],
            'customer' => [
                'label' => __('Customer', 'outreach'),
                'icon' => 'user',
                'fields' => [
                    [
                        'value' => 'email',
                        'label' => __('Email', 'outreach'),
                        'type' => 'email',
                        'description' => __('Customer email', 'outreach'),
                    ],
                    [
                        'value' => 'first_name',
                        'label' => __('First Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer first name', 'outreach'),
                    ],
                    [
                        'value' => 'last_name',
                        'label' => __('Last Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer last name', 'outreach'),
                    ],
                    [
                        'value' => 'billing_city',
                        'label' => __('Billing City', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer billing city', 'outreach'),
                    ],
                    [
                        'value' => 'billing_country',
                        'label' => __('Billing Country', 'outreach'),
                        'type' => 'string',
                        'description' => __('Customer billing country code', 'outreach'),
                    ],
                    [
                        'value' => 'is_returning_customer',
                        'label' => __('Is Returning Customer', 'outreach'),
                        'type' => 'boolean',
                        'description' => __('Has made previous purchases', 'outreach'),
                    ],
                ],
            ],
        ];
    }

    /**
     * Get form submission specific fields
     */
    private static function getFormSubmissionFields(string $triggerType): array
    {
        $formPlugin = match ($triggerType) {
            'cf7_form_submitted' => 'Contact Form 7',
            'wpforms_form_submitted' => 'WPForms',
            'gf_form_submitted' => 'Gravity Forms',
            default => 'Form',
        };

        return [
            'form' => [
                'label' => __('Form Details', 'outreach'),
                'icon' => 'mail',
                'fields' => [
                    [
                        'value' => 'form_id',
                        'label' => __('Form ID', 'outreach'),
                        'type' => 'number',
                        'description' => sprintf(__('%s form ID', 'outreach'), $formPlugin),
                    ],
                    [
                        'value' => 'form_name',
                        'label' => __('Form Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Name/title of the form', 'outreach'),
                    ],
                ],
            ],
            'fields' => [
                'label' => __('Form Fields', 'outreach'),
                'icon' => 'edit',
                'description' => __('Access any submitted form field using fields.field_name', 'outreach'),
                'dynamic' => true,
                'fields' => [
                    [
                        'value' => 'fields.*',
                        'label' => __('Any Field', 'outreach'),
                        'type' => 'string',
                        'description' => __('Replace * with the field name (e.g., fields.interest)', 'outreach'),
                        'placeholder' => 'fields.your_field_name',
                        'editable' => true,
                    ],
                    [
                        'value' => 'email',
                        'label' => __('Email', 'outreach'),
                        'type' => 'email',
                        'description' => __('Extracted email from form', 'outreach'),
                    ],
                    [
                        'value' => 'first_name',
                        'label' => __('First Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Extracted first name from form', 'outreach'),
                    ],
                ],
            ],
        ];
    }

    /**
     * Get list-related fields
     */
    private static function getListFields(): array
    {
        global $wpdb;
        $lists = $wpdb->get_results("SELECT id, name FROM {$wpdb->prefix}outreach_lists ORDER BY name");
        $listOptions = array_map(fn($l) => ['value' => (string) $l->id, 'label' => $l->name], $lists ?: []);

        return [
            'list' => [
                'label' => __('List', 'outreach'),
                'icon' => 'users',
                'fields' => [
                    [
                        'value' => 'list_id',
                        'label' => __('List', 'outreach'),
                        'type' => 'select',
                        'description' => __('The list that was joined/left', 'outreach'),
                        'options' => $listOptions,
                    ],
                    [
                        'value' => 'list_name',
                        'label' => __('List Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Name of the list', 'outreach'),
                    ],
                ],
            ],
        ];
    }

    /**
     * Get tag-related fields
     */
    private static function getTagFields(): array
    {
        global $wpdb;
        $tags = $wpdb->get_results("SELECT id, name FROM {$wpdb->prefix}outreach_tags ORDER BY name");
        $tagOptions = array_map(fn($t) => ['value' => (string) $t->id, 'label' => $t->name], $tags ?: []);

        return [
            'tag' => [
                'label' => __('Tag', 'outreach'),
                'icon' => 'tag',
                'fields' => [
                    [
                        'value' => 'tag_id',
                        'label' => __('Tag', 'outreach'),
                        'type' => 'select',
                        'description' => __('The tag that was added/removed', 'outreach'),
                        'options' => $tagOptions,
                    ],
                    [
                        'value' => 'tag_name',
                        'label' => __('Tag Name', 'outreach'),
                        'type' => 'string',
                        'description' => __('Name of the tag', 'outreach'),
                    ],
                ],
            ],
        ];
    }

    /**
     * Get post-related fields
     */
    private static function getPostFields(): array
    {
        return [
            'post' => [
                'label' => __('Post/Page', 'outreach'),
                'icon' => 'file-text',
                'fields' => [
                    [
                        'value' => 'post_id',
                        'label' => __('Post ID', 'outreach'),
                        'type' => 'number',
                        'description' => __('ID of the post/page', 'outreach'),
                    ],
                    [
                        'value' => 'post_title',
                        'label' => __('Post Title', 'outreach'),
                        'type' => 'string',
                        'description' => __('Title of the post/page', 'outreach'),
                    ],
                    [
                        'value' => 'post_type',
                        'label' => __('Post Type', 'outreach'),
                        'type' => 'select',
                        'description' => __('Type of content', 'outreach'),
                        'options' => [
                            ['value' => 'post', 'label' => __('Post', 'outreach')],
                            ['value' => 'page', 'label' => __('Page', 'outreach')],
                            ['value' => 'wpdmpro', 'label' => __('Download', 'outreach')],
                        ],
                    ],
                    [
                        'value' => 'post_author',
                        'label' => __('Author ID', 'outreach'),
                        'type' => 'number',
                        'description' => __('WordPress user ID of the author', 'outreach'),
                    ],
                ],
            ],
        ];
    }

    /**
     * Get incoming webhook fields
     *
     * Provides fields for evaluating conditions based on webhook payload data.
     * Supports both predefined common fields and dynamic payload access.
     */
    private static function getWebhookFields(): array
    {
        return [
            'webhook' => [
                'label' => __('Webhook Data', 'outreach'),
                'icon' => 'webhook',
                'fields' => [
                    [
                        'value' => 'webhook_payload.event_type',
                        'label' => __('Event Type', 'outreach'),
                        'type' => 'string',
                        'description' => __('Common field: event_type, type, event, or action', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.action',
                        'label' => __('Action', 'outreach'),
                        'type' => 'string',
                        'description' => __('Action or event name from webhook', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.status',
                        'label' => __('Status', 'outreach'),
                        'type' => 'string',
                        'description' => __('Status field from webhook payload', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.amount',
                        'label' => __('Amount', 'outreach'),
                        'type' => 'number',
                        'description' => __('Numeric amount (e.g., order total, payment)', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.currency',
                        'label' => __('Currency', 'outreach'),
                        'type' => 'string',
                        'description' => __('Currency code (e.g., USD, EUR)', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.product_id',
                        'label' => __('Product ID', 'outreach'),
                        'type' => 'string',
                        'description' => __('Product or item ID from webhook', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.plan',
                        'label' => __('Plan/Tier', 'outreach'),
                        'type' => 'string',
                        'description' => __('Subscription plan or tier name', 'outreach'),
                    ],
                    [
                        'value' => 'webhook_payload.source',
                        'label' => __('Source', 'outreach'),
                        'type' => 'string',
                        'description' => __('Source or origin of the webhook event', 'outreach'),
                    ],
                ],
            ],
            'webhook_custom' => [
                'label' => __('Custom Payload Fields', 'outreach'),
                'icon' => 'code',
                'description' => __('Access any field from the webhook JSON payload using dot notation', 'outreach'),
                'dynamic' => true,
                'fields' => [
                    [
                        'value' => 'webhook_payload.*',
                        'label' => __('Any Payload Field', 'outreach'),
                        'type' => 'string',
                        'description' => __('Replace * with field path (e.g., webhook_payload.data.customer.id)', 'outreach'),
                        'placeholder' => 'webhook_payload.your_field',
                        'editable' => true,
                    ],
                ],
            ],
        ];
    }

    /**
     * Get operators for a specific field type
     */
    public static function getOperatorsForType(string $fieldType): array
    {
        $typeConfig = self::FIELD_TYPES[$fieldType] ?? self::FIELD_TYPES['string'];
        $operatorKeys = $typeConfig['operators'];

        $operators = [];
        foreach ($operatorKeys as $key) {
            if (isset(self::OPERATORS[$key])) {
                $operators[$key] = self::OPERATORS[$key];
            }
        }

        return $operators;
    }

    /**
     * Evaluate a single condition
     *
     * @param array $condition Condition config with field, operator, value
     * @param array $context Data context to evaluate against
     * @return bool
     */
    public function evaluate(array $condition, array $context): bool
    {
        $field = $condition['field'] ?? '';
        $operator = $condition['operator'] ?? 'equals';
        $expectedValue = $condition['value'] ?? '';

        // Handle computed fields
        $context = $this->addComputedFields($context);

        // Get the actual value from context using dot notation
        $actualValue = $this->getFieldValue($field, $context);

        return $this->compare($actualValue, $operator, $expectedValue);
    }

    /**
     * Add computed fields to context
     */
    private function addComputedFields(array $context): array
    {
        // product_count from product_ids
        if (isset($context['product_ids']) && is_array($context['product_ids'])) {
            $context['product_count'] = count($context['product_ids']);
        }

        // is_registered from user_id
        if (isset($context['user_id'])) {
            $context['is_registered'] = (int) $context['user_id'] > 0;
        }

        return $context;
    }

    /**
     * Evaluate multiple conditions with AND/OR logic
     *
     * @param array $conditions Array of condition configs
     * @param array $context Data context to evaluate against
     * @param string $logic 'and' or 'or'
     * @return bool
     */
    public function evaluateGroup(array $conditions, array $context, string $logic = 'and'): bool
    {
        if (empty($conditions)) {
            return true; // No conditions = pass
        }

        $logic = strtolower($logic);

        foreach ($conditions as $condition) {
            $result = $this->evaluate($condition, $context);

            if ($logic === 'or' && $result) {
                return true; // OR: any true = pass
            }

            if ($logic === 'and' && !$result) {
                return false; // AND: any false = fail
            }
        }

        // AND: all passed, OR: none passed
        return $logic === 'and';
    }

    /**
     * Get a field value from context using dot notation
     *
     * Examples:
     * - 'email' => $context['email']
     * - 'subscriber.first_name' => $context['subscriber']['first_name']
     * - 'fields.interest' => $context['fields']['interest']
     *
     * @param string $field Field path with dot notation
     * @param array $context Data context
     * @return mixed
     */
    public function getFieldValue(string $field, array $context): mixed
    {
        if (empty($field)) {
            return null;
        }

        $keys = explode('.', $field);
        $value = $context;

        foreach ($keys as $key) {
            if (is_array($value) && array_key_exists($key, $value)) {
                $value = $value[$key];
            } elseif (is_object($value) && property_exists($value, $key)) {
                $value = $value->$key;
            } else {
                return null;
            }
        }

        return $value;
    }

    /**
     * Compare values using the specified operator
     *
     * @param mixed $actual The actual value from context
     * @param string $operator The comparison operator
     * @param mixed $expected The expected value to compare against
     * @return bool
     */
    public function compare(mixed $actual, string $operator, mixed $expected): bool
    {
        // Normalize string values
        if (is_string($actual)) {
            $actual = trim($actual);
        }
        if (is_string($expected)) {
            $expected = trim($expected);
        }

        return match ($operator) {
            'equals' => $this->compareEquals($actual, $expected),
            'not_equals' => !$this->compareEquals($actual, $expected),
            'contains' => $this->compareContains($actual, $expected),
            'not_contains' => !$this->compareContains($actual, $expected),
            'starts_with' => $this->compareStartsWith($actual, $expected),
            'ends_with' => $this->compareEndsWith($actual, $expected),
            'is_empty' => $this->isEmpty($actual),
            'is_not_empty' => !$this->isEmpty($actual),
            'greater_than' => $this->compareNumeric($actual, $expected, '>'),
            'less_than' => $this->compareNumeric($actual, $expected, '<'),
            'greater_than_or_equals' => $this->compareNumeric($actual, $expected, '>='),
            'less_than_or_equals' => $this->compareNumeric($actual, $expected, '<='),
            'matches_regex' => $this->matchesRegex($actual, $expected),
            'in_list' => $this->inList($actual, $expected),
            'not_in_list' => !$this->inList($actual, $expected),
            default => false,
        };
    }

    /**
     * Check if values are equal (case-insensitive for strings)
     */
    private function compareEquals(mixed $actual, mixed $expected): bool
    {
        if (is_string($actual) && is_string($expected)) {
            return strtolower($actual) === strtolower($expected);
        }

        // Handle boolean-like values
        if (is_bool($expected)) {
            return $this->toBool($actual) === $expected;
        }

        if (is_numeric($actual) && is_numeric($expected)) {
            return (float) $actual === (float) $expected;
        }

        return $actual === $expected;
    }

    /**
     * Check if actual contains expected (case-insensitive)
     */
    private function compareContains(mixed $actual, mixed $expected): bool
    {
        // For arrays, check if expected is in array
        if (is_array($actual)) {
            // If checking if array contains a specific value
            if (!is_array($expected)) {
                return in_array($expected, $actual, false) ||
                       in_array((string) $expected, array_map('strval', $actual), false);
            }
            // If checking if all expected values are in array
            foreach ((array) $expected as $exp) {
                if (!in_array($exp, $actual, false) &&
                    !in_array((string) $exp, array_map('strval', $actual), false)) {
                    return false;
                }
            }
            return true;
        }

        if (!is_string($actual) || !is_string($expected)) {
            return false;
        }

        return str_contains(strtolower($actual), strtolower($expected));
    }

    /**
     * Check if actual starts with expected (case-insensitive)
     */
    private function compareStartsWith(mixed $actual, mixed $expected): bool
    {
        if (!is_string($actual) || !is_string($expected)) {
            return false;
        }

        return str_starts_with(strtolower($actual), strtolower($expected));
    }

    /**
     * Check if actual ends with expected (case-insensitive)
     */
    private function compareEndsWith(mixed $actual, mixed $expected): bool
    {
        if (!is_string($actual) || !is_string($expected)) {
            return false;
        }

        return str_ends_with(strtolower($actual), strtolower($expected));
    }

    /**
     * Check if value is empty
     */
    private function isEmpty(mixed $value): bool
    {
        if ($value === null) {
            return true;
        }

        if (is_string($value)) {
            return trim($value) === '';
        }

        if (is_array($value)) {
            return count($value) === 0;
        }

        return empty($value);
    }

    /**
     * Compare numeric values
     */
    private function compareNumeric(mixed $actual, mixed $expected, string $operator): bool
    {
        if (!is_numeric($actual) || !is_numeric($expected)) {
            return false;
        }

        $actual = (float) $actual;
        $expected = (float) $expected;

        return match ($operator) {
            '>' => $actual > $expected,
            '<' => $actual < $expected,
            '>=' => $actual >= $expected,
            '<=' => $actual <= $expected,
            default => false,
        };
    }

    /**
     * Check if value matches regex pattern
     */
    private function matchesRegex(mixed $actual, mixed $pattern): bool
    {
        if (!is_string($actual) || !is_string($pattern)) {
            return false;
        }

        // Add delimiters if not present
        if (!preg_match('/^[\/\#\~\@]/', $pattern)) {
            $pattern = '/' . $pattern . '/i';
        }

        // Suppress errors for invalid regex
        $result = @preg_match($pattern, $actual);

        return $result === 1;
    }

    /**
     * Check if value is in a list
     *
     * Expected can be:
     * - Comma-separated string: "a,b,c"
     * - Array: ['a', 'b', 'c']
     */
    private function inList(mixed $actual, mixed $expected): bool
    {
        if (is_string($expected)) {
            $list = array_map('trim', explode(',', $expected));
        } elseif (is_array($expected)) {
            $list = $expected;
        } else {
            return false;
        }

        // Case-insensitive comparison for strings
        if (is_string($actual)) {
            $actual = strtolower($actual);
            $list = array_map('strtolower', $list);
        }

        return in_array($actual, $list, false);
    }

    /**
     * Convert value to boolean
     */
    private function toBool(mixed $value): bool
    {
        if (is_bool($value)) {
            return $value;
        }

        if (is_string($value)) {
            $lower = strtolower(trim($value));
            return in_array($lower, ['true', 'yes', '1', 'on'], true);
        }

        return (bool) $value;
    }

    /**
     * Get available operators for UI
     *
     * @param string|null $type Filter by type: 'string', 'numeric', 'all'
     * @return array
     */
    public static function getOperators(?string $type = 'all'): array
    {
        if ($type && $type !== 'all' && isset(self::FIELD_TYPES[$type])) {
            return self::getOperatorsForType($type);
        }

        return self::OPERATORS;
    }

    /**
     * Get available context fields for conditions (legacy method)
     *
     * @deprecated Use getFieldsForTrigger() instead
     * @return array
     */
    public static function getAvailableFields(): array
    {
        return self::getFieldsForTrigger(null);
    }
}
