<?php

namespace WPOutreach\Automation;

/**
 * Automation Engine
 *
 * Handles triggering automations and processing the automation queue.
 * Uses TriggerRegistry and ActionRegistry for extensible trigger/action support.
 *
 * @since 1.0.0
 */
class AutomationEngine
{
    /**
     * Maximum retry attempts before marking as failed
     */
    public const MAX_RETRIES = 3;

    /**
     * Base delay in seconds for exponential backoff (5 minutes)
     */
    public const RETRY_BASE_DELAY = 300;
    /**
     * Register WordPress hooks for automation triggers
     */
    public static function init(): void
    {
        // Initialize registries
        TriggerRegistry::init();
        ActionRegistry::init();

        // Hook into core subscriber events and delegate to TriggerRegistry
        add_action('wp_outreach_subscriber_created', [self::class, 'onSubscriberCreated'], 10, 2);
        add_action('wp_outreach_subscriber_added_to_list', [self::class, 'onSubscriberAddedToList'], 10, 3);
        add_action('wp_outreach_subscriber_removed_from_list', [self::class, 'onSubscriberRemovedFromList'], 10, 3);
        add_action('wp_outreach_tag_added_to_subscriber', [self::class, 'onTagAdded'], 10, 3);
        add_action('wp_outreach_tag_removed_from_subscriber', [self::class, 'onTagRemoved'], 10, 3);

        // Generic trigger hook for third-party plugins
        add_action('wp_outreach_trigger', [self::class, 'onGenericTrigger'], 10, 3);

        // Cron hook for processing queue
        add_action('wp_outreach_process_automation_queue', [self::class, 'processQueue']);
    }

    /**
     * Handle generic trigger from third-party plugins
     *
     * Usage:
     *   do_action('wp_outreach_trigger', 'wpdm_package_updated', $subscriber_id, $context);
     *
     * @param string $trigger_id   The trigger ID
     * @param int    $subscriber_id The subscriber ID
     * @param array  $context      Context data for matching
     */
    public static function onGenericTrigger(string $trigger_id, int $subscriber_id, array $context = []): void
    {
        TriggerRegistry::fire($trigger_id, $subscriber_id, $context);
    }

    /**
     * Trigger automations when a new subscriber is created
     */
    public static function onSubscriberCreated(int $subscriber_id, array $subscriber_data): void
    {
        TriggerRegistry::fire('subscriber_created', $subscriber_id, $subscriber_data);
    }

    /**
     * Trigger automations when a subscriber joins a list
     */
    public static function onSubscriberAddedToList(int $subscriber_id, int $list_id, array $subscriber_data): void
    {
        TriggerRegistry::fire('subscriber_joins_list', $subscriber_id, [
            'list_id' => $list_id,
            ...$subscriber_data,
        ]);
    }

    /**
     * Trigger automations when a subscriber leaves a list
     */
    public static function onSubscriberRemovedFromList(int $subscriber_id, int $list_id, array $subscriber_data): void
    {
        TriggerRegistry::fire('subscriber_leaves_list', $subscriber_id, [
            'list_id' => $list_id,
            ...$subscriber_data,
        ]);
    }

    /**
     * Trigger automations when a tag is added to a subscriber
     */
    public static function onTagAdded(int $subscriber_id, int $tag_id, array $subscriber_data): void
    {
        TriggerRegistry::fire('subscriber_added_tag', $subscriber_id, [
            'tag_id' => $tag_id,
            ...$subscriber_data,
        ]);
    }

    /**
     * Trigger automations when a tag is removed from a subscriber
     */
    public static function onTagRemoved(int $subscriber_id, int $tag_id, array $subscriber_data): void
    {
        TriggerRegistry::fire('subscriber_removed_tag', $subscriber_id, [
            'tag_id' => $tag_id,
            ...$subscriber_data,
        ]);
    }

    /**
     * Enqueue a subscriber into an automation
     *
     * @param int   $automation_id The automation ID
     * @param int   $subscriber_id The subscriber ID
     * @param array $context       Additional context data to store
     * @return int|false Insert ID or false on failure
     */
    public static function enqueueSubscriber(int $automation_id, int $subscriber_id, array $context = []): int|false
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        // Check if subscriber is already in this automation
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table}
             WHERE automation_id = %d AND subscriber_id = %d AND status = 'active'",
            $automation_id,
            $subscriber_id
        ));

        if ($existing) {
            return false; // Already in automation
        }

        // Get automation to check steps
        $automations_table = $wpdb->prefix . 'outreach_automations';
        $automation = $wpdb->get_row($wpdb->prepare(
            "SELECT steps FROM {$automations_table} WHERE id = %d",
            $automation_id
        ));

        if (!$automation) {
            return false;
        }

        $steps = json_decode($automation->steps, true) ?: [];
        if (empty($steps)) {
            return false; // No steps to process
        }

        // Calculate next run time based on first step
        $next_run_at = self::calculateNextRunTime($steps[0]);

        $result = $wpdb->insert($table, [
            'automation_id' => $automation_id,
            'subscriber_id' => $subscriber_id,
            'current_step' => '0', // String path: "0", "1.yes.0", etc.
            'status' => 'active',
            'next_run_at' => $next_run_at,
            'started_at' => current_time('mysql'),
            'context' => !empty($context) ? wp_json_encode($context) : null,
        ]);

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Calculate next run time based on step configuration
     *
     * @param array $step
     * @return string MySQL datetime
     */
    private static function calculateNextRunTime(array $step): string
    {
        $now = current_time('timestamp');

        // If it's a wait step, calculate the delay
        if ($step['type'] === 'wait' && !empty($step['config'])) {
            $duration = (int) ($step['config']['duration'] ?? 1);
            $unit = $step['config']['unit'] ?? 'days';

            switch ($unit) {
                case 'minutes':
                    $now += $duration * 60;
                    break;
                case 'hours':
                    $now += $duration * 3600;
                    break;
                case 'weeks':
                    $now += $duration * 604800;
                    break;
                case 'days':
                default:
                    $now += $duration * 86400;
                    break;
            }
        }

        return gmdate('Y-m-d H:i:s', $now);
    }

    /**
     * Process the automation queue
     *
     * Called by cron every minute.
     * Supports branching via step paths like "0", "1.yes.0", "1.no.0", "2"
     *
     * @param int $batch_size Number of items to process
     * @return int Number of items processed
     */
    public static function processQueue(int $batch_size = 50): int
    {
        global $wpdb;
        $queue_table = $wpdb->prefix . 'outreach_automation_queue';
        $automations_table = $wpdb->prefix . 'outreach_automations';
        $now = current_time('mysql');

        // Get items that are due to run
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT q.*, a.steps
             FROM {$queue_table} q
             INNER JOIN {$automations_table} a ON q.automation_id = a.id
             WHERE q.status = 'active'
             AND q.next_run_at <= %s
             AND a.status = 'active'
             ORDER BY q.next_run_at ASC
             LIMIT %d",
            $now,
            $batch_size
        ));

        $processed = 0;

        foreach ($items as $item) {
            $steps = json_decode($item->steps, true) ?: [];
            $step_path = $item->current_step ?? '0'; // Can be "0", "1.yes.0", etc.
            $context = !empty($item->context) ? json_decode($item->context, true) : [];
            $attempts = (int) ($item->attempts ?? 0);

            // Fetch subscriber data and add to context for condition evaluation
            $subscriber = self::getSubscriberData($item->subscriber_id);
            if ($subscriber) {
                // Add subscriber data nested under 'subscriber' key for conditions like subscriber.email
                $context['subscriber'] = [
                    'id'         => $subscriber->id,
                    'email'      => $subscriber->email,
                    'first_name' => $subscriber->first_name ?? '',
                    'last_name'  => $subscriber->last_name ?? '',
                    'status'     => $subscriber->status ?? 'active',
                    'source'     => $subscriber->source ?? '',
                ];
                // Also add top-level aliases for backward compatibility
                $context['email'] = $context['email'] ?? $subscriber->email;
                $context['first_name'] = $context['first_name'] ?? ($subscriber->first_name ?? '');
                $context['last_name'] = $context['last_name'] ?? ($subscriber->last_name ?? '');
            }

            // Get current step using path
            $step = self::getStepByPath($steps, $step_path);

            if ($step === null) {
                // No more steps, mark as completed
                self::completeQueueItem($item->id);
                $processed++;
                continue;
            }

            $error_message = null;

            try {
                // Execute the step using ActionRegistry
                $result = ActionRegistry::execute(
                    $step['type'],
                    $step['config'] ?? [],
                    $item->subscriber_id,
                    $item->automation_id,
                    $context
                );

                // Handle result - can be true, false, or array with 'success'/'condition_met'
                $success = true;
                $condition_met = true;

                if (is_array($result)) {
                    $success = $result['success'] ?? true;
                    $error_message = $result['error'] ?? null;
                    if (isset($result['condition_met'])) {
                        $condition_met = $result['condition_met'];
                    }
                } else {
                    $success = (bool) $result;
                }
            } catch (\Throwable $e) {
                $success = false;
                $error_message = $e->getMessage();
            }

            if ($success) {
                // Determine next step path
                $next_path = self::getNextStepPath($steps, $step_path, $step, $condition_met);

                if ($next_path === null) {
                    // No more steps, mark as completed
                    self::completeQueueItem($item->id);
                } else {
                    // Get next step to calculate run time
                    $next_step = self::getStepByPath($steps, $next_path);
                    $next_run_at = $next_step ? self::calculateNextRunTime($next_step) : current_time('mysql');

                    $wpdb->update(
                        $queue_table,
                        [
                            'current_step' => $next_path,
                            'next_run_at' => $next_run_at,
                            'attempts' => 0,
                            'last_error' => null,
                        ],
                        ['id' => $item->id]
                    );
                }
            } else {
                // Handle failure with retry logic
                self::handleStepFailure($item->id, $attempts, $error_message);
            }

            $processed++;
        }

        return $processed;
    }

    /**
     * Get a step by its path
     *
     * Path examples:
     * - "0" → steps[0]
     * - "1" → steps[1]
     * - "1.yes.0" → steps[1]['yes_steps'][0]
     * - "1.no.0" → steps[1]['no_steps'][0]
     * - "2.yes.1.no.0" → steps[2]['yes_steps'][1]['no_steps'][0]
     *
     * @param array  $steps All steps
     * @param string $path  Step path
     * @return array|null Step data or null if not found
     */
    private static function getStepByPath(array $steps, string $path): ?array
    {
        $parts = explode('.', $path);
        $current = $steps;

        foreach ($parts as $i => $part) {
            if ($part === 'yes') {
                // Next part is index in yes_steps (stored in config)
                continue;
            } elseif ($part === 'no') {
                // Next part is index in no_steps (stored in config)
                continue;
            } else {
                $index = (int) $part;

                // Check if previous part was 'yes' or 'no'
                if ($i > 0) {
                    $prev = $parts[$i - 1];
                    if ($prev === 'yes') {
                        // Branch steps are in config.yes_steps
                        $config = $current['config'] ?? [];
                        $current = $config['yes_steps'] ?? [];
                    } elseif ($prev === 'no') {
                        // Branch steps are in config.no_steps
                        $config = $current['config'] ?? [];
                        $current = $config['no_steps'] ?? [];
                    }
                }

                if (!isset($current[$index])) {
                    return null;
                }
                $current = $current[$index];
            }
        }

        return is_array($current) && isset($current['type']) ? $current : null;
    }

    /**
     * Get the next step path after executing current step
     *
     * @param array  $steps         All steps
     * @param string $current_path  Current step path
     * @param array  $current_step  Current step data
     * @param bool   $condition_met For condition steps, whether condition passed
     * @return string|null Next step path or null if no more steps
     */
    private static function getNextStepPath(
        array $steps,
        string $current_path,
        array $current_step,
        bool $condition_met = true
    ): ?string {
        // If this is a condition step with branches, enter the appropriate branch
        if ($current_step['type'] === 'condition') {
            $branch = $condition_met ? 'yes' : 'no';
            // Branch steps are stored in config.yes_steps / config.no_steps
            $config = $current_step['config'] ?? [];
            $branch_steps = $config[$branch . '_steps'] ?? [];

            if (!empty($branch_steps)) {
                // Enter the branch at step 0
                return $current_path . '.' . $branch . '.0';
            }
            // No branch steps, continue to next main step
        }

        // Find the next step in the current context
        return self::findNextPath($steps, $current_path);
    }

    /**
     * Find the next step path by incrementing or exiting branches
     *
     * @param array  $steps All steps
     * @param string $path  Current path
     * @return string|null Next path or null if done
     */
    private static function findNextPath(array $steps, string $path): ?string
    {
        $parts = explode('.', $path);

        // Try to increment the last index
        while (!empty($parts)) {
            $last_index = count($parts) - 1;

            // The last part should be a number
            if (is_numeric($parts[$last_index])) {
                $next_index = (int) $parts[$last_index] + 1;
                $parts[$last_index] = (string) $next_index;
                $next_path = implode('.', $parts);

                // Check if this path exists
                if (self::getStepByPath($steps, $next_path) !== null) {
                    return $next_path;
                }

                // This index doesn't exist, exit the branch
                // Remove the last 2 parts (branch type + index) to go up a level
                if (count($parts) >= 3) {
                    // Remove: index, branch_type (yes/no)
                    array_pop($parts); // Remove index
                    array_pop($parts); // Remove 'yes' or 'no'

                    // Now increment from parent level
                    continue;
                } else {
                    // We're at the main level and there's no next step
                    return null;
                }
            } else {
                // Shouldn't happen, but handle gracefully
                array_pop($parts);
            }
        }

        return null;
    }

    /**
     * Handle step failure with exponential backoff retry
     *
     * @param int         $queue_id      Queue item ID
     * @param int         $attempts      Current attempt count
     * @param string|null $error_message Error message
     */
    private static function handleStepFailure(int $queue_id, int $attempts, ?string $error_message): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        $new_attempts = $attempts + 1;

        if ($new_attempts >= self::MAX_RETRIES) {
            // Max retries reached, mark as failed
            self::failQueueItem($queue_id, $error_message);
            return;
        }

        // Calculate exponential backoff delay: base * 2^attempts
        // Attempt 1: 5 min, Attempt 2: 10 min, Attempt 3: 20 min
        $delay = self::RETRY_BASE_DELAY * pow(2, $attempts);
        $next_run_at = gmdate('Y-m-d H:i:s', current_time('timestamp') + $delay);

        $wpdb->update(
            $table,
            [
                'attempts' => $new_attempts,
                'last_error' => $error_message,
                'next_run_at' => $next_run_at,
            ],
            ['id' => $queue_id]
        );

        do_action('wp_outreach_automation_step_retry', [
            'queue_id' => $queue_id,
            'attempt' => $new_attempts,
            'next_run_at' => $next_run_at,
            'error' => $error_message,
        ]);
    }

    /**
     * Mark a queue item as failed
     *
     * @param int         $id            Queue item ID
     * @param string|null $error_message Error message
     */
    private static function failQueueItem(int $id, ?string $error_message = null): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        $wpdb->update(
            $table,
            [
                'status' => 'failed',
                'last_error' => $error_message,
                'failed_at' => current_time('mysql'),
            ],
            ['id' => $id]
        );

        do_action('wp_outreach_automation_step_failed', [
            'queue_id' => $id,
            'error' => $error_message,
        ]);
    }

    /**
     * Mark a queue item as completed
     */
    private static function completeQueueItem(int $id): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        $wpdb->update(
            $table,
            ['status' => 'completed'],
            ['id' => $id]
        );
    }

    /**
     * Get subscriber data by ID
     *
     * @param int $subscriber_id
     * @return object|null
     */
    private static function getSubscriberData(int $subscriber_id): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';

        return $wpdb->get_row($wpdb->prepare(
            "SELECT id, email, first_name, last_name, status, source FROM {$table} WHERE id = %d",
            $subscriber_id
        ));
    }

    /**
     * Cancel a subscriber's automation
     */
    public static function cancelSubscriber(int $automation_id, int $subscriber_id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        $result = $wpdb->update(
            $table,
            ['status' => 'cancelled'],
            [
                'automation_id' => $automation_id,
                'subscriber_id' => $subscriber_id,
                'status' => 'active',
            ]
        );

        return $result !== false;
    }

    /**
     * Get queue stats for an automation
     */
    public static function getStats(int $automation_id): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        // Get all stats in a single query for better performance
        $stats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
             FROM {$table}
             WHERE automation_id = %d",
            $automation_id
        ));

        return [
            'active' => (int) ($stats->active ?? 0),
            'completed' => (int) ($stats->completed ?? 0),
            'cancelled' => (int) ($stats->cancelled ?? 0),
            'failed' => (int) ($stats->failed ?? 0),
        ];
    }

    /**
     * Retry a failed queue item manually
     *
     * @param int $queue_id Queue item ID
     * @return bool True if successfully reset for retry
     */
    public static function retryFailedItem(int $queue_id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automation_queue';

        // Only retry failed items
        $item = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d AND status = 'failed'",
            $queue_id
        ));

        if (!$item) {
            return false;
        }

        $result = $wpdb->update(
            $table,
            [
                'status' => 'active',
                'attempts' => 0,
                'last_error' => null,
                'failed_at' => null,
                'next_run_at' => current_time('mysql'),
            ],
            ['id' => $queue_id]
        );

        return $result !== false;
    }
}
