<?php

namespace WPOutreach\Automation;

use WPOutreach\Mailer\Queue\QueueManager;
use WPOutreach\Mailer\Template\TemplateEngine;

/**
 * Action Registry
 *
 * Manages automation actions that can be registered by WP Outreach core
 * and third-party plugins.
 *
 * @since 1.0.0
 *
 * ## Third-Party Integration
 *
 * To register a custom action from your plugin:
 *
 * ```php
 * add_action('wp_outreach_register_actions', function($registry) {
 *     $registry->register([
 *         'id'          => 'my_plugin_custom_action',
 *         'name'        => 'Do Something Custom',
 *         'description' => 'Performs a custom action in my plugin',
 *         'group'       => 'My Plugin',
 *         'icon'        => 'settings',
 *         'config_fields' => [
 *             [
 *                 'key'   => 'action_type',
 *                 'label' => 'Action Type',
 *                 'type'  => 'select',
 *                 'options' => [
 *                     ['value' => 'grant_access', 'label' => 'Grant Access'],
 *                     ['value' => 'revoke_access', 'label' => 'Revoke Access'],
 *                 ],
 *             ],
 *         ],
 *         'executor' => function($config, $subscriber_id, $context) {
 *             // Perform your action here
 *             // Return true on success, false on failure
 *             $action_type = $config['action_type'] ?? 'grant_access';
 *
 *             if ($action_type === 'grant_access') {
 *                 my_plugin_grant_access($subscriber_id);
 *             }
 *
 *             return true;
 *         },
 *     ]);
 * });
 * ```
 */
class ActionRegistry
{
    /**
     * Registered actions
     *
     * @var array<string, array>
     */
    private static array $actions = [];

    /**
     * Whether actions have been initialized
     *
     * @var bool
     */
    private static bool $initialized = false;

    /**
     * Initialize the registry and register core actions
     */
    public static function init(): void
    {
        if (self::$initialized) {
            return;
        }

        self::$initialized = true;

        // Register core actions
        self::registerCoreActions();

        // Allow third-party plugins to register their actions
        do_action('wp_outreach_register_actions', new self());
    }

    /**
     * Register a new action
     *
     * @param array $action {
     *     Action configuration.
     *
     *     @type string   $id            Required. Unique action identifier.
     *     @type string   $name          Required. Human-readable action name.
     *     @type string   $description   Optional. Detailed description.
     *     @type string   $group         Optional. Group name for UI. Default: 'General'.
     *     @type string   $icon          Optional. Icon name or SVG. Default: 'zap'.
     *     @type array    $config_fields Optional. Configuration fields for the action.
     *     @type callable $executor      Required. Function to execute the action.
     *                                   Receives ($config, $subscriber_id, $context).
     * }
     * @return bool True on success, false if action ID already exists or no executor.
     */
    public function register(array $action): bool
    {
        if (empty($action['id'])) {
            return false;
        }

        $id = sanitize_key($action['id']);

        if (isset(self::$actions[$id])) {
            return false; // Already registered
        }

        if (!isset($action['executor']) || !is_callable($action['executor'])) {
            // Allow registration without executor for core actions (they use switch/case)
            if (($action['source'] ?? 'third-party') !== 'core') {
                return false;
            }
        }

        self::$actions[$id] = [
            'id'            => $id,
            'name'          => $action['name'] ?? $id,
            'description'   => $action['description'] ?? '',
            'group'         => $action['group'] ?? 'General',
            'icon'          => $action['icon'] ?? 'zap',
            'config_fields' => $action['config_fields'] ?? [],
            'executor'      => $action['executor'] ?? null,
            'source'        => $action['source'] ?? 'third-party',
            'has_branches'  => $action['has_branches'] ?? false,
        ];

        return true;
    }

    /**
     * Unregister an action
     *
     * @param string $id Action ID
     * @return bool True if removed, false if not found.
     */
    public function unregister(string $id): bool
    {
        if (!isset(self::$actions[$id])) {
            return false;
        }

        unset(self::$actions[$id]);
        return true;
    }

    /**
     * Get all registered actions
     *
     * @return array<string, array>
     */
    public static function getAll(): array
    {
        self::init();
        return self::$actions;
    }

    /**
     * Get actions grouped by their group name
     *
     * @return array<string, array>
     */
    public static function getGrouped(): array
    {
        self::init();
        $grouped = [];

        foreach (self::$actions as $action) {
            $group = $action['group'];
            if (!isset($grouped[$group])) {
                $grouped[$group] = [];
            }
            $grouped[$group][] = $action;
        }

        return $grouped;
    }

    /**
     * Get a single action by ID
     *
     * @param string $id Action ID
     * @return array|null Action data or null if not found.
     */
    public static function get(string $id): ?array
    {
        self::init();
        return self::$actions[$id] ?? null;
    }

    /**
     * Check if an action exists
     *
     * @param string $id Action ID
     * @return bool
     */
    public static function exists(string $id): bool
    {
        self::init();
        return isset(self::$actions[$id]);
    }

    /**
     * Get actions for REST API response
     *
     * @return array
     */
    public static function forApi(): array
    {
        self::init();
        $grouped = self::getGrouped();
        $result = [];

        foreach ($grouped as $group => $actions) {
            $result[] = [
                'group' => $group,
                'actions' => array_map(function ($action) {
                    return [
                        'id'            => $action['id'],
                        'name'          => $action['name'],
                        'description'   => $action['description'],
                        'icon'          => $action['icon'],
                        'config_fields' => $action['config_fields'],
                        'has_branches'  => $action['has_branches'] ?? false,
                    ];
                }, $actions),
            ];
        }

        return $result;
    }

    /**
     * Execute an action
     *
     * @param string $action_id    The action ID.
     * @param array  $config       Action configuration.
     * @param int    $subscriber_id The subscriber ID.
     * @param int    $automation_id The automation ID (for logging).
     * @param array  $context      Additional context data.
     * @return bool|array True on success, false on failure. For condition actions,
     *                    returns array with 'success' and 'condition_met' keys.
     */
    public static function execute(
        string $action_id,
        array $config,
        int $subscriber_id,
        int $automation_id = 0,
        array $context = []
    ): bool|array {
        self::init();

        $action = self::get($action_id);

        if (!$action) {
            do_action('wp_outreach_action_not_found', $action_id, $subscriber_id, $config);
            return false;
        }

        // Log action start
        do_action('wp_outreach_action_started', [
            'action_id'      => $action_id,
            'subscriber_id'  => $subscriber_id,
            'automation_id'  => $automation_id,
            'config'         => $config,
        ]);

        try {
            // Use custom executor if provided
            if (is_callable($action['executor'])) {
                $result = call_user_func($action['executor'], $config, $subscriber_id, $context);
            } else {
                // Fall back to core action handlers
                $result = self::executeCoreAction($action_id, $config, $subscriber_id, $automation_id, $context);
            }

            // Log action result
            do_action('wp_outreach_action_completed', [
                'action_id'      => $action_id,
                'subscriber_id'  => $subscriber_id,
                'automation_id'  => $automation_id,
                'success'        => $result,
            ]);

            return $result;
        } catch (\Exception $e) {
            do_action('wp_outreach_action_failed', [
                'action_id'      => $action_id,
                'subscriber_id'  => $subscriber_id,
                'automation_id'  => $automation_id,
                'error'          => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Execute core WP Outreach actions
     *
     * @return bool|array Bool for most actions, array with 'condition_met' for conditions
     */
    private static function executeCoreAction(
        string $action_id,
        array $config,
        int $subscriber_id,
        int $automation_id,
        array $context = []
    ): bool|array {
        switch ($action_id) {
            case 'send_email':
                return self::executeSendEmail($config, $subscriber_id, $automation_id, $context);

            case 'email_admin':
                return self::executeEmailAdmin($config, $context);

            case 'wait':
                return true; // Wait steps are handled by scheduling

            case 'add_tag':
                return self::executeAddTag($config, $subscriber_id);

            case 'remove_tag':
                return self::executeRemoveTag($config, $subscriber_id);

            case 'add_to_list':
                return self::executeAddToList($config, $subscriber_id);

            case 'remove_from_list':
                return self::executeRemoveFromList($config, $subscriber_id);

            case 'condition':
                // Condition steps return evaluation result for branching
                return self::executeCondition($config, $context);

            case 'webhook_call':
                return self::executeWebhookCall($config, $subscriber_id, $context);

            default:
                return true; // Unknown action, skip
        }
    }

    /**
     * Register core WP Outreach actions
     */
    private static function registerCoreActions(): void
    {
        $registry = new self();

        // Communication actions
        $registry->register([
            'id'          => 'send_email',
            'name'        => __('Send Email', 'outreach'),
            'description' => __('Send an email to the subscriber', 'outreach'),
            'group'       => __('Communication', 'outreach'),
            'icon'        => 'mail',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'recipient_type',
                    'label'       => __('Send To', 'outreach'),
                    'type'        => 'select',
                    'default'     => 'auto',
                    'options'     => [
                        ['value' => 'auto', 'label' => __('Automatic (based on trigger)', 'outreach')],
                        ['value' => 'lists', 'label' => __('Specific lists', 'outreach')],
                        ['value' => 'both', 'label' => __('Trigger targets + Specific lists', 'outreach')],
                    ],
                    'help_text'   => __('For post triggers: "Automatic" sends to post subscribers. Choose "Specific lists" to target your subscriber lists.', 'outreach'),
                ],
                [
                    'key'              => 'list_ids',
                    'label'            => __('Target Lists', 'outreach'),
                    'type'             => 'multiselect',
                    'options_endpoint' => '/lists',
                    'placeholder'      => __('Select lists...', 'outreach'),
                    'condition'        => ['recipient_type', 'not_auto'],
                ],
                [
                    'key'         => 'template_id',
                    'label'       => __('Email Template', 'outreach'),
                    'type'        => 'select',
                    'placeholder' => __('Compose custom email', 'outreach'),
                    'options_endpoint' => '/templates',
                ],
                [
                    'key'   => 'subject',
                    'label' => __('Subject', 'outreach'),
                    'type'  => 'text',
                    'placeholder' => __('Email subject line', 'outreach'),
                    'condition' => ['template_id', 'empty'],
                ],
                [
                    'key'   => 'preheader',
                    'label' => __('Preheader', 'outreach'),
                    'type'  => 'text',
                    'placeholder' => __('Preview text shown in inbox', 'outreach'),
                    'condition' => ['template_id', 'empty'],
                ],
                [
                    'key'   => 'content',
                    'label' => __('Content', 'outreach'),
                    'type'  => 'textarea',
                    'placeholder' => __('Email body content...', 'outreach'),
                    'condition' => ['template_id', 'empty'],
                ],
            ],
        ]);

        // Email to Admin action (for notifications, alerts, etc.)
        $registry->register([
            'id'          => 'email_admin',
            'name'        => __('Email to Admin', 'outreach'),
            'description' => __('Send notification email to site administrator(s)', 'outreach'),
            'group'       => __('Communication', 'outreach'),
            'icon'        => 'shield',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'to',
                    'label'       => __('Send To', 'outreach'),
                    'type'        => 'text',
                    'placeholder' => get_option('admin_email'),
                    'help_text'   => __('Leave empty to use site admin email. Separate multiple emails with commas.', 'outreach'),
                ],
                [
                    'key'         => 'subject',
                    'label'       => __('Subject', 'outreach'),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => __('Notification subject...', 'outreach'),
                    'help_text'   => __('Supports merge tags from webhook payload: {{webhook_event}}, {{webhook_vulnerability_name}}, etc.', 'outreach'),
                ],
                [
                    'key'         => 'content',
                    'label'       => __('Content', 'outreach'),
                    'type'        => 'textarea',
                    'required'    => true,
                    'placeholder' => __('Notification message...', 'outreach'),
                    'help_text'   => __('Supports merge tags. For webhook data use: {{webhook_*}} pattern', 'outreach'),
                ],
            ],
        ]);

        // Timing actions
        $registry->register([
            'id'          => 'wait',
            'name'        => __('Wait', 'outreach'),
            'description' => __('Wait for a specified duration before continuing', 'outreach'),
            'group'       => __('Timing', 'outreach'),
            'icon'        => 'clock',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'     => 'duration',
                    'label'   => __('Duration', 'outreach'),
                    'type'    => 'number',
                    'default' => 1,
                    'min'     => 1,
                ],
                [
                    'key'     => 'unit',
                    'label'   => __('Unit', 'outreach'),
                    'type'    => 'select',
                    'default' => 'days',
                    'options' => [
                        ['value' => 'minutes', 'label' => __('Minutes', 'outreach')],
                        ['value' => 'hours', 'label' => __('Hours', 'outreach')],
                        ['value' => 'days', 'label' => __('Days', 'outreach')],
                        ['value' => 'weeks', 'label' => __('Weeks', 'outreach')],
                    ],
                ],
            ],
        ]);

        // Segmentation actions
        $registry->register([
            'id'          => 'add_tag',
            'name'        => __('Add Tag', 'outreach'),
            'description' => __('Add a tag to the subscriber', 'outreach'),
            'group'       => __('Segmentation', 'outreach'),
            'icon'        => 'tag',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'              => 'tag_id',
                    'label'            => __('Tag', 'outreach'),
                    'type'             => 'select',
                    'required'         => true,
                    'options_endpoint' => '/tags',
                ],
            ],
        ]);

        $registry->register([
            'id'          => 'remove_tag',
            'name'        => __('Remove Tag', 'outreach'),
            'description' => __('Remove a tag from the subscriber', 'outreach'),
            'group'       => __('Segmentation', 'outreach'),
            'icon'        => 'tag',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'              => 'tag_id',
                    'label'            => __('Tag', 'outreach'),
                    'type'             => 'select',
                    'required'         => true,
                    'options_endpoint' => '/tags',
                ],
            ],
        ]);

        $registry->register([
            'id'          => 'add_to_list',
            'name'        => __('Add to List', 'outreach'),
            'description' => __('Add the subscriber to a list', 'outreach'),
            'group'       => __('Segmentation', 'outreach'),
            'icon'        => 'users',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'              => 'list_id',
                    'label'            => __('List', 'outreach'),
                    'type'             => 'select',
                    'required'         => true,
                    'options_endpoint' => '/lists',
                ],
            ],
        ]);

        $registry->register([
            'id'          => 'remove_from_list',
            'name'        => __('Remove from List', 'outreach'),
            'description' => __('Remove the subscriber from a list', 'outreach'),
            'group'       => __('Segmentation', 'outreach'),
            'icon'        => 'user-minus',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'              => 'list_id',
                    'label'            => __('List', 'outreach'),
                    'type'             => 'select',
                    'required'         => true,
                    'options_endpoint' => '/lists',
                ],
            ],
        ]);

        // Logic/Flow actions - Conditional branching
        $registry->register([
            'id'          => 'condition',
            'name'        => __('Condition', 'outreach'),
            'description' => __('Check conditions and branch the workflow based on result', 'outreach'),
            'group'       => __('Logic', 'outreach'),
            'icon'        => 'git-branch',
            'source'      => 'core',
            'has_branches' => true, // Indicates this action supports branching
            'config_fields' => [
                [
                    'key'     => 'logic',
                    'label'   => __('Match', 'outreach'),
                    'type'    => 'select',
                    'default' => 'and',
                    'options' => [
                        ['value' => 'and', 'label' => __('All conditions (AND)', 'outreach')],
                        ['value' => 'or', 'label' => __('Any condition (OR)', 'outreach')],
                    ],
                ],
                [
                    'key'     => 'conditions',
                    'label'   => __('Conditions', 'outreach'),
                    'type'    => 'condition_builder', // Special type for condition UI
                    'default' => [],
                ],
            ],
        ]);

        // Integration actions - Webhook
        $registry->register([
            'id'          => 'webhook_call',
            'name'        => __('Webhook Call', 'outreach'),
            'description' => __('Send data to an external URL (Zapier, Make, custom APIs)', 'outreach'),
            'group'       => __('Integration', 'outreach'),
            'icon'        => 'webhook',
            'source'      => 'core',
            'config_fields' => [
                [
                    'key'         => 'url',
                    'label'       => __('Webhook URL', 'outreach'),
                    'type'        => 'text',
                    'required'    => true,
                    'placeholder' => 'https://hooks.zapier.com/...',
                ],
                [
                    'key'     => 'method',
                    'label'   => __('HTTP Method', 'outreach'),
                    'type'    => 'select',
                    'default' => 'POST',
                    'options' => [
                        ['value' => 'POST', 'label' => 'POST'],
                        ['value' => 'GET', 'label' => 'GET'],
                        ['value' => 'PUT', 'label' => 'PUT'],
                        ['value' => 'DELETE', 'label' => 'DELETE'],
                    ],
                ],
                [
                    'key'     => 'body_format',
                    'label'   => __('Body Format', 'outreach'),
                    'type'    => 'select',
                    'default' => 'json',
                    'options' => [
                        ['value' => 'json', 'label' => 'JSON'],
                        ['value' => 'form', 'label' => 'Form Data'],
                    ],
                ],
                [
                    'key'         => 'body',
                    'label'       => __('Request Body', 'outreach'),
                    'type'        => 'textarea',
                    'placeholder' => '{"email": "{{email}}", "name": "{{first_name}}"}',
                ],
                [
                    'key'     => 'headers',
                    'label'   => __('Custom Headers', 'outreach'),
                    'type'    => 'key_value', // Array of {key, value} pairs
                    'default' => [],
                ],
                [
                    'key'     => 'retry_on_failure',
                    'label'   => __('Retry on failure', 'outreach'),
                    'type'    => 'checkbox',
                    'default' => true,
                ],
            ],
        ]);

        // Allow filtering core actions
        do_action('wp_outreach_core_actions_registered', $registry);
    }

    /**
     * Execute send_email action
     */
    private static function executeSendEmail(array $config, int $subscriber_id, int $automation_id, array $context = []): bool
    {
        global $wpdb;

        // Get subscriber data
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_subscribers WHERE id = %d",
            $subscriber_id
        ), ARRAY_A);

        if (!$subscriber || $subscriber['status'] !== 'active') {
            return false;
        }

        // Get template or build email
        $subject = '';
        $content = '';

        if (!empty($config['template_id'])) {
            $template = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}outreach_templates WHERE id = %d",
                (int) $config['template_id']
            ));

            if ($template) {
                $subject = $template->subject;
                $content = $template->content;
            }
        } else {
            $subject = $config['subject'] ?? '';
            $content = $config['content'] ?? '';
        }

        if (empty($subject) || empty($content)) {
            return false;
        }

        // Merge subscriber with context for variable parsing
        // Context includes post data like post_title, post_url, etc.
        $merge_data = array_merge($subscriber, $context);

        // Parse variables
        $template_engine = new TemplateEngine();
        $parsed_subject = $template_engine->parseVariables($subject, $merge_data);
        $parsed_content = $template_engine->render($content, $merge_data);

        // Queue the email
        $queue = new QueueManager();
        $queue->add([
            'automation_id' => $automation_id,
            'subscriber_id' => $subscriber_id,
            'to_email' => $subscriber['email'],
            'subject' => $parsed_subject,
            'content' => $parsed_content,
            'priority' => 5,
        ]);

        return true;
    }

    /**
     * Execute add_tag action
     */
    private static function executeAddTag(array $config, int $subscriber_id): bool
    {
        global $wpdb;

        if (empty($config['tag_id'])) {
            return false;
        }

        $table = $wpdb->prefix . 'outreach_subscriber_tag';
        $tag_id = (int) $config['tag_id'];

        // Check if already has tag
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT subscriber_id FROM {$table} WHERE subscriber_id = %d AND tag_id = %d",
            $subscriber_id,
            $tag_id
        ));

        if ($existing) {
            return true; // Already has tag
        }

        $result = $wpdb->insert($table, [
            'subscriber_id' => $subscriber_id,
            'tag_id' => $tag_id,
            'created_at' => current_time('mysql'),
        ]);

        return (bool) $result;
    }

    /**
     * Execute remove_tag action
     */
    private static function executeRemoveTag(array $config, int $subscriber_id): bool
    {
        global $wpdb;

        if (empty($config['tag_id'])) {
            return false;
        }

        $table = $wpdb->prefix . 'outreach_subscriber_tag';
        $result = $wpdb->delete($table, [
            'subscriber_id' => $subscriber_id,
            'tag_id' => (int) $config['tag_id'],
        ]);

        return $result !== false;
    }

    /**
     * Execute add_to_list action
     */
    private static function executeAddToList(array $config, int $subscriber_id): bool
    {
        global $wpdb;

        if (empty($config['list_id'])) {
            return false;
        }

        $table = $wpdb->prefix . 'outreach_subscriber_list';
        $list_id = (int) $config['list_id'];

        // Check if already in list
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT subscriber_id FROM {$table} WHERE subscriber_id = %d AND list_id = %d",
            $subscriber_id,
            $list_id
        ));

        if ($existing) {
            return true; // Already in list
        }

        $result = $wpdb->insert($table, [
            'subscriber_id' => $subscriber_id,
            'list_id' => $list_id,
            'subscribed_at' => current_time('mysql'),
        ]);

        return (bool) $result;
    }

    /**
     * Execute remove_from_list action
     */
    private static function executeRemoveFromList(array $config, int $subscriber_id): bool
    {
        global $wpdb;

        if (empty($config['list_id'])) {
            return false;
        }

        $table = $wpdb->prefix . 'outreach_subscriber_list';
        $result = $wpdb->delete($table, [
            'subscriber_id' => $subscriber_id,
            'list_id' => (int) $config['list_id'],
        ]);

        return $result !== false;
    }

    /**
     * Execute condition action
     *
     * Evaluates conditions using the ConditionEvaluator.
     * Returns an array with condition result for branching support.
     *
     * @param array $config  Condition configuration with 'conditions' array and 'logic'
     * @param array $context Context data (subscriber, form fields, order data, etc.)
     * @return array Array with 'success' (always true) and 'condition_met' (bool)
     */
    private static function executeCondition(array $config, array $context): array
    {
        $conditions = $config['conditions'] ?? [];
        $logic = $config['logic'] ?? 'and';

        // If no conditions, pass by default
        if (empty($conditions)) {
            return [
                'success'       => true,
                'condition_met' => true,
            ];
        }

        // Use the ConditionEvaluator for flexible condition checking
        $evaluator = new ConditionEvaluator();
        $result = $evaluator->evaluateGroup($conditions, $context, $logic);

        return [
            'success'       => true, // Condition action always "succeeds" (it ran without error)
            'condition_met' => $result,
        ];
    }

    /**
     * Get available condition operators for API
     *
     * @return array
     */
    public static function getConditionOperators(): array
    {
        return ConditionEvaluator::getOperators();
    }

    /**
     * Get available condition fields for API
     *
     * @return array
     */
    public static function getConditionFields(): array
    {
        return ConditionEvaluator::getAvailableFields();
    }

    /**
     * Execute webhook_call action
     *
     * Sends HTTP request to external URL with subscriber and context data.
     * Supports merge tags in URL, body, and headers.
     *
     * @param array $config       Webhook configuration.
     * @param int   $subscriber_id The subscriber ID.
     * @param array $context      Trigger context data.
     * @return bool True on success (2xx response), false on failure.
     */
    private static function executeWebhookCall(array $config, int $subscriber_id, array $context = []): bool
    {
        global $wpdb;

        $url = $config['url'] ?? '';
        if (empty($url)) {
            return false;
        }

        // Get subscriber data for merge tag replacement
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_subscribers WHERE id = %d",
            $subscriber_id
        ), ARRAY_A);

        if (!$subscriber) {
            return false;
        }

        // Merge subscriber data with context for variable replacement
        $merge_data = array_merge($subscriber, $context, [
            'subscriber_id' => $subscriber_id,
            'site_name'     => get_bloginfo('name'),
            'site_url'      => home_url(),
            'current_date'  => current_time('Y-m-d'),
            'current_time'  => current_time('H:i:s'),
            'timestamp'     => time(),
        ]);

        // Parse merge tags in URL
        $url = self::parseMergeTags($url, $merge_data);

        // Prepare request args
        $method = strtoupper($config['method'] ?? 'POST');
        $body_format = $config['body_format'] ?? 'json';
        $retry_on_failure = $config['retry_on_failure'] ?? true;

        $args = [
            'method'  => $method,
            'timeout' => 30,
            'headers' => [
                'User-Agent' => 'WP-Outreach/' . (defined('WP_OUTREACH_VERSION') ? WP_OUTREACH_VERSION : '1.0'),
            ],
        ];

        // Add custom headers
        $custom_headers = $config['headers'] ?? [];
        if (is_array($custom_headers)) {
            foreach ($custom_headers as $header) {
                if (!empty($header['key']) && isset($header['value'])) {
                    $header_value = self::parseMergeTags($header['value'], $merge_data);
                    $args['headers'][$header['key']] = $header_value;
                }
            }
        }

        // Add body for POST/PUT/DELETE
        if (in_array($method, ['POST', 'PUT', 'DELETE'])) {
            $body_content = $config['body'] ?? '';

            if (!empty($body_content)) {
                // Parse merge tags in body
                $body_content = self::parseMergeTags($body_content, $merge_data);

                if ($body_format === 'json') {
                    $args['headers']['Content-Type'] = 'application/json';
                    // Validate and re-encode JSON to ensure proper formatting
                    $decoded = json_decode($body_content, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $args['body'] = json_encode($decoded);
                    } else {
                        // If not valid JSON, send as-is (might be intentional)
                        $args['body'] = $body_content;
                    }
                } else {
                    // Form data - parse as query string or key-value
                    $args['headers']['Content-Type'] = 'application/x-www-form-urlencoded';
                    $args['body'] = $body_content;
                }
            } else {
                // Default body with subscriber data
                if ($body_format === 'json') {
                    $args['headers']['Content-Type'] = 'application/json';
                    $args['body'] = json_encode([
                        'email'         => $subscriber['email'],
                        'first_name'    => $subscriber['first_name'] ?? '',
                        'last_name'     => $subscriber['last_name'] ?? '',
                        'subscriber_id' => $subscriber_id,
                        'timestamp'     => time(),
                        'context'       => $context,
                    ]);
                } else {
                    $args['headers']['Content-Type'] = 'application/x-www-form-urlencoded';
                    $args['body'] = http_build_query([
                        'email'         => $subscriber['email'],
                        'first_name'    => $subscriber['first_name'] ?? '',
                        'last_name'     => $subscriber['last_name'] ?? '',
                        'subscriber_id' => $subscriber_id,
                    ]);
                }
            }
        }

        // Execute request with retry logic
        $max_attempts = $retry_on_failure ? 3 : 1;
        $attempt = 0;
        $last_error = '';

        while ($attempt < $max_attempts) {
            $attempt++;

            $response = wp_remote_request($url, $args);

            if (is_wp_error($response)) {
                $last_error = $response->get_error_message();
                // Wait before retry (exponential backoff)
                if ($attempt < $max_attempts) {
                    sleep(pow(2, $attempt - 1)); // 1s, 2s, 4s
                }
                continue;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            // Consider 2xx as success
            if ($response_code >= 200 && $response_code < 300) {
                // Log successful webhook
                do_action('wp_outreach_webhook_success', [
                    'url'           => $url,
                    'method'        => $method,
                    'subscriber_id' => $subscriber_id,
                    'response_code' => $response_code,
                ]);

                return true;
            }

            // Non-2xx response
            $last_error = "HTTP {$response_code}";

            // Don't retry on 4xx client errors (except 429 rate limit)
            if ($response_code >= 400 && $response_code < 500 && $response_code !== 429) {
                break;
            }

            // Wait before retry
            if ($attempt < $max_attempts) {
                sleep(pow(2, $attempt - 1));
            }
        }

        // Log failed webhook
        do_action('wp_outreach_webhook_failed', [
            'url'           => $url,
            'method'        => $method,
            'subscriber_id' => $subscriber_id,
            'error'         => $last_error,
            'attempts'      => $attempt,
        ]);

        return false;
    }

    /**
     * Execute email_admin action
     *
     * Sends notification email to site administrator(s).
     * Useful for alerts, security notifications, and admin notifications from webhooks.
     *
     * @param array $config  Action configuration.
     * @param array $context Trigger context data (includes webhook_payload for webhook triggers).
     * @return bool True on success.
     */
    private static function executeEmailAdmin(array $config, array $context = []): bool
    {
        $subject = $config['subject'] ?? '';
        $content = $config['content'] ?? '';

        if (empty($subject) || empty($content)) {
            return false;
        }

        // Determine recipients
        $to = $config['to'] ?? '';
        if (empty($to)) {
            $to = get_option('admin_email');
        }

        // Support multiple comma-separated emails
        $recipients = array_map('trim', explode(',', $to));
        $recipients = array_filter($recipients, 'is_email');

        if (empty($recipients)) {
            return false;
        }

        // Build merge data from context
        // Flatten webhook_payload into merge data with webhook_ prefix
        $merge_data = [
            'site_name'     => get_bloginfo('name'),
            'site_url'      => home_url(),
            'current_date'  => current_time('Y-m-d'),
            'current_time'  => current_time('H:i:s'),
            'timestamp'     => time(),
        ];

        // Add context data (includes subscriber info if available)
        foreach ($context as $key => $value) {
            if ($key === 'webhook_payload' && is_array($value)) {
                // Flatten webhook payload with webhook_ prefix
                self::flattenArrayToMergeData($value, 'webhook', $merge_data);
                // Add webhook_dataset as HTML table of all payload data
                $merge_data['webhook_dataset'] = self::arrayToHtmlTable($value);
            } elseif (!is_array($value)) {
                $merge_data[$key] = $value;
            }
        }

        // Parse merge tags in subject and content
        $subject = self::parseMergeTags($subject, $merge_data);
        $content = self::parseMergeTags($content, $merge_data);

        // Get general settings for from name/email
        $general_settings = get_option('wp_outreach_general', []);
        $from_name = $general_settings['from_name'] ?? get_bloginfo('name');
        $from_email = $general_settings['from_email'] ?? get_option('admin_email');

        // Build headers
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            "From: {$from_name} <{$from_email}>",
        ];

        // Wrap content in simple HTML
        $html_content = "<!DOCTYPE html>
<html>
<head><meta charset=\"UTF-8\"></head>
<body style=\"font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif; line-height: 1.6; color: #333;\">
<div style=\"max-width: 600px; margin: 0 auto; padding: 20px;\">
{$content}
<hr style=\"border: none; border-top: 1px solid #eee; margin: 30px 0;\">
<p style=\"color: #666; font-size: 12px;\">
This notification was sent by WP Outreach from " . esc_html(get_bloginfo('name')) . "
</p>
</div>
</body>
</html>";

        // Send to each recipient
        $success = true;
        foreach ($recipients as $recipient) {
            $sent = wp_mail($recipient, $subject, $html_content, $headers);
            if (!$sent) {
                $success = false;
            }
        }

        // Log the notification
        do_action('wp_outreach_admin_email_sent', [
            'recipients' => $recipients,
            'subject'    => $subject,
            'success'    => $success,
        ]);

        return $success;
    }

    /**
     * Flatten nested array into merge data with dot notation prefix
     *
     * @param array  $array   The array to flatten.
     * @param string $prefix  The prefix for keys.
     * @param array  &$result The result array to populate.
     * @param int    $depth   Max depth to prevent infinite recursion.
     */
    private static function flattenArrayToMergeData(array $array, string $prefix, array &$result, int $depth = 3): void
    {
        if ($depth <= 0) {
            return;
        }

        foreach ($array as $key => $value) {
            $full_key = $prefix . '_' . $key;

            if (is_array($value)) {
                // For arrays, also add a JSON representation
                $result[$full_key] = wp_json_encode($value);
                // Recursively flatten
                self::flattenArrayToMergeData($value, $full_key, $result, $depth - 1);
            } else {
                $result[$full_key] = (string) $value;
            }
        }
    }

    /**
     * Convert an array to a styled HTML table
     *
     * Creates a formatted table showing all key-value pairs from the array.
     * Nested arrays are displayed as formatted JSON.
     *
     * @param array $data  The array to convert.
     * @param int   $depth Max recursion depth for nested arrays.
     * @return string HTML table.
     */
    private static function arrayToHtmlTable(array $data, int $depth = 2): string
    {
        if (empty($data)) {
            return '<p style="color: #666; font-style: italic;">No data available</p>';
        }

        $html = '<table style="width: 100%; border-collapse: collapse; font-size: 14px; margin: 16px 0;">';
        $html .= '<thead><tr style="background: linear-gradient(135deg, #6366f1 0%, #4f46e5 100%);">';
        $html .= '<th style="text-align: left; padding: 12px 16px; color: #fff; font-weight: 600; border-radius: 8px 0 0 0;">Field</th>';
        $html .= '<th style="text-align: left; padding: 12px 16px; color: #fff; font-weight: 600; border-radius: 0 8px 0 0;">Value</th>';
        $html .= '</tr></thead><tbody>';

        $row_count = 0;
        foreach ($data as $key => $value) {
            $bg_color = ($row_count % 2 === 0) ? '#f8fafc' : '#ffffff';
            $html .= '<tr style="background: ' . $bg_color . ';">';
            $html .= '<td style="padding: 10px 16px; border-bottom: 1px solid #e2e8f0; font-weight: 500; color: #374151; vertical-align: top;">';
            $html .= esc_html($key);
            $html .= '</td>';
            $html .= '<td style="padding: 10px 16px; border-bottom: 1px solid #e2e8f0; color: #4b5563; word-break: break-word;">';

            if (is_array($value)) {
                if ($depth > 0 && !empty($value)) {
                    // For nested arrays, show as nested table or formatted JSON
                    if (self::isAssociativeArray($value)) {
                        $html .= self::arrayToHtmlTable($value, $depth - 1);
                    } else {
                        // Sequential array - show as list
                        $html .= '<ul style="margin: 0; padding-left: 20px;">';
                        foreach ($value as $item) {
                            $html .= '<li style="margin: 4px 0;">';
                            if (is_array($item)) {
                                $html .= '<code style="background: #f1f5f9; padding: 2px 6px; border-radius: 4px; font-size: 12px;">';
                                $html .= esc_html(wp_json_encode($item, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                                $html .= '</code>';
                            } else {
                                $html .= esc_html((string) $item);
                            }
                            $html .= '</li>';
                        }
                        $html .= '</ul>';
                    }
                } else {
                    // Deep nested or empty - show as JSON
                    $html .= '<code style="background: #f1f5f9; padding: 4px 8px; border-radius: 4px; font-size: 12px; display: block; white-space: pre-wrap;">';
                    $html .= esc_html(wp_json_encode($value, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                    $html .= '</code>';
                }
            } elseif (is_bool($value)) {
                $badge_color = $value ? '#10b981' : '#ef4444';
                $badge_text = $value ? 'true' : 'false';
                $html .= '<span style="background: ' . $badge_color . '; color: #fff; padding: 2px 8px; border-radius: 4px; font-size: 12px; font-weight: 500;">';
                $html .= $badge_text;
                $html .= '</span>';
            } elseif (is_numeric($value)) {
                $html .= '<span style="font-family: monospace; color: #6366f1; font-weight: 500;">';
                $html .= esc_html((string) $value);
                $html .= '</span>';
            } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                $html .= '<a href="' . esc_url($value) . '" style="color: #6366f1; text-decoration: underline;" target="_blank">';
                $html .= esc_html($value);
                $html .= '</a>';
            } else {
                $html .= esc_html((string) $value);
            }

            $html .= '</td></tr>';
            $row_count++;
        }

        $html .= '</tbody></table>';

        return $html;
    }

    /**
     * Check if array is associative (has string keys)
     *
     * @param array $arr The array to check.
     * @return bool True if associative, false if sequential.
     */
    private static function isAssociativeArray(array $arr): bool
    {
        if (empty($arr)) {
            return false;
        }
        return array_keys($arr) !== range(0, count($arr) - 1);
    }

    /**
     * Parse merge tags in a string
     *
     * Replaces {{variable}} and {{variable|fallback}} patterns.
     *
     * @param string $content The content with merge tags.
     * @param array  $data    The data for replacement.
     * @return string Parsed content.
     */
    private static function parseMergeTags(string $content, array $data): string
    {
        return preg_replace_callback(
            '/\{\{\s*([a-z_\.]+)(?:\s*\|\s*([^}]*))?\s*\}\}/i',
            function ($matches) use ($data) {
                $key = strtolower(trim($matches[1]));
                $fallback = isset($matches[2]) ? trim($matches[2]) : '';

                // Check for nested key (e.g., context.package_title)
                if (strpos($key, '.') !== false) {
                    $parts = explode('.', $key, 2);
                    if (isset($data[$parts[0]]) && is_array($data[$parts[0]])) {
                        $value = $data[$parts[0]][$parts[1]] ?? '';
                    } else {
                        $value = '';
                    }
                } else {
                    $value = $data[$key] ?? '';
                }

                // Use fallback if empty
                if ($value === '' && $fallback !== '') {
                    return $fallback;
                }

                // Convert arrays to JSON
                if (is_array($value)) {
                    return json_encode($value);
                }

                return (string) $value;
            },
            $content
        );
    }
}
