<?php

namespace WPOutreach\Admin;

/**
 * License validation and management
 */
class License {

    /**
     * License server URL
     */
    private const LICENSE_SERVER = 'https://wpoutreach.com/';

    /**
     * Product ID for license validation
     */
    private const PRODUCT_ID = 'outreach';

    /**
     * Option keys
     */
    private const OPTION_KEY = 'wp_outreach_license_key';
    private const OPTION_STATUS = 'wp_outreach_license_status';
    private const OPTION_DATA = 'wp_outreach_license_data';

    /**
     * Check if license is valid
     */
    public static function is_valid(): bool {
        $status = get_option(self::OPTION_STATUS, '');
        return $status === 'valid';
    }

    /**
     * Get license key (masked for display)
     */
    public static function get_key(bool $masked = true): string {
        $key = get_option(self::OPTION_KEY, '');

        if ($masked && !empty($key) && strlen($key) > 8) {
            // Show first 4 and last 4 characters
            return substr($key, 0, 4) . str_repeat('*', strlen($key) - 8) . substr($key, -4);
        }

        return $key;
    }

    /**
     * Get license data
     */
    public static function get_data(): array {
        return get_option(self::OPTION_DATA, []);
    }

    /**
     * Validate and activate license
     */
    public static function validate(string $license_key): array {
        $license_key = sanitize_text_field(trim($license_key));

        if (empty($license_key)) {
            return [
                'success' => false,
                'message' => 'Please enter a license key',
            ];
        }

        // Get current domain
        $domain = parse_url(home_url(), PHP_URL_HOST);

        // Make API request to license server
        $response = wp_remote_post(self::LICENSE_SERVER, [
            'timeout' => 30,
            'body' => [
                'wpdmLicense' => 'validate',
                'licenseKey' => $license_key,
                'domain' => $domain,
                'productId' => self::PRODUCT_ID,
            ],
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'message' => 'Could not connect to license server: ' . $response->get_error_message(),
            ];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data) {
            return [
                'success' => false,
                'message' => 'Invalid response from license server',
            ];
        }

        // Check if license is valid
        if (isset($data['status']) && strtoupper($data['status']) === 'VALID') {
            // Format expiration date
            $expire_date = 'Lifetime';
            $expire_timestamp = 0;

            if (!empty($data['expire']) && $data['expire'] > 0) {
                $expire_timestamp = (int) $data['expire'];
                $expire_date = date_i18n(get_option('date_format'), $expire_timestamp);
            }

            // Save license data (legacy format)
            update_option(self::OPTION_KEY, $license_key);
            update_option(self::OPTION_STATUS, 'valid');
            update_option(self::OPTION_DATA, [
                'order_id' => $data['order_id'] ?? '',
                'expire_date' => $expire_date,
                'expire_timestamp' => $expire_timestamp,
                'activation_date' => $data['activation_date'] ?? date('Y-m-d'),
                'order_status' => $data['order_status'] ?? 'active',
                'auto_renew' => $data['auto_renew'] ?? 0,
                'activated_at' => current_time('mysql'),
            ]);

            // Save in FeatureManager format for feature checks
            update_option('wp_outreach_license', [
                'key' => $license_key,
                'status' => 'active',
                'expires_at' => $expire_timestamp > 0 ? date('Y-m-d H:i:s', $expire_timestamp) : null,
            ]);

            // Clear FeatureManager cache
            \WPOutreach\License\FeatureManager::clearCache();

            return [
                'success' => true,
                'message' => 'License activated successfully',
                'order_id' => $data['order_id'] ?? '',
                'expire_date' => $expire_date,
            ];
        }

        // License validation failed
        $error_message = $data['message'] ?? $data['status'] ?? 'License validation failed';

        return [
            'success' => false,
            'message' => $error_message,
        ];
    }

    /**
     * Deactivate license
     */
    public static function deactivate(): array {
        // Get current license key for server notification (optional)
        $license_key = get_option(self::OPTION_KEY, '');
        $domain = parse_url(home_url(), PHP_URL_HOST);

        // Optionally notify server about deactivation
        if (!empty($license_key)) {
            wp_remote_post(self::LICENSE_SERVER, [
                'timeout' => 10,
                'body' => [
                    'wpdmLicense' => 'deactivate',
                    'licenseKey' => $license_key,
                    'domain' => $domain,
                    'productId' => self::PRODUCT_ID,
                ],
            ]);
        }

        // Delete all license options
        delete_option(self::OPTION_KEY);
        delete_option(self::OPTION_STATUS);
        delete_option(self::OPTION_DATA);
        delete_option('wp_outreach_license');

        // Clear FeatureManager cache
        \WPOutreach\License\FeatureManager::clearCache();

        return [
            'success' => true,
            'message' => 'License deactivated successfully',
        ];
    }

    /**
     * Get full license info for API response
     */
    public static function get_info(): array {
        $is_valid = self::is_valid();
        $data = self::get_data();

        return [
            'is_valid' => $is_valid,
            'key' => self::get_key(true), // Always masked
            'order_id' => $data['order_id'] ?? '',
            'expire_date' => $data['expire_date'] ?? '',
            'activation_date' => $data['activation_date'] ?? '',
            'activated_at' => $data['activated_at'] ?? '',
        ];
    }
}
