<?php

namespace WPOutreach\Admin;

/**
 * Admin assets enqueuing
 */
class Assets {

    /**
     * Initialize assets
     */
    public function init(): void {
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_assets(string $hook): void {
        // Only load on our plugin pages
        if (!$this->is_plugin_page($hook)) {
            return;
        }

        $is_dev = $this->is_dev_mode();

        // Enqueue WordPress Media Library
        wp_enqueue_media();

        if ($is_dev) {
            $this->enqueue_dev_assets();
        } else {
            $this->enqueue_prod_assets();
        }

        // Localize script data
        wp_localize_script('outreach-app', 'wpOutreach', $this->get_script_data());
    }

    /**
     * Check if current page is our plugin page
     */
    private function is_plugin_page(string $hook): bool {
        return strpos($hook, 'outreach') !== false;
    }

    /**
     * Check if running in development mode
     */
    private function is_dev_mode(): bool {
        // Only use dev mode if explicitly enabled AND on localhost
        if (defined('WP_OUTREACH_DEV') && WP_OUTREACH_DEV) {
            $host = $_SERVER['HTTP_HOST'] ?? '';
            // Only allow dev mode on localhost
            if (strpos($host, 'localhost') !== false || strpos($host, '127.0.0.1') !== false || strpos($host, '.local') !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Enqueue development assets (Vite HMR)
     */
    private function enqueue_dev_assets(): void {
        $dev_server = 'http://localhost:5173';

        // Vite client for HMR
        wp_enqueue_script(
            'vite-client',
            $dev_server . '/@vite/client',
            [],
            null,
            false
        );

        // Main app entry
        wp_enqueue_script(
            'outreach-app',
            $dev_server . '/src/main.js',
            ['vite-client'],
            null,
            true
        );

        // Add type="module" to scripts
        add_filter('script_loader_tag', function($tag, $handle) {
            if (in_array($handle, ['vite-client', 'outreach-app'])) {
                return str_replace(' src', ' type="module" src', $tag);
            }
            return $tag;
        }, 10, 2);
    }

    /**
     * Enqueue production assets
     */
    private function enqueue_prod_assets(): void {
        $manifest_path = WP_OUTREACH_PATH . 'assets/.vite/manifest.json';

        // Fallback to old manifest location
        if (!file_exists($manifest_path)) {
            $manifest_path = WP_OUTREACH_PATH . 'assets/manifest.json';
        }

        if (!file_exists($manifest_path)) {
            // Show admin notice if no build exists
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>';
                echo '<strong>WP Outreach:</strong> Production assets not found. ';
                echo 'Please run <code>npm run build</code> in the plugin directory.';
                echo '</p></div>';
            });
            return;
        }

        $manifest = json_decode(file_get_contents($manifest_path), true);

        // Get entry point files
        $entry = $manifest['src/main.js'] ?? null;

        if (!$entry) {
            return;
        }

        // Enqueue CSS
        if (!empty($entry['css'])) {
            foreach ($entry['css'] as $index => $css_file) {
                wp_enqueue_style(
                    'outreach-app-' . $index,
                    WP_OUTREACH_URL . 'assets/' . $css_file,
                    [],
                    WP_OUTREACH_VERSION
                );
            }
        }

        // Enqueue JS
        wp_enqueue_script(
            'outreach-app',
            WP_OUTREACH_URL . 'assets/' . $entry['file'],
            [],
            WP_OUTREACH_VERSION,
            true
        );

        // Add type="module"
        add_filter('script_loader_tag', function($tag, $handle) {
            if ($handle === 'outreach-app') {
                return str_replace(' src', ' type="module" src', $tag);
            }
            return $tag;
        }, 10, 2);
    }

    /**
     * Get localized script data
     */
    private function get_script_data(): array {
        return [
            'apiUrl' => rest_url('outreach/v1'),
            'nonce' => wp_create_nonce('wp_rest'),
            'adminUrl' => admin_url(),
            'pluginUrl' => WP_OUTREACH_URL,
            'version' => WP_OUTREACH_VERSION,
            'siteName' => get_bloginfo('name'),
            'siteUrl' => home_url(),
            'adminEmail' => get_option('admin_email'),
            'isSiteAdmin' => current_user_can('manage_options'),
            'user' => [
                'id' => get_current_user_id(),
                'name' => wp_get_current_user()->display_name,
                'email' => wp_get_current_user()->user_email,
            ],
        ];
    }
}
