<?php

namespace WPOutreach\Admin;

/**
 * Admin page registration and rendering
 */
class AdminPage {

    /**
     * Initialize admin page
     */
    public function init(): void {
        add_action('admin_menu', [$this, 'register_menu']);
    }

    /**
     * Check if current user can access WP Outreach
     */
    private function current_user_can_access(): bool {
        // Site admins always have access
        if (current_user_can('manage_options')) {
            return true;
        }

        // Check allowed users list
        $allowed_users = get_option('wp_outreach_allowed_users', []);
        if (!is_array($allowed_users) || empty($allowed_users)) {
            return false;
        }

        $allowed_users = array_map('intval', $allowed_users);
        return in_array(get_current_user_id(), $allowed_users, true);
    }

    /**
     * Register admin menu
     */
    public function register_menu(): void {
        // Check if current user has access
        if (!$this->current_user_can_access()) {
            return;
        }

        // Use 'read' capability since we do our own access check
        $capability = 'read';

        add_menu_page(
            __('WP Outreach', 'outreach'),
            __('Outreach', 'outreach'),
            $capability,
            'outreach',
            [$this, 'render_app'],
            $this->get_menu_icon(),
            30
        );

        // Submenu items (Vue router handles these)
        $submenus = [
            'outreach' => __('Dashboard', 'outreach'),
            'outreach#/campaigns' => __('Campaigns', 'outreach'),
            'outreach#/automations' => __('Automations', 'outreach'),
            'outreach#/subscribers' => __('Subscribers', 'outreach'),
            'outreach#/lists' => __('Lists', 'outreach'),
            'outreach#/tags' => __('Tags', 'outreach'),
            'outreach#/post-subscriptions' => __('Post Subscriptions', 'outreach'),
            'outreach#/templates' => __('Templates', 'outreach'),
            'outreach#/logs' => __('Email Logs', 'outreach'),
            'outreach#/analytics' => __('Analytics', 'outreach'),
            'outreach#/settings' => __('Settings', 'outreach'),
        ];

        foreach ($submenus as $slug => $title) {
            add_submenu_page(
                'outreach',
                $title,
                $title,
                $capability,
                $slug,
                [$this, 'render_app']
            );
        }

        // Remove duplicate parent menu item
        remove_submenu_page('outreach', 'outreach');

        // Re-add as Dashboard
        add_submenu_page(
            'outreach',
            __('Dashboard', 'outreach'),
            __('Dashboard', 'outreach'),
            $capability,
            'outreach',
            [$this, 'render_app'],
            0
        );
    }

    /**
     * Render the Vue.js app container
     */
    public function render_app(): void {
        include WP_OUTREACH_PATH . 'templates/admin-app.php';
    }

    /**
     * Get menu icon as base64 SVG
     */
    private function get_menu_icon(): string {
        //$svg = 'dashicons-email-alt';
		//return $svg;
	    $svg = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="512" height="506"><g transform="translate(18 18) scale(0.85)">
				<path d="M0 0 C6.46855229 3.57414241 12.99339027 7.03891412 19.52587891 10.49443054 C22.21518124 11.91761465 24.90240372 13.34470234 27.58966064 14.77174377 C30.98542057 16.57497317 34.38186294 18.3769165 37.77819824 20.17906189 C43.19521174 23.0534088 48.61200329 25.92817281 54.02819824 28.80406189 C54.69428101 29.15772827 55.36036377 29.51139465 56.04663086 29.8757782 C66.44466664 35.39782226 76.82653267 40.94935943 87.2010498 46.51548767 C96.04812595 51.25937249 104.91076745 55.97336874 113.77819824 60.67906189 C127.41884552 67.91957754 141.05076655 75.17543267 154.64685059 82.49937439 C157.61762732 84.09292997 160.59703559 85.6690303 163.57897949 87.24156189 C182.28587576 97.1805008 196.5732547 108.6714441 203.56945801 129.53916931 C205.72469566 136.59838108 206.14950054 143.01918298 205.96569824 150.36656189 C205.95216309 151.41392517 205.93862793 152.46128845 205.92468262 153.54039001 C205.88979504 156.08718313 205.84078694 158.63282514 205.77819824 161.17906189 C204.51556152 161.87515564 204.51556152 161.87515564 203.22741699 162.58531189 C193.77191466 167.83545926 184.45756013 173.30768936 175.16418457 178.83912659 C166.63506403 183.9138584 158.07092137 188.92145272 149.46569824 193.86656189 C141.49993902 198.44490644 133.55687349 203.05679126 125.65319824 207.74156189 C117.60265215 212.51209059 109.54206852 217.26485209 101.46569824 221.99156189 C100.41430664 222.60910339 99.36291504 223.2266449 98.27966309 223.86289978 C97.31367188 224.42742859 96.34768066 224.9919574 95.35241699 225.57359314 C94.49220947 226.07737488 93.63200195 226.58115662 92.74572754 227.10020447 C90.77819824 228.17906189 90.77819824 228.17906189 89.77819824 228.17906189 C89.71632324 226.22999939 89.71632324 226.22999939 89.65319824 224.24156189 C89.00838399 218.49272499 86.5562657 214.482851 82.77819824 210.17906189 C77.68872788 206.27318928 71.93678012 206.62983163 65.77819824 207.17906189 C63.43608903 207.81421015 61.10134855 208.47773349 58.77819824 209.17906189 C52.23378979 209.81239174 47.40362135 208.65737694 41.77819824 205.17906189 C37.97118209 201.88950425 35.8567844 198.08172624 34.77819824 193.17906189 C34.50569683 188.88716467 34.5379987 185.97472287 35.77819824 181.86656189 C37.55245079 175.32400563 38.15880595 168.81230443 34.71569824 162.74156189 C31.29351003 158.09920629 27.89945925 155.6689459 22.16491699 154.76890564 C15.36907728 154.40232507 10.93471752 155.76076225 5.77819824 160.17906189 C4.59706628 161.37325947 3.43033311 162.58186477 2.27819824 163.80406189 C-2.11485105 168.38681103 -6.48829933 169.88618663 -12.72180176 170.55406189 C-20.12491244 169.76087146 -24.92358424 167.53657314 -29.69445801 161.69468689 C-32.07843792 157.76813174 -33.12730994 153.57894811 -34.33898926 149.17515564 C-35.88147857 143.94024723 -38.01188778 141.11736253 -42.28430176 137.80406189 C-47.04664155 135.36656324 -52.01569688 135.419998 -57.22180176 136.17906189 C-62.15879733 137.87927952 -65.26122032 140.95649524 -68.22180176 145.17906189 C-69.3995837 148.30328861 -69.3995837 148.30328861 -70.22180176 151.61656189 C-72.35974988 158.98707565 -74.63931833 163.79073961 -81.22180176 168.17906189 C-87.07867059 170.7255266 -91.92923636 170.73371481 -98.09680176 169.24156189 C-102.78040737 167.36267359 -105.71355749 163.86434289 -109.14367676 160.27671814 C-113.29126775 156.09014603 -116.84291718 155.00311308 -122.59680176 154.67906189 C-128.24265506 154.85573485 -132.09311277 156.30354723 -136.22180176 160.17906189 C-139.37721894 164.12333336 -140.73483668 168.17996468 -141.22180176 173.17906189 C-140.66745428 176.29466029 -140.66745428 176.29466029 -139.72180176 179.42906189 C-137.85251616 187.03601579 -137.44392146 193.40457314 -141.15930176 200.49156189 C-144.43316745 205.1563474 -148.89104006 207.42659436 -154.22180176 209.17906189 C-158.24225724 209.86354804 -161.6182178 209.36127591 -165.53430176 208.17906189 C-171.58736133 206.53755421 -178.29906222 205.81816858 -184.09680176 208.67906189 C-188.47026167 211.65427621 -191.44955062 216.14154074 -192.74914551 221.25328064 C-193.08744318 223.35080284 -193.08744318 223.35080284 -193.22180176 226.17906189 C-196.90337197 224.7977976 -200.22404626 223.13097686 -203.62414551 221.15953064 C-204.6673999 220.55641052 -205.7106543 219.95329041 -206.78552246 219.33189392 C-207.89886963 218.68333435 -209.0122168 218.03477478 -210.15930176 217.36656189 C-211.33392018 216.6852287 -212.508645 216.0040789 -213.68347168 215.32310486 C-222.62593115 210.133475 -231.53099954 204.8810211 -240.42492676 199.60874939 C-250.08928282 193.8833404 -259.7892115 188.21822217 -269.48233032 182.54170227 C-274.1392721 179.81416521 -278.79412016 177.08307418 -283.44836426 174.35093689 C-285.45329236 173.17433836 -287.45825316 171.99779556 -289.46322632 170.8212738 C-290.75414398 170.06359427 -292.04493711 169.30570253 -293.33560181 168.54759216 C-295.12415139 167.49708487 -296.91315964 166.44736289 -298.70227051 165.39781189 C-299.72216064 164.79904236 -300.74205078 164.20027283 -301.79284668 163.58335876 C-303.24711423 162.74257608 -304.71932124 161.93030215 -306.22180176 161.17906189 C-306.9589707 139.23978015 -304.72498649 122.21708721 -289.86804199 104.96617126 C-282.07224672 96.75537702 -272.39555006 91.34391748 -262.34680176 86.36656189 C-259.95514317 85.15499584 -257.56454222 83.94151019 -255.17492676 82.72593689 C-253.99865723 82.12974548 -252.8223877 81.53355408 -251.61047363 80.91929626 C-246.72196687 78.40882968 -241.90670011 75.77373205 -237.09680176 73.11656189 C-229.36199835 68.85668161 -221.56720947 64.73156109 -213.72180176 60.67906189 C-204.87825097 56.11061827 -196.11208067 51.42366328 -187.38977051 46.62828064 C-180.87794545 43.05758292 -174.31553547 39.59587539 -167.72180176 36.17906189 C-160.17184544 32.26655712 -152.67473483 28.27337705 -145.22180176 24.17906189 C-136.82109855 19.56408331 -128.3535178 15.09592118 -119.84265137 10.68785095 C-113.10685666 7.189242 -106.43553178 3.58553355 -99.78430176 -0.07093811 C-93.39817617 -3.56695929 -87.01058418 -7.0566077 -80.53430176 -10.38343811 C-79.37543457 -10.98285217 -78.21656738 -11.58226624 -77.02258301 -12.19984436 C-49.36796536 -24.82797065 -24.32790494 -13.45715163 0 0 Z " fill="#0DA56D" transform="translate(306.2218017578125,16.820938110351563)"/>
				<path d="M0 0 C10.08929184 4.84696023 19.87833539 10.11695591 29.60083008 15.66137695 C37.44730735 20.13357089 45.33738978 24.52144119 53.25 28.875 C54.48009454 29.55234949 55.71015631 30.22975849 56.94018555 30.90722656 C57.55061203 31.24343018 58.16103851 31.57963379 58.78996277 31.92602539 C59.4006813 32.26239014 60.01139984 32.59875488 60.640625 32.9453125 C61.25146439 33.28167725 61.86230377 33.61804199 62.49165344 33.96459961 C73.346346 39.9430013 84.18358003 45.95259253 95 52 C96.05477539 52.5892627 97.10955078 53.17852539 98.19628906 53.78564453 C101.1763711 55.45454637 104.15159117 57.13174369 107.125 58.8125 C108.45917969 59.55935059 108.45917969 59.55935059 109.8203125 60.32128906 C111.026875 61.00723145 111.026875 61.00723145 112.2578125 61.70703125 C112.9605127 62.1038208 113.66321289 62.50061035 114.38720703 62.90942383 C116 64 116 64 117 66 C114.50613117 74.81946322 110.34032127 82.60481599 105 90 C104.48953125 90.70769531 103.9790625 91.41539062 103.453125 92.14453125 C93.69548189 104.3898903 78.30222636 111.89881422 63 114 C57.54067795 114.287409 52.08824536 114.27486192 46.62345886 114.26069641 C44.96592182 114.26372034 43.30838618 114.26761103 41.65085298 114.27228898 C37.11508756 114.28254663 32.57940138 114.28039285 28.04362953 114.27610838 C23.14438841 114.27363709 18.24516617 114.28266035 13.34593201 114.29014587 C3.75497817 114.30288382 -5.83594766 114.30468768 -15.42690853 114.30234561 C-23.22452165 114.30056123 -31.02212628 114.30231467 -38.81973839 114.30657005 C-40.48561877 114.30746284 -40.48561877 114.30746284 -42.1851533 114.30837366 C-44.44145746 114.30958543 -46.69776161 114.31079957 -48.95406576 114.31201603 C-70.10566739 114.32288815 -91.25725306 114.32073067 -112.40885558 114.31463119 C-131.74730475 114.30946269 -151.08570355 114.32071629 -170.42414276 114.33972941 C-190.29353492 114.35911438 -210.16290326 114.36736262 -230.03230494 114.36360615 C-241.18235379 114.36169507 -252.33235977 114.36417267 -263.48240089 114.37832069 C-272.97491341 114.39024387 -282.46735665 114.39162887 -291.95987014 114.37953053 C-296.79995704 114.37364837 -301.63991731 114.37252407 -306.47999573 114.38453293 C-310.91705527 114.3954136 -315.35388276 114.39226247 -319.79093155 114.37827124 C-321.38977653 114.37563032 -322.98863984 114.37797172 -324.58746681 114.38600711 C-344.81794699 114.48056514 -362.24630705 109.59431664 -377 95 C-384.50673153 86.53117155 -391.15970742 76.35544171 -393 65 C-386.3788498 60.75669939 -379.57285555 56.92148646 -372.6875 53.125 C-371.4933619 52.46442347 -370.29935415 51.80361125 -369.10546875 51.14257812 C-366.6311976 49.77285482 -364.15637535 48.40413521 -361.68115234 47.03613281 C-355.94884319 43.86689517 -350.22485948 40.68266787 -344.5 37.5 C-337.70476109 33.72360606 -330.90868618 29.9487454 -324.109375 26.1796875 C-322.97314697 25.54933594 -321.83691895 24.91898437 -320.66625977 24.26953125 C-318.40971062 23.0181326 -316.15240004 21.76810588 -313.89428711 20.51953125 C-308.34195599 17.44266036 -302.81747589 14.32760527 -297.33203125 11.1328125 C-296.27532227 10.52131348 -295.21861328 9.90981445 -294.12988281 9.27978516 C-292.13222035 8.12068301 -290.14032801 6.95156013 -288.15527344 5.77099609 C-287.27000977 5.25875488 -286.38474609 4.74651367 -285.47265625 4.21875 C-284.70187744 3.76435547 -283.93109863 3.30996094 -283.13696289 2.84179688 C-280.55491546 1.82467137 -279.53920984 1.94423864 -277 3 C-274.9832342 4.64633943 -272.98260426 6.31267723 -271 8 C-265.27247349 9.9091755 -258.70700688 9.68012657 -252.9375 7.9375 C-246.94470463 6.22703069 -241.20875047 6.8747875 -235.6875 9.6875 C-230.92337016 13.78686754 -227.1883065 17.70197553 -226 24 C-226.08353076 29.36267498 -226.78812245 34.40938163 -228.25 39.5625 C-229.49038198 44.84595913 -228.43500975 49.2032751 -226 54 C-222.46012149 58.62760398 -218.04201497 61.00556717 -212.32421875 61.9375 C-204.93290987 62.13625789 -199.97619928 58.57870998 -194.76953125 53.7109375 C-194.18558594 53.14632812 -193.60164062 52.58171875 -193 52 C-188.66474638 48.08769795 -184.76956631 46.06015265 -178.796875 45.96484375 C-172.14706945 46.42852067 -167.52766229 48.47233771 -162.75 53.25 C-160.15011885 57.33552752 -158.62271862 61.62965171 -157.39453125 66.3046875 C-155.73590423 72.27574477 -153.26701556 75.64826282 -148 79 C-142.88090626 80.70636458 -137.11255844 80.90831166 -131.8984375 79.38671875 C-126.95142962 76.91321481 -124.2110251 72.97088622 -122 68 C-121.37015168 65.81975582 -120.76752924 63.63152363 -120.1875 61.4375 C-118.32923105 54.9785168 -115.76620757 51.45536446 -110 48 C-104.34103957 45.48490648 -98.95780453 45.6540508 -93 47 C-89.14007006 48.96780742 -86.48098742 51.49058922 -83.5 54.5625 C-79.08697811 59.09190929 -74.42227964 61.7481459 -68 62 C-61.29637259 60.56181043 -56.22243122 58.14205793 -52.1875 52.4375 C-49.64499755 45.07762448 -50.79493056 38.30429253 -53 31 C-53.80834029 23.95589176 -52.10378493 18.8111991 -48 13 C-43.5187366 8.64790827 -38.56834253 7.26888815 -32.43359375 6.9140625 C-29.47654703 7.0184847 -26.95690235 7.674973 -24.1328125 8.50390625 C-18.4742226 9.82009841 -12.29029653 10.08345653 -7.0625 7.3125 C-4.11741042 5.08865685 -2.2307153 2.90522965 0 0 Z " fill="#0DA56D" transform="translate(394,392)"/>
				<path d="M0 0 C20.06360783 17.28885355 33.30115672 38.89287095 36.9375 65.53125 C39.05636881 94.36598765 32.89175883 120.40444077 14.1796875 142.96484375 C-2.04910797 161.54031174 -22.44256841 173.80290527 -47.0625 177.53125 C-47.84367187 177.66015625 -48.62484375 177.7890625 -49.4296875 177.921875 C-74.44639827 181.40405452 -101.35635561 175.10309236 -121.9453125 160.484375 C-124.38052909 158.56301145 -126.72635328 156.57120669 -129.0625 154.53125 C-129.7225 154.01433594 -130.3825 153.49742187 -131.0625 152.96484375 C-150.34764935 137.4511853 -160.92571503 112.60356497 -163.6875 88.53125 C-166.12491524 59.90270013 -157.06619508 33.96223368 -138.91015625 11.91796875 C-104.43122543 -28.01457151 -40.9426296 -32.48708653 0 0 Z M-12.35791016 20.59130859 C-16.56502045 20.66599496 -20.77178379 20.75518393 -24.97851562 20.84863281 C-26.78801189 20.88711885 -28.59758983 20.92193557 -30.40722656 20.953125 C-33.03477646 20.99883808 -35.66181484 21.05675813 -38.2890625 21.1171875 C-39.09437408 21.12873871 -39.89968567 21.14028992 -40.72940063 21.15219116 C-44.88226453 21.26107477 -48.34353183 21.46255708 -52.0625 23.53125 C-53.31403259 25.87220531 -53.31403259 25.87220531 -53.0625 28.53125 C-51.51534297 30.80903883 -51.51534297 30.80903883 -49.0625 32.53125 C-46.01046746 33.05665545 -46.01046746 33.05665545 -42.5625 32.84375 C-41.25152344 32.80894531 -39.94054687 32.77414062 -38.58984375 32.73828125 C-37.08060109 32.67213103 -35.57149165 32.60289816 -34.0625 32.53125 C-33.18609863 32.49306152 -32.30969727 32.45487305 -31.40673828 32.41552734 C-25.62287588 32.16034199 -19.84263044 31.85986655 -14.0625 31.53125 C-18.77810049 38.1799538 -23.6767911 44.66608117 -28.6875 51.09375 C-29.27764893 51.85099365 -29.86779785 52.6082373 -30.47583008 53.38842773 C-33.65523902 57.46189923 -36.84547637 61.52664852 -40.04296875 65.5859375 C-40.68476074 66.40376709 -41.32655273 67.22159668 -41.98779297 68.06420898 C-43.22412427 69.6394518 -44.46391143 71.2119898 -45.70751953 72.78149414 C-46.2642334 73.49168701 -46.82094727 74.20187988 -47.39453125 74.93359375 C-47.88671143 75.55741943 -48.3788916 76.18124512 -48.88598633 76.82397461 C-50.25228585 78.63739743 -50.25228585 78.63739743 -51.0625 81.53125 C-54.96128432 79.3983449 -57.80107869 76.97931999 -60.875 73.78125 C-64.47188212 70.10455541 -68.13512996 66.65310464 -72.05859375 63.33203125 C-75.17045837 60.5355993 -77.97141165 57.44155136 -80.7890625 54.3515625 C-83.53951474 52.14931036 -85.59609267 51.83915438 -89.0625 51.53125 C-95.22934432 58.8522244 -100.51307605 66.58174206 -105.75927734 74.56347656 C-107.86589123 77.74431357 -110.02350172 80.88927551 -112.1875 84.03125 C-112.62545898 84.66861084 -113.06341797 85.30597168 -113.51464844 85.96264648 C-116.40105145 90.15237458 -119.31754249 94.32009273 -122.24609375 98.48046875 C-122.96974121 99.51171875 -123.69338867 100.54296875 -124.43896484 101.60546875 C-125.87377138 103.64887426 -127.31276777 105.68934552 -128.75634766 107.7265625 C-129.74513916 109.13615234 -129.74513916 109.13615234 -130.75390625 110.57421875 C-131.63864624 111.82755005 -131.63864624 111.82755005 -132.54125977 113.10620117 C-134.05385336 115.51746617 -135.09510804 117.86012866 -136.0625 120.53125 C-134.4125 122.18125 -132.7625 123.83125 -131.0625 125.53125 C-126.50532972 124.16879703 -124.91320461 122.40882089 -122.4375 118.40625 C-121.75042969 117.32601562 -121.06335938 116.24578125 -120.35546875 115.1328125 C-119.98728027 114.54918945 -119.6190918 113.96556641 -119.23974609 113.36425781 C-117.15417686 110.11696337 -114.94697365 106.95375275 -112.75 103.78125 C-111.80705904 102.41422293 -110.86435327 101.04703361 -109.921875 99.6796875 C-109.44943359 98.99487305 -108.97699219 98.31005859 -108.49023438 97.60449219 C-107.07819142 95.55403583 -105.67108741 93.50028384 -104.265625 91.4453125 C-98.30165724 82.73116475 -92.23429501 74.09946867 -86.0625 65.53125 C-80.92430144 69.8732954 -75.95979389 74.35760068 -71.0625 78.96875 C-70.38960938 79.59845703 -69.71671875 80.22816406 -69.0234375 80.87695312 C-68.35570312 81.50408203 -67.68796875 82.13121094 -67 82.77734375 C-66.01 83.70329346 -66.01 83.70329346 -65 84.64794922 C-63.71025707 85.90161315 -62.45895646 87.19609093 -61.25 88.52783203 C-58.33862963 91.71133105 -56.58100021 93.35397072 -52.41796875 94.77734375 C-48.97796978 94.84386255 -48.97796978 94.84386255 -46.61328125 92.56640625 C-45.87464844 91.58542969 -45.13601563 90.60445313 -44.375 89.59375 C-43.95984131 89.05669434 -43.54468262 88.51963867 -43.11694336 87.96630859 C-41.740254 86.17230947 -40.4013727 84.35360451 -39.0625 82.53125 C-38.09023402 81.23960978 -37.11623952 79.94926909 -36.140625 78.66015625 C-34.10987926 75.96912898 -32.09233987 73.26882114 -30.08203125 70.5625 C-22.98151063 61.03724415 -15.5066604 51.78845806 -8.0625 42.53125 C-8.03744385 43.15056396 -8.0123877 43.76987793 -7.98657227 44.40795898 C-7.86927342 47.26169888 -7.74716414 50.11521441 -7.625 52.96875 C-7.58568359 53.94263672 -7.54636719 54.91652344 -7.50585938 55.91992188 C-7.38097904 58.79216962 -7.2315581 61.66127554 -7.0625 64.53125 C-7.01206543 65.39766113 -6.96163086 66.26407227 -6.90966797 67.15673828 C-6.84537598 67.95192871 -6.78108398 68.74711914 -6.71484375 69.56640625 C-6.66191162 70.26258057 -6.60897949 70.95875488 -6.55444336 71.67602539 C-5.92387833 74.05402226 -5.00162601 75.04584381 -3.0625 76.53125 C0.27942692 76.28671876 1.54293557 75.92581443 3.9375 73.53125 C4.2315722 71.6487123 4.2315722 71.6487123 4.16796875 69.47265625 C4.16700195 68.62429199 4.16603516 67.77592773 4.16503906 66.90185547 C4.14521973 65.51184326 4.14521973 65.51184326 4.125 64.09375 C4.1156543 63.1361377 4.10630859 62.17852539 4.09667969 61.19189453 C4.07061434 59.10098582 4.03790555 57.01015226 3.99951172 54.91943359 C3.95676694 52.58395305 3.92243916 50.2483084 3.89501953 47.91259766 C3.84811629 44.24302344 3.77185646 40.57513763 3.6875 36.90625 C3.67356201 35.73908447 3.65962402 34.57191895 3.64526367 33.36938477 C3.59656128 31.72860962 3.59656128 31.72860962 3.546875 30.0546875 C3.52592773 29.08450684 3.50498047 28.11432617 3.48339844 27.11474609 C2.76036277 23.69293777 1.81354433 22.51474903 -1.0625 20.53125 C-4.84381197 20.22343509 -8.56832005 20.5580675 -12.35791016 20.59130859 Z " fill="#0DA56D" transform="translate(316.0625,237.46875)"/>
				<path d="M0 0 C0 68.97 0 137.94 0 209 C-9.7437798 204.1281101 -19.33780502 199.26847088 -28.8125 193.9375 C-30.56401165 192.95830845 -32.31552665 191.97912284 -34.06723022 191.00027466 C-35.86133736 189.99718059 -37.65460472 188.99259829 -39.44775391 187.98779297 C-45.12309347 184.80948646 -50.81247158 181.65642288 -56.5 178.5 C-58.75006147 177.25011065 -61.00006137 176.00011046 -63.25 174.75 C-64.36375 174.13125 -65.4775 173.5125 -66.625 172.875 C-70 171 -73.375 169.125 -76.75 167.25 C-77.86528076 166.63044434 -78.98056152 166.01088867 -80.12963867 165.37255859 C-82.36433021 164.13097336 -84.5988697 162.88911441 -86.83325195 161.64697266 C-92.06907693 158.73669378 -97.30631704 155.8290659 -102.54916382 152.93145752 C-105.0052297 151.57381704 -107.46040999 150.21457944 -109.91552734 148.85522461 C-111.63293148 147.90518636 -113.35160178 146.9574381 -115.0703125 146.00976562 C-120.8217832 142.82428155 -126.4641777 139.54774585 -132 136 C-130.51837875 129.77540858 -128.64123094 125.57130883 -123.58203125 121.48828125 C-119.3956424 119.07524925 -114.87969724 118.12859203 -110.22265625 116.953125 C-104.82496268 115.35671624 -102.39377182 112.95128054 -99.125 108.375 C-97.27504185 102.82512555 -96.79565402 97.21694193 -98.87890625 91.65234375 C-103.05331263 84.28169222 -108.98397988 82.2414193 -116.75 79.9375 C-117.85859375 79.63714844 -118.9671875 79.33679688 -120.109375 79.02734375 C-122.91157149 78.03142797 -124.25331666 77.36786297 -126 75 C-126 74.34 -126 73.68 -126 73 C-125.03066528 72.4440918 -125.03066528 72.4440918 -124.04174805 71.87695312 C-111.47845033 64.66577249 -98.94909286 57.40691853 -86.5 50 C-76.52042118 44.06290095 -66.48988994 38.2223229 -56.42236328 32.43579102 C-49.54681141 28.48012993 -42.69860647 24.48207802 -35.875 20.4375 C-27.82445391 15.6669713 -19.76387027 10.9142098 -11.6875 6.1875 C-10.6361084 5.5699585 -9.5847168 4.95241699 -8.50146484 4.31616211 C-7.05247803 3.4693689 -7.05247803 3.4693689 -5.57421875 2.60546875 C-4.71401123 2.10168701 -3.85380371 1.59790527 -2.9675293 1.07885742 C-1 0 -1 0 0 0 Z " fill="#0DA56D" transform="translate(512,213)"/>
				<path d="M0 0 C13.39226519 6.6961326 13.39226519 6.6961326 18.4375 9.8125 C25.7825326 14.29529063 33.19821992 18.65454745 40.625 23 C41.88665592 23.73865647 43.14829311 24.47734494 44.40991211 25.21606445 C46.98956839 26.72637805 49.56943426 28.23633289 52.14941406 29.74609375 C58.26834765 33.32719414 64.38398865 36.9139121 70.5 40.5 C72.8918913 41.90240479 75.28381912 43.30474728 77.67578125 44.70703125 C78.27005417 45.05542557 78.86432709 45.40381989 79.47660828 45.76277161 C83.29178627 47.99922207 87.10764179 50.23451265 90.92391968 52.46908569 C93.54774425 54.00548936 96.17134114 55.54228157 98.79492188 57.07910156 C100.03390381 57.80471792 101.27299467 58.53014834 102.51220703 59.25537109 C109.70017022 63.46204192 116.86966213 67.69628478 124 72 C122.08875873 76.40136813 119.30008879 78.11427299 115 80 C113.32099917 80.52783754 111.63275628 81.02696469 109.9375 81.5 C104.11796967 83.16019985 100.39682502 85.1988309 96.625 90.0625 C94.17635717 94.81635051 94.36237171 99.79663828 95 105 C96.65129204 109.95387611 99.42973457 113.45617301 104 116 C107.28347288 117.13208798 110.59324309 118.08901965 113.93359375 119.0390625 C119.70300764 120.84705717 123.76068695 123.79396117 127 129 C127.4192706 130.54601828 127.78623498 132.1068514 128.109375 133.67578125 C128.27920898 134.48571533 128.44904297 135.29564941 128.62402344 136.13012695 C128.7480957 136.74718506 128.87216797 137.36424316 129 138 C108.75 149.25 88.5 160.5 68.25 171.75 C66.57852905 172.6786084 66.57852905 172.6786084 64.87329102 173.62597656 C62.62901835 174.87277666 60.38471509 176.11952168 58.14038086 177.36621094 C52.42749731 180.53971256 46.71518306 183.71423272 41.00390625 186.890625 C34.33800386 190.5974784 27.66994849 194.30043216 21 198 C19.97471191 198.56895996 18.94942383 199.13791992 17.89306641 199.72412109 C15.05506375 201.2977083 12.21538375 202.86821578 9.375 204.4375 C8.53332275 204.90446289 7.69164551 205.37142578 6.82446289 205.85253906 C1.11529828 209 1.11529828 209 0 209 C0 140.03 0 71.06 0 0 Z " fill="#0DA56D" transform="translate(0,213)"/>
				</g></svg>';
        return 'data:image/svg+xml;base64,' . base64_encode($svg);
    }
}
