<?php

namespace WPOutreach;

/**
 * Plugin activation handler
 */
class Activator {

    /**
     * Run activation tasks
     */
    public static function activate(): void {
        // Check requirements
        if (version_compare(PHP_VERSION, '8.0', '<')) {
            deactivate_plugins(WP_OUTREACH_BASENAME);
            wp_die(
                esc_html__('WP Outreach requires PHP 8.0 or higher.', 'outreach'),
                'Plugin Activation Error',
                ['back_link' => true]
            );
        }

        // Install database tables
        $installer = new Installer();
        $installer->install();

        // Set default options
        self::set_default_options();

        // Create default list
        self::create_default_list();

        // Flush rewrite rules
        flush_rewrite_rules();

        // Schedule cron jobs
        self::schedule_cron_jobs();

        // Set activation flag
        update_option('wp_outreach_activated', time());
        update_option('wp_outreach_version', WP_OUTREACH_VERSION);
    }

    /**
     * Schedule cron jobs for queue processing
     */
    private static function schedule_cron_jobs(): void {
        // Register the queue worker cron
        if (!wp_next_scheduled('wp_outreach_process_queue')) {
            wp_schedule_event(time(), 'wp_outreach_minute', 'wp_outreach_process_queue');
        }

        // Daily cleanup
        if (!wp_next_scheduled('wp_outreach_cleanup_queue')) {
            wp_schedule_event(time(), 'daily', 'wp_outreach_cleanup_queue');
        }
    }

    /**
     * Set default plugin options
     */
    private static function set_default_options(): void {
        $defaults = [
            'wp_outreach_general' => [
                'from_name' => get_bloginfo('name'),
                'from_email' => get_bloginfo('admin_email'),
                'reply_to' => get_bloginfo('admin_email'),
                'company_name' => get_bloginfo('name'),
                'company_address' => '',
            ],
            'wp_outreach_mailer' => [
                'driver' => 'wp_mail',
                'queue_enabled' => true,
                'batch_size' => 50,
                'rate_limit' => 100,
            ],
            'wp_outreach_subscription' => [
                'double_optin' => true,
                'welcome_email' => true,
            ],
            'wp_outreach_tracking' => [
                'open_tracking' => true,
                'click_tracking' => true,
            ],
        ];

        foreach ($defaults as $option_name => $default_value) {
            if (get_option($option_name) === false) {
                update_option($option_name, $default_value);
            }
        }
    }

    /**
     * Create default subscriber list
     */
    private static function create_default_list(): void {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';

        // Check if default list exists
        $exists = $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE slug = 'newsletter'");

        if (!$exists) {
            $wpdb->insert($table, [
                'name' => __('Newsletter', 'outreach'),
                'slug' => 'newsletter',
                'description' => __('Default newsletter list', 'outreach'),
                'is_public' => 1,
                'double_optin' => 1,
                'subscriber_count' => 0,
                'created_at' => current_time('mysql'),
            ]);
        }
    }
}
