<?php

namespace WPOutreach\API;

use WP_REST_Controller;
use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use WPOutreach\Helpers\DateHelper;
use WPOutreach\License\FeatureManager;

/**
 * Base REST API controller
 */
class RestController extends WP_REST_Controller {

    /**
     * Constructor - set namespace
     */
    public function __construct() {
        $this->namespace = 'outreach/v1';
    }

    /**
     * Register all API routes
     */
    public function register_routes(): void {
        // Dashboard stats
        register_rest_route($this->namespace, '/dashboard', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_dashboard'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Analytics data
        register_rest_route($this->namespace, '/analytics', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_analytics'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Health check
        register_rest_route($this->namespace, '/health', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_health'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Register sub-controllers
        $this->register_subscriber_routes();
        $this->register_list_routes();
        $this->register_tag_routes();
        $this->register_campaign_routes();
        $this->register_automation_routes();
        $this->register_template_routes();
        $this->register_settings_routes();
        $this->register_ses_routes();
        $this->register_queue_routes();
        $this->register_tracking_routes();
        $this->register_post_subscription_routes();
        $this->register_logs_routes();
        $this->register_license_routes();
        $this->register_wpdm_routes();
        $this->register_webhook_routes();
    }

    /**
     * Register email logs routes
     */
    private function register_logs_routes(): void {
        register_rest_route($this->namespace, '/logs', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_logs'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'page' => [
                        'default' => 1,
                        'sanitize_callback' => 'absint',
                    ],
                    'per_page' => [
                        'default' => 20,
                        'sanitize_callback' => 'absint',
                    ],
                    'type' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'status' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'search' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        register_rest_route($this->namespace, '/logs/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_log'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_log'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/logs/stats', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_logs_stats'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register license routes
     */
    private function register_license_routes(): void {
        // Get license info
        register_rest_route($this->namespace, '/license', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_license'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Validate and activate license
        register_rest_route($this->namespace, '/license/validate', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'validate_license'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Deactivate license
        register_rest_route($this->namespace, '/license/deactivate', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'deactivate_license'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register subscriber routes
     */
    private function register_subscriber_routes(): void {
        register_rest_route($this->namespace, '/subscribers', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_subscribers'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => $this->get_collection_params(),
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_subscriber'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/subscribers/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_subscriber'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_subscriber'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_subscriber'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Export subscribers to CSV
        register_rest_route($this->namespace, '/subscribers/export', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'export_subscribers'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'list_id' => [
                        'default' => 0,
                        'sanitize_callback' => 'absint',
                    ],
                    'status' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Import subscribers - preview (analyze CSV)
        register_rest_route($this->namespace, '/subscribers/import/preview', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'preview_import_subscribers'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Import subscribers - execute
        register_rest_route($this->namespace, '/subscribers/import', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'import_subscribers'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Import from WordPress users - preview
        register_rest_route($this->namespace, '/subscribers/import/wp-users/preview', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'preview_import_wp_users'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'roles' => [
                        'default' => [],
                        'sanitize_callback' => function($value) {
                            if (is_string($value)) {
                                return array_filter(array_map('sanitize_text_field', explode(',', $value)));
                            }
                            return is_array($value) ? array_map('sanitize_text_field', $value) : [];
                        },
                    ],
                    'date_from' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'date_to' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'exclude_existing' => [
                        'default' => true,
                        'sanitize_callback' => 'rest_sanitize_boolean',
                    ],
                ],
            ],
        ]);

        // Import from WordPress users - execute
        register_rest_route($this->namespace, '/subscribers/import/wp-users', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'import_wp_users'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register list routes
     */
    private function register_list_routes(): void {
        register_rest_route($this->namespace, '/lists', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_lists'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_list'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/lists/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_list'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_list'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_list'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register tag routes
     */
    private function register_tag_routes(): void {
        register_rest_route($this->namespace, '/tags', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_tags'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_tag'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/tags/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_tag'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_tag'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register campaign routes
     */
    private function register_campaign_routes(): void {
        register_rest_route($this->namespace, '/campaigns', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_campaigns'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => $this->get_collection_params(),
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Campaign sending routes
        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/send', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'send_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/progress', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_campaign_progress'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/pause', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'pause_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/resume', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'resume_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/cancel', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'cancel_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/test', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'test_campaign'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Campaign stats
        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/stats', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_campaign_stats'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Campaign recipients list
        register_rest_route($this->namespace, '/campaigns/(?P<id>\d+)/recipients', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_campaign_recipients'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'page' => [
                        'default' => 1,
                        'sanitize_callback' => 'absint',
                    ],
                    'per_page' => [
                        'default' => 20,
                        'sanitize_callback' => 'absint',
                    ],
                    'status' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'search' => [
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);
    }

    /**
     * Register automation routes
     */
    private function register_automation_routes(): void {
        register_rest_route($this->namespace, '/automations', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_automations'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_automation'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/automations/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_automation'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_automation'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_automation'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Automation triggers and actions registry
        register_rest_route($this->namespace, '/automations/triggers', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_automation_triggers'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/automations/actions', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_automation_actions'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Condition operators and fields for condition builder UI
        register_rest_route($this->namespace, '/automations/condition-options', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_condition_options'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'trigger_type' => [
                        'type' => 'string',
                        'required' => false,
                        'description' => 'Trigger type to get context-specific fields',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Duplicate automation
        register_rest_route($this->namespace, '/automations/(?P<id>\d+)/duplicate', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'duplicate_automation'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register template routes
     */
    private function register_template_routes(): void {
        register_rest_route($this->namespace, '/templates', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_templates'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_template'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/templates/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_template'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_template'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_template'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register settings routes
     */
    private function register_settings_routes(): void {
        register_rest_route($this->namespace, '/settings', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_settings'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_settings'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/settings/test-email', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'send_test_email'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/settings/regenerate-cron-key', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'regenerate_cron_key'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get available post types for post subscriptions settings
        register_rest_route($this->namespace, '/settings/post-types', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_available_post_types'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get posts by type for admin dropdowns
        register_rest_route($this->namespace, '/settings/posts', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_posts_by_type'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'post_type' => [
                        'required' => true,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_key',
                    ],
                    'per_page' => [
                        'type' => 'integer',
                        'default' => 50,
                        'sanitize_callback' => 'absint',
                    ],
                    'search' => [
                        'type' => 'string',
                        'default' => '',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // User access management (site admins only)
        register_rest_route($this->namespace, '/settings/users', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_allowed_users'],
                'permission_callback' => [$this, 'check_site_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/settings/users/add', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'add_allowed_user'],
                'permission_callback' => [$this, 'check_site_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/settings/users/remove', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'remove_allowed_user'],
                'permission_callback' => [$this, 'check_site_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/settings/users/search', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'search_users'],
                'permission_callback' => [$this, 'check_site_admin_permission'],
            ],
        ]);

        // External cron endpoint (public, but secured by secret key)
        register_rest_route($this->namespace, '/cron', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'run_cron'],
                'permission_callback' => '__return_true',
            ],
        ]);
    }

    /**
     * Register SES management routes
     */
    private function register_ses_routes(): void {
        // Get SES regions list
        register_rest_route($this->namespace, '/ses/regions', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_ses_regions'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get SES account status
        register_rest_route($this->namespace, '/ses/status', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_ses_status'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Test SES connection
        register_rest_route($this->namespace, '/ses/test', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'test_ses_connection'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get SES sending statistics
        register_rest_route($this->namespace, '/ses/statistics', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_ses_statistics'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get all SES identities
        register_rest_route($this->namespace, '/ses/identities', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_ses_identities'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Check if a sender email is verified (email or domain)
        register_rest_route($this->namespace, '/ses/check-sender', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'check_ses_sender'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'email' => [
                        'required' => true,
                        'sanitize_callback' => 'sanitize_email',
                    ],
                ],
            ],
        ]);

        // Verify email identity
        register_rest_route($this->namespace, '/ses/identities/email', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'verify_ses_email'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Verify domain identity
        register_rest_route($this->namespace, '/ses/identities/domain', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'verify_ses_domain'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Refresh identity status
        register_rest_route($this->namespace, '/ses/identities/(?P<identity>[^/]+)/refresh', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'refresh_ses_identity'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Delete identity
        register_rest_route($this->namespace, '/ses/identities/(?P<identity>[^/]+)', [
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_ses_identity'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Suppression list - get all
        register_rest_route($this->namespace, '/ses/suppression', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_ses_suppression_list'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'add_ses_suppression'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Suppression list - single email
        register_rest_route($this->namespace, '/ses/suppression/(?P<email>[^/]+)', [
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_ses_suppression'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Register tracking routes (public endpoints for email tracking)
     */
    private function register_tracking_routes(): void {
        // Unsubscribe endpoint (public) - GET shows management page, POST processes form
        register_rest_route($this->namespace, '/unsubscribe', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'handle_unsubscribe'],
                'permission_callback' => '__return_true',
            ],
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'handle_unsubscribe_form'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Open tracking pixel (public)
        register_rest_route($this->namespace, '/track/open', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'track_open'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Click tracking redirect (public)
        register_rest_route($this->namespace, '/track/click', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'track_click'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Public subscription endpoint
        register_rest_route($this->namespace, '/subscribe', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'handle_subscribe'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Email confirmation endpoint (public)
        register_rest_route($this->namespace, '/confirm', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'handle_confirm'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Resend confirmation email (public)
        register_rest_route($this->namespace, '/resend-confirmation', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'resend_confirmation'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Post subscription (public)
        register_rest_route($this->namespace, '/post-subscribe', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'handle_post_subscribe'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Post unsubscribe (public, via token)
        register_rest_route($this->namespace, '/post-unsubscribe', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'handle_post_unsubscribe'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Get available post types (admin, for trigger config)
        register_rest_route($this->namespace, '/post-types', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_post_types'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get available user roles (admin, for trigger config)
        register_rest_route($this->namespace, '/user-roles', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_user_roles'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Check if current user has admin permission
     */
    public function check_admin_permission(): bool {
        // Site admins always have access
        if (current_user_can('manage_options')) {
            return true;
        }

        // Check if user is in the allowed users list
        $allowed_users = get_option('wp_outreach_allowed_users', []);
        if (!is_array($allowed_users) || empty($allowed_users)) {
            return false;
        }

        // Cast to integers for reliable comparison (WP options can return strings)
        $allowed_users = array_map('intval', $allowed_users);
        $current_user_id = get_current_user_id();

        return in_array($current_user_id, $allowed_users, true);
    }

    /**
     * Check if current user is a site admin (for user management)
     */
    public function check_site_admin_permission(): bool {
        return current_user_can('manage_options');
    }

    /**
     * Get collection params for pagination
     */
    public function get_collection_params(): array {
        return [
            'page' => [
                'default' => 1,
                'type' => 'integer',
                'minimum' => 1,
            ],
            'per_page' => [
                'default' => 20,
                'type' => 'integer',
                'minimum' => 1,
                'maximum' => 100,
            ],
            'search' => [
                'default' => '',
                'type' => 'string',
            ],
            'orderby' => [
                'default' => 'created_at',
                'type' => 'string',
            ],
            'order' => [
                'default' => 'desc',
                'type' => 'string',
                'enum' => ['asc', 'desc'],
            ],
        ];
    }

    /**
     * Health check endpoint
     */
    public function get_health(): WP_REST_Response {
        return new WP_REST_Response([
            'status' => 'ok',
            'version' => WP_OUTREACH_VERSION,
            'timestamp' => current_time('mysql'),
        ]);
    }

    /**
     * Get dashboard stats with trends and recent activity
     */
    public function get_dashboard(): WP_REST_Response {
        global $wpdb;

        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $lists_table = $wpdb->prefix . 'outreach_lists';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $automations_table = $wpdb->prefix . 'outreach_automations';
        $automation_queue_table = $wpdb->prefix . 'outreach_automation_queue';

        // Date boundaries
        $today = current_time('Y-m-d');
        $week_ago = date('Y-m-d', strtotime('-7 days'));
        $two_weeks_ago = date('Y-m-d', strtotime('-14 days'));
        $month_ago = date('Y-m-d', strtotime('-30 days'));

        // Subscriber stats with trends
        $total_subscribers = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$subscribers_table}");
        $active_subscribers = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$subscribers_table} WHERE status = 'active'");
        $new_this_week = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$subscribers_table} WHERE created_at >= %s",
            $week_ago
        ));
        $new_last_week = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$subscribers_table} WHERE created_at >= %s AND created_at < %s",
            $two_weeks_ago, $week_ago
        ));
        $subscriber_trend = $new_last_week > 0
            ? round((($new_this_week - $new_last_week) / $new_last_week) * 100, 1)
            : ($new_this_week > 0 ? 100 : 0);

        // Campaign stats
        $total_campaigns = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$campaigns_table}");
        $sent_campaigns = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$campaigns_table} WHERE status = 'sent'");
        $draft_campaigns = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$campaigns_table} WHERE status = 'draft'");
        $sending_campaigns = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$campaigns_table} WHERE status = 'sending'");

        // Email stats with trends
        $emails_sent = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table}");
        $emails_opened = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE opens > 0");
        $emails_clicked = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE clicks > 0");
        $emails_bounced = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE status = 'bounced'");

        // Emails this week vs last week
        $emails_this_week = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$logs_table} WHERE sent_at >= %s",
            $week_ago
        ));
        $emails_last_week = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$logs_table} WHERE sent_at >= %s AND sent_at < %s",
            $two_weeks_ago, $week_ago
        ));
        $email_trend = $emails_last_week > 0
            ? round((($emails_this_week - $emails_last_week) / $emails_last_week) * 100, 1)
            : ($emails_this_week > 0 ? 100 : 0);

        // Open/click rates
        $open_rate = $emails_sent > 0 ? round(($emails_opened / $emails_sent) * 100, 1) : 0;
        $click_rate = $emails_sent > 0 ? round(($emails_clicked / $emails_sent) * 100, 1) : 0;

        // Automation stats
        $active_automations = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$automations_table} WHERE status = 'active'");
        $running_workflows = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$automation_queue_table} WHERE status = 'active'");

        // Recent activity (combined from multiple sources)
        $recent_activity = $this->get_recent_activity($wpdb, 10);

        // Top campaigns by opens
        $top_campaigns = $wpdb->get_results($wpdb->prepare("
            SELECT c.id, c.name, c.subject, c.sent_at,
                   COUNT(DISTINCT l.id) as emails_sent,
                   COUNT(DISTINCT CASE WHEN l.opens > 0 THEN l.id END) as opened,
                   COUNT(DISTINCT CASE WHEN l.clicks > 0 THEN l.id END) as clicked
            FROM {$campaigns_table} c
            LEFT JOIN {$logs_table} l ON l.campaign_id = c.id
            WHERE c.status = 'sent'
            GROUP BY c.id
            ORDER BY c.sent_at DESC
            LIMIT %d
        ", 5));

        foreach ($top_campaigns as &$campaign) {
            $campaign->open_rate = $campaign->emails_sent > 0
                ? round(($campaign->opened / $campaign->emails_sent) * 100, 1) : 0;
            $campaign->click_rate = $campaign->emails_sent > 0
                ? round(($campaign->clicked / $campaign->emails_sent) * 100, 1) : 0;
            $campaign->sent_at = DateHelper::toIso8601($campaign->sent_at);
        }

        // Additional subscriber stats
        $pending_subscribers = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$subscribers_table} WHERE status = 'pending'");
        $unsubscribed_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$subscribers_table} WHERE status = 'unsubscribed'");
        $bounced_subscribers = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$subscribers_table} WHERE status = 'bounced'");
        $unsubscribe_rate = $total_subscribers > 0
            ? round(($unsubscribed_count / $total_subscribers) * 100, 1)
            : 0;

        // Additional email stats
        $bounce_rate = $emails_sent > 0 ? round(($emails_bounced / $emails_sent) * 100, 1) : 0;

        $stats = [
            'subscribers' => [
                'total' => $total_subscribers,
                'active' => $active_subscribers,
                'pending' => $pending_subscribers,
                'unsubscribed' => $unsubscribed_count,
                'bounced' => $bounced_subscribers,
                'new_this_week' => $new_this_week,
                'trend' => $subscriber_trend,
                'unsubscribe_rate' => $unsubscribe_rate,
            ],
            'campaigns' => [
                'total' => $total_campaigns,
                'sent' => $sent_campaigns,
                'draft' => $draft_campaigns,
                'sending' => $sending_campaigns,
            ],
            'lists' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$lists_table}"),
            'emails' => [
                'sent' => $emails_sent,
                'opened' => $emails_opened,
                'clicked' => $emails_clicked,
                'bounced' => $emails_bounced,
                'open_rate' => $open_rate,
                'click_rate' => $click_rate,
                'bounce_rate' => $bounce_rate,
                'this_week' => $emails_this_week,
                'trend' => $email_trend,
            ],
            'automations' => [
                'active' => $active_automations,
                'running' => $running_workflows,
            ],
            'recent_activity' => $recent_activity,
            'top_campaigns' => $top_campaigns,
        ];

        return new WP_REST_Response($stats);
    }

    /**
     * Get recent activity from multiple sources
     */
    private function get_recent_activity($wpdb, int $limit = 10): array {
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $logs_table = $wpdb->prefix . 'outreach_logs';

        $activities = [];

        // Recent subscribers
        $new_subscribers = $wpdb->get_results($wpdb->prepare("
            SELECT 'subscriber' as type, id, email, first_name, created_at as timestamp
            FROM {$subscribers_table}
            ORDER BY created_at DESC
            LIMIT %d
        ", $limit));

        foreach ($new_subscribers as $sub) {
            $activities[] = [
                'type' => 'subscriber_added',
                'message' => ($sub->first_name ?: $sub->email) . ' subscribed',
                'email' => $sub->email,
                'timestamp' => $sub->timestamp,
            ];
        }

        // Recent campaign sends
        $sent_campaigns = $wpdb->get_results($wpdb->prepare("
            SELECT 'campaign' as type, id, name, sent_at as timestamp
            FROM {$campaigns_table}
            WHERE status = 'sent' AND sent_at IS NOT NULL
            ORDER BY sent_at DESC
            LIMIT %d
        ", $limit));

        foreach ($sent_campaigns as $campaign) {
            $activities[] = [
                'type' => 'campaign_sent',
                'message' => "Campaign \"{$campaign->name}\" was sent",
                'campaign_id' => $campaign->id,
                'timestamp' => $campaign->timestamp,
            ];
        }

        // Recent opens
        $recent_opens = $wpdb->get_results($wpdb->prepare("
            SELECT l.email, l.subject, l.first_opened_at as timestamp
            FROM {$logs_table} l
            WHERE l.first_opened_at IS NOT NULL
            ORDER BY l.first_opened_at DESC
            LIMIT %d
        ", $limit));

        foreach ($recent_opens as $open) {
            $activities[] = [
                'type' => 'email_opened',
                'message' => "{$open->email} opened an email",
                'email' => $open->email,
                'subject' => $open->subject,
                'timestamp' => $open->timestamp,
            ];
        }

        // Sort all activities by timestamp desc
        usort($activities, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });

        // Limit and format timestamps
        $activities = array_slice($activities, 0, $limit);
        foreach ($activities as &$activity) {
            $activity['time_ago'] = human_time_diff(strtotime($activity['timestamp']), current_time('timestamp')) . ' ago';
            // Convert timestamp to ISO 8601 with timezone for proper frontend parsing
            $activity['timestamp'] = DateHelper::toIso8601($activity['timestamp']);
        }

        return $activities;
    }

    /**
     * Get analytics data for charts and detailed metrics
     */
    public function get_analytics(WP_REST_Request $request): WP_REST_Response|WP_Error {
        // Analytics requires Pro license
        if (!FeatureManager::can('analytics')) {
            return new WP_Error(
                'feature_restricted',
                FeatureManager::getRestrictionMessage('analytics'),
                ['status' => 403, 'upgrade_url' => FeatureManager::getUpgradeUrl()]
            );
        }

        global $wpdb;

        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $lists_table = $wpdb->prefix . 'outreach_lists';

        $days = (int) ($request->get_param('days') ?: 30);

        // Subscriber growth over time (daily)
        $subscriber_growth = $wpdb->get_results($wpdb->prepare("
            SELECT DATE(created_at) as date,
                   COUNT(*) as count,
                   SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
                   SUM(CASE WHEN status = 'unsubscribed' THEN 1 ELSE 0 END) as unsubscribed
            FROM {$subscribers_table}
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
            GROUP BY DATE(created_at)
            ORDER BY date ASC
        ", $days));

        // Fill in missing dates with 0 counts
        $subscriber_chart = $this->fill_date_gaps($subscriber_growth, $days, ['count' => 0, 'active' => 0, 'unsubscribed' => 0]);

        // Email performance over time (daily)
        $email_performance = $wpdb->get_results($wpdb->prepare("
            SELECT DATE(sent_at) as date,
                   COUNT(*) as sent,
                   SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as opened,
                   SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END) as clicked,
                   SUM(CASE WHEN status = 'bounced' THEN 1 ELSE 0 END) as bounced
            FROM {$logs_table}
            WHERE sent_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
            GROUP BY DATE(sent_at)
            ORDER BY date ASC
        ", $days));

        $email_chart = $this->fill_date_gaps($email_performance, $days, ['sent' => 0, 'opened' => 0, 'clicked' => 0, 'bounced' => 0]);

        // Campaign performance comparison
        $campaign_performance = $wpdb->get_results("
            SELECT c.id, c.name, c.subject, c.sent_at,
                   COUNT(DISTINCT l.id) as emails_sent,
                   COUNT(DISTINCT CASE WHEN l.opens > 0 THEN l.id END) as opened,
                   COUNT(DISTINCT CASE WHEN l.clicks > 0 THEN l.id END) as clicked,
                   COUNT(DISTINCT CASE WHEN l.status = 'bounced' THEN l.id END) as bounced
            FROM {$campaigns_table} c
            LEFT JOIN {$logs_table} l ON l.campaign_id = c.id
            WHERE c.status = 'sent'
            GROUP BY c.id
            ORDER BY c.sent_at DESC
            LIMIT 10
        ");

        foreach ($campaign_performance as &$campaign) {
            $campaign->open_rate = $campaign->emails_sent > 0
                ? round(($campaign->opened / $campaign->emails_sent) * 100, 1) : 0;
            $campaign->click_rate = $campaign->emails_sent > 0
                ? round(($campaign->clicked / $campaign->emails_sent) * 100, 1) : 0;
            $campaign->sent_at = DateHelper::toIso8601($campaign->sent_at);
        }

        // List performance
        $list_performance = $wpdb->get_results("
            SELECT l.id, l.name, l.subscriber_count,
                   COUNT(DISTINCT sl.subscriber_id) as actual_subscribers
            FROM {$lists_table} l
            LEFT JOIN {$wpdb->prefix}outreach_subscriber_list sl ON sl.list_id = l.id
            LEFT JOIN {$subscribers_table} s ON s.id = sl.subscriber_id AND s.status = 'active'
            GROUP BY l.id
            ORDER BY l.subscriber_count DESC
            LIMIT 10
        ");

        // Summary stats
        $total_sent = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table}");
        $total_opened = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE opens > 0");
        $total_clicked = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE clicks > 0");
        $total_bounced = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$logs_table} WHERE status = 'bounced'");

        return new WP_REST_Response([
            'summary' => [
                'emails_sent' => $total_sent,
                'emails_opened' => $total_opened,
                'emails_clicked' => $total_clicked,
                'emails_bounced' => $total_bounced,
                'open_rate' => $total_sent > 0 ? round(($total_opened / $total_sent) * 100, 1) : 0,
                'click_rate' => $total_sent > 0 ? round(($total_clicked / $total_sent) * 100, 1) : 0,
                'bounce_rate' => $total_sent > 0 ? round(($total_bounced / $total_sent) * 100, 1) : 0,
            ],
            'subscriber_growth' => $subscriber_chart,
            'email_performance' => $email_chart,
            'campaigns' => $campaign_performance,
            'lists' => $list_performance,
        ]);
    }

    /**
     * Fill in missing dates with default values for chart data
     */
    private function fill_date_gaps(array $data, int $days, array $defaults): array {
        $result = [];
        $data_by_date = [];

        foreach ($data as $row) {
            $data_by_date[$row->date] = (array) $row;
        }

        for ($i = $days - 1; $i >= 0; $i--) {
            $date = date('Y-m-d', strtotime("-{$i} days"));
            if (isset($data_by_date[$date])) {
                $result[] = $data_by_date[$date];
            } else {
                $result[] = array_merge(['date' => $date], $defaults);
            }
        }

        return $result;
    }

    // =========================================================================
    // SUBSCRIBER ENDPOINTS
    // =========================================================================

    /**
     * Get paginated list of subscribers with optional filtering.
     *
     * Supports filtering by search term (email/name), status, and list membership.
     * Returns subscriber data along with their associated lists.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     Request parameters.
     *
     *     @type int    $page     Page number (default: 1)
     *     @type int    $per_page Items per page (default: 20)
     *     @type string $search   Search term for email/first_name/last_name
     *     @type string $status   Filter by status (active|pending|unsubscribed|bounced)
     *     @type int    $list_id  Filter by list membership
     * }
     *
     * @return WP_REST_Response {
     *     @type array $items       Array of subscriber objects with 'lists' property
     *     @type int   $total       Total number of matching subscribers
     *     @type int   $page        Current page number
     *     @type int   $per_page    Items per page
     *     @type int   $total_pages Total number of pages
     * }
     */
    public function get_subscribers(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $lists_table = $wpdb->prefix . 'outreach_lists';
        $tag_pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';
        $tags_table = $wpdb->prefix . 'outreach_tags';

        $page = $request->get_param('page');
        $per_page = $request->get_param('per_page');
        $search = $request->get_param('search');
        $status = $request->get_param('status');
        $list_id = $request->get_param('list_id');
        $tag_id = $request->get_param('tag_id');
        $offset = ($page - 1) * $per_page;

        $where = '1=1';
        $join = '';

        if ($search) {
            $search = '%' . $wpdb->esc_like($search) . '%';
            $where .= $wpdb->prepare(" AND (s.email LIKE %s OR s.first_name LIKE %s OR s.last_name LIKE %s)", $search, $search, $search);
        }
        if ($status) {
            $where .= $wpdb->prepare(" AND s.status = %s", $status);
        }
        if ($list_id) {
            $join .= " INNER JOIN {$pivot_table} sl ON s.id = sl.subscriber_id";
            $where .= $wpdb->prepare(" AND sl.list_id = %d", $list_id);
        }
        if ($tag_id) {
            $join .= " INNER JOIN {$tag_pivot_table} st ON s.id = st.subscriber_id";
            $where .= $wpdb->prepare(" AND st.tag_id = %d", $tag_id);
        }

        $total = (int) $wpdb->get_var("SELECT COUNT(DISTINCT s.id) FROM {$table} s {$join} WHERE {$where}");
        $items = $wpdb->get_results(
            "SELECT DISTINCT s.* FROM {$table} s {$join} WHERE {$where} ORDER BY s.created_at DESC LIMIT {$per_page} OFFSET {$offset}"
        );

        // Get lists and tags for each subscriber
        if (!empty($items)) {
            $subscriber_ids = array_column($items, 'id');
            $placeholders = implode(',', array_fill(0, count($subscriber_ids), '%d'));

            // Fetch lists
            $subscriber_lists = $wpdb->get_results($wpdb->prepare(
                "SELECT sl.subscriber_id, l.id, l.name
                 FROM {$pivot_table} sl
                 INNER JOIN {$lists_table} l ON sl.list_id = l.id
                 WHERE sl.subscriber_id IN ({$placeholders})",
                ...$subscriber_ids
            ));

            // Group lists by subscriber
            $lists_by_subscriber = [];
            foreach ($subscriber_lists as $row) {
                if (!isset($lists_by_subscriber[$row->subscriber_id])) {
                    $lists_by_subscriber[$row->subscriber_id] = [];
                }
                $lists_by_subscriber[$row->subscriber_id][] = [
                    'id' => (int) $row->id,
                    'name' => $row->name,
                ];
            }

            // Fetch tags
            $subscriber_tags = $wpdb->get_results($wpdb->prepare(
                "SELECT st.subscriber_id, t.id, t.name, t.color
                 FROM {$tag_pivot_table} st
                 INNER JOIN {$tags_table} t ON st.tag_id = t.id
                 WHERE st.subscriber_id IN ({$placeholders})",
                ...$subscriber_ids
            ));

            // Group tags by subscriber
            $tags_by_subscriber = [];
            foreach ($subscriber_tags as $row) {
                if (!isset($tags_by_subscriber[$row->subscriber_id])) {
                    $tags_by_subscriber[$row->subscriber_id] = [];
                }
                $tags_by_subscriber[$row->subscriber_id][] = [
                    'id' => (int) $row->id,
                    'name' => $row->name,
                    'color' => $row->color,
                ];
            }

            // Attach lists and tags to each subscriber
            foreach ($items as &$item) {
                $item->lists = $lists_by_subscriber[$item->id] ?? [];
                $item->tags = $tags_by_subscriber[$item->id] ?? [];
                // Convert dates to ISO 8601 with timezone
                $item->created_at = DateHelper::toIso8601($item->created_at);
            }
        }

        return new WP_REST_Response([
            'items' => $items,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page),
        ]);
    }

    /**
     * Create a new subscriber.
     *
     * Creates a subscriber with 'pending' status and 'admin' source.
     * Generates a unique token for email confirmation.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     Request parameters.
     *
     *     @type string $email      Required. Subscriber email address.
     *     @type string $first_name Optional. Subscriber first name.
     *     @type string $last_name  Optional. Subscriber last name.
     *     @type array  $list_ids   Optional. Array of list IDs to add subscriber to.
     * }
     *
     * @return WP_REST_Response|WP_Error The created subscriber data with ID (HTTP 201).
     */
    public function create_subscriber(WP_REST_Request $request): WP_REST_Response|WP_Error {
        // Check subscriber limit for free users
        if (FeatureManager::isSubscriberLimitReached()) {
            return new WP_Error(
                'subscriber_limit_reached',
                FeatureManager::getRestrictionMessage('subscriber_limit'),
                [
                    'status' => 403,
                    'upgrade_url' => FeatureManager::getUpgradeUrl(),
                    'limit' => FeatureManager::getSubscriberLimit(),
                    'count' => FeatureManager::getSubscriberCount(),
                ]
            );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $tag_pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';

        $data = [
            'email' => sanitize_email($request->get_param('email')),
            'first_name' => sanitize_text_field($request->get_param('first_name') ?? ''),
            'last_name' => sanitize_text_field($request->get_param('last_name') ?? ''),
            'status' => 'active',
            'source' => 'admin',
            'token' => wp_generate_password(32, false),
            'created_at' => current_time('mysql'),
        ];

        $wpdb->insert($table, $data);
        $subscriber_id = $wpdb->insert_id;
        $data['id'] = $subscriber_id;

        // Handle list assignments
        $list_ids = $request->get_param('list_ids');
        if (is_array($list_ids) && !empty($list_ids)) {
            foreach ($list_ids as $list_id) {
                $wpdb->insert($pivot_table, [
                    'subscriber_id' => $subscriber_id,
                    'list_id' => (int) $list_id,
                    'subscribed_at' => current_time('mysql'),
                ]);

                // Fire automation trigger for list addition
                do_action('wp_outreach_subscriber_added_to_list', $subscriber_id, (int) $list_id, $data);
            }
        }

        // Handle tag assignments
        $tag_ids = $request->get_param('tag_ids');
        if (is_array($tag_ids) && !empty($tag_ids)) {
            foreach ($tag_ids as $tag_id) {
                $wpdb->insert($tag_pivot_table, [
                    'subscriber_id' => $subscriber_id,
                    'tag_id' => (int) $tag_id,
                    'created_at' => current_time('mysql'),
                ]);

                // Fire automation trigger for tag addition
                do_action('wp_outreach_tag_added_to_subscriber', $subscriber_id, (int) $tag_id, $data);
            }
        }

        // Fire automation trigger for new subscriber
        do_action('wp_outreach_subscriber_created', $subscriber_id, $data);

        return new WP_REST_Response($data, 201);
    }

    /**
     * Get a single subscriber by ID.
     *
     * Returns subscriber data including their associated lists and list_ids
     * for easy form binding.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Required. Subscriber ID from URL parameter.
     * }
     *
     * @return WP_REST_Response {
     *     Subscriber object with additional properties:
     *     @type array $lists    Array of {id, name} for each associated list.
     *     @type array $list_ids Array of list IDs for form binding.
     * }
     *                          Returns 404 if subscriber not found.
     */
    public function get_subscriber(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $lists_table = $wpdb->prefix . 'outreach_lists';
        $tag_pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';
        $tags_table = $wpdb->prefix . 'outreach_tags';
        $id = (int) $request->get_param('id');

        $item = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$item) {
            return new WP_REST_Response(['message' => 'Not found'], 404);
        }

        // Get subscriber's lists
        $subscriber_lists = $wpdb->get_results($wpdb->prepare(
            "SELECT l.id, l.name
             FROM {$pivot_table} sl
             INNER JOIN {$lists_table} l ON sl.list_id = l.id
             WHERE sl.subscriber_id = %d",
            $id
        ));

        $item->lists = array_map(function($list) {
            return ['id' => (int) $list->id, 'name' => $list->name];
        }, $subscriber_lists);

        $item->list_ids = array_column($item->lists, 'id');

        // Get subscriber's tags
        $subscriber_tags = $wpdb->get_results($wpdb->prepare(
            "SELECT t.id, t.name, t.color
             FROM {$tag_pivot_table} st
             INNER JOIN {$tags_table} t ON st.tag_id = t.id
             WHERE st.subscriber_id = %d",
            $id
        ));

        $item->tags = array_map(function($tag) {
            return ['id' => (int) $tag->id, 'name' => $tag->name, 'color' => $tag->color];
        }, $subscriber_tags);

        $item->tag_ids = array_column($item->tags, 'id');

        return new WP_REST_Response($item);
    }

    /**
     * Update an existing subscriber.
     *
     * Updates subscriber fields and optionally their list associations.
     * Only provided fields are updated; omitted fields remain unchanged.
     * If list_ids is provided, ALL existing list associations are replaced.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int    $id         Required. Subscriber ID from URL parameter.
     *     @type string $email      Optional. New email address.
     *     @type string $first_name Optional. New first name.
     *     @type string $last_name  Optional. New last name.
     *     @type string $status     Optional. New status (active|pending|unsubscribed|bounced).
     *     @type array  $list_ids   Optional. Array of list IDs. Replaces all existing associations.
     * }
     *
     * @return WP_REST_Response Updated subscriber object (via get_subscriber).
     */
    public function update_subscriber(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $tag_pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';
        $id = (int) $request->get_param('id');

        // Update basic subscriber fields
        $data = [];
        foreach (['email', 'first_name', 'last_name', 'status'] as $field) {
            if ($request->has_param($field)) {
                $data[$field] = $field === 'email'
                    ? sanitize_email($request->get_param($field))
                    : sanitize_text_field($request->get_param($field));
            }
        }

        if (!empty($data)) {
            $wpdb->update($table, $data, ['id' => $id]);
        }

        // Get subscriber data for triggers (used by both list and tag updates)
        $subscriber = null;

        // Update list associations if list_ids is provided
        if ($request->has_param('list_ids')) {
            $new_list_ids = $request->get_param('list_ids');
            if (is_array($new_list_ids)) {
                // Get existing list IDs
                $existing_list_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT list_id FROM {$pivot_table} WHERE subscriber_id = %d",
                    $id
                ));
                $existing_list_ids = array_map('intval', $existing_list_ids);
                $new_list_ids = array_map('intval', $new_list_ids);

                // Find lists to add and remove
                $lists_to_add = array_diff($new_list_ids, $existing_list_ids);
                $lists_to_remove = array_diff($existing_list_ids, $new_list_ids);

                // Remove old associations
                if (!empty($lists_to_remove)) {
                    $placeholders = implode(',', array_fill(0, count($lists_to_remove), '%d'));
                    $wpdb->query($wpdb->prepare(
                        "DELETE FROM {$pivot_table} WHERE subscriber_id = %d AND list_id IN ({$placeholders})",
                        array_merge([$id], $lists_to_remove)
                    ));
                }

                // Add new associations and fire triggers
                if (!empty($lists_to_add)) {
                    // Get subscriber data for trigger
                    if (!$subscriber) {
                        $subscriber = $wpdb->get_row($wpdb->prepare(
                            "SELECT * FROM {$table} WHERE id = %d",
                            $id
                        ), ARRAY_A);
                    }

                    foreach ($lists_to_add as $list_id) {
                        $wpdb->insert($pivot_table, [
                            'subscriber_id' => $id,
                            'list_id' => $list_id,
                            'subscribed_at' => current_time('mysql'),
                        ]);

                        // Fire automation trigger for list addition
                        do_action('wp_outreach_subscriber_added_to_list', $id, $list_id, $subscriber);
                    }
                }
            }
        }

        // Update tag associations if tag_ids is provided
        if ($request->has_param('tag_ids')) {
            $new_tag_ids = $request->get_param('tag_ids');
            if (is_array($new_tag_ids)) {
                // Get existing tag IDs
                $existing_tag_ids = $wpdb->get_col($wpdb->prepare(
                    "SELECT tag_id FROM {$tag_pivot_table} WHERE subscriber_id = %d",
                    $id
                ));
                $existing_tag_ids = array_map('intval', $existing_tag_ids);
                $new_tag_ids = array_map('intval', $new_tag_ids);

                // Find tags to add and remove
                $tags_to_add = array_diff($new_tag_ids, $existing_tag_ids);
                $tags_to_remove = array_diff($existing_tag_ids, $new_tag_ids);

                // Get subscriber data for triggers if not already fetched
                if (!$subscriber && (!empty($tags_to_add) || !empty($tags_to_remove))) {
                    $subscriber = $wpdb->get_row($wpdb->prepare(
                        "SELECT * FROM {$table} WHERE id = %d",
                        $id
                    ), ARRAY_A);
                }

                // Remove old tag associations and fire triggers
                if (!empty($tags_to_remove)) {
                    $placeholders = implode(',', array_fill(0, count($tags_to_remove), '%d'));
                    $wpdb->query($wpdb->prepare(
                        "DELETE FROM {$tag_pivot_table} WHERE subscriber_id = %d AND tag_id IN ({$placeholders})",
                        array_merge([$id], $tags_to_remove)
                    ));

                    // Fire automation trigger for each tag removal
                    foreach ($tags_to_remove as $tag_id) {
                        do_action('wp_outreach_tag_removed_from_subscriber', $id, $tag_id, $subscriber);
                    }
                }

                // Add new tag associations and fire triggers
                if (!empty($tags_to_add)) {
                    foreach ($tags_to_add as $tag_id) {
                        $wpdb->insert($tag_pivot_table, [
                            'subscriber_id' => $id,
                            'tag_id' => $tag_id,
                            'created_at' => current_time('mysql'),
                        ]);

                        // Fire automation trigger for tag addition
                        do_action('wp_outreach_tag_added_to_subscriber', $id, $tag_id, $subscriber);
                    }
                }
            }
        }

        return $this->get_subscriber($request);
    }

    /**
     * Delete a subscriber.
     *
     * Permanently removes a subscriber from the database.
     * Note: List associations are not explicitly deleted (consider adding CASCADE).
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Required. Subscriber ID from URL parameter.
     * }
     *
     * @return WP_REST_Response { @type bool $deleted Always true on success. }
     */
    public function delete_subscriber(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_subscribers';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    // =========================================================================
    // LIST ENDPOINTS
    // =========================================================================

    /**
     * Get all subscriber lists.
     *
     * Returns all lists ordered by creation date (newest first).
     * Each list includes subscriber_count from the lists table.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request Unused.
     *
     * @return WP_REST_Response Array of list objects.
     */
    public function get_lists(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';

        // Get lists with subscriber count
        $items = $wpdb->get_results(
            "SELECT l.*, COUNT(sl.subscriber_id) as subscriber_count
             FROM {$table} l
             LEFT JOIN {$pivot_table} sl ON l.id = sl.list_id
             GROUP BY l.id
             ORDER BY l.created_at DESC"
        );

        // Ensure numeric fields are integers
        foreach ($items as $item) {
            $item->id = (int) $item->id;
            $item->subscriber_count = (int) $item->subscriber_count;
            // Convert dates to ISO 8601 with timezone
            $item->created_at = DateHelper::toIso8601($item->created_at);
        }

        return new WP_REST_Response($items);
    }

    /**
     * Create a new subscriber list.
     *
     * Creates a list with the given name, description, and settings.
     * Auto-generates a URL-friendly slug from the name.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $name        Required. List display name.
     *     @type string $description Optional. List description.
     *     @type bool   $is_public   Optional. Whether list is publicly visible (default: true).
     *     @type bool   $double_optin Optional. Require email confirmation (default: true).
     * }
     *
     * @return WP_REST_Response The created list object with ID (HTTP 201).
     */
    public function create_list(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';

        $name = sanitize_text_field($request->get_param('name'));
        $data = [
            'name' => $name,
            'slug' => sanitize_title($name),
            'description' => sanitize_textarea_field($request->get_param('description') ?? ''),
            'is_public' => (int) ($request->get_param('is_public') ?? 1),
            'double_optin' => (int) ($request->get_param('double_optin') ?? 1),
            'created_at' => current_time('mysql'),
        ];

        $wpdb->insert($table, $data);
        $data['id'] = $wpdb->insert_id;

        return new WP_REST_Response($data, 201);
    }

    /**
     * Get a single list by ID.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Required. List ID from URL parameter.
     * }
     *
     * @return WP_REST_Response List object or 404 if not found.
     */
    public function get_list(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';
        $id = (int) $request->get_param('id');

        $item = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$item) {
            return new WP_REST_Response(['message' => 'Not found'], 404);
        }

        // Convert dates to ISO 8601 with timezone
        $item->created_at = DateHelper::toIso8601($item->created_at);

        return new WP_REST_Response($item);
    }

    /**
     * Update an existing list.
     *
     * Only provided fields are updated; omitted fields remain unchanged.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int    $id          Required. List ID from URL parameter.
     *     @type string $name        Optional. New list name.
     *     @type string $description Optional. New description.
     *     @type bool   $double_optin Optional. Enable/disable double opt-in.
     * }
     *
     * @return WP_REST_Response Updated list object (via get_list).
     */
    public function update_list(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';
        $id = (int) $request->get_param('id');

        $data = [];
        if ($request->has_param('name')) {
            $data['name'] = sanitize_text_field($request->get_param('name'));
        }
        if ($request->has_param('description')) {
            $data['description'] = sanitize_textarea_field($request->get_param('description'));
        }
        if ($request->has_param('double_optin')) {
            $data['double_optin'] = (int) $request->get_param('double_optin');
        }

        $wpdb->update($table, $data, ['id' => $id]);

        return $this->get_list($request);
    }

    /**
     * Delete a list.
     *
     * Permanently removes a list. Does NOT delete subscribers in the list,
     * only removes the list and its subscriber associations.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Required. List ID from URL parameter.
     * }
     *
     * @return WP_REST_Response { @type bool $deleted Always true on success. }
     */
    public function delete_list(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_lists';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    public function get_tags(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_tags';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';

        $items = $wpdb->get_results("
            SELECT t.*, COUNT(st.subscriber_id) as subscriber_count
            FROM {$table} t
            LEFT JOIN {$pivot_table} st ON t.id = st.tag_id
            GROUP BY t.id
            ORDER BY t.name ASC
        ");
        return new WP_REST_Response($items);
    }

    public function create_tag(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_tags';

        $name = sanitize_text_field($request->get_param('name'));
        $data = [
            'name' => $name,
            'slug' => sanitize_title($name),
            'color' => sanitize_hex_color($request->get_param('color') ?? '#6366f1'),
        ];

        $wpdb->insert($table, $data);
        $data['id'] = $wpdb->insert_id;

        return new WP_REST_Response($data, 201);
    }

    public function update_tag(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_tags';
        $id = (int) $request->get_param('id');

        $data = [];
        if ($request->has_param('name')) {
            $name = sanitize_text_field($request->get_param('name'));
            $data['name'] = $name;
            $data['slug'] = sanitize_title($name);
        }
        if ($request->has_param('color')) {
            $data['color'] = sanitize_hex_color($request->get_param('color'));
        }

        if (!empty($data)) {
            $wpdb->update($table, $data, ['id' => $id]);
        }

        $tag = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));
        return new WP_REST_Response($tag);
    }

    public function delete_tag(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_tags';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    public function get_campaigns(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $lists_table = $wpdb->prefix . 'outreach_lists';

        $page = $request->get_param('page');
        $per_page = $request->get_param('per_page');
        $status = $request->get_param('status');
        $type = $request->get_param('type');
        $search = $request->get_param('search');
        $offset = ($page - 1) * $per_page;

        // Build WHERE clause
        $where = "WHERE 1=1";
        $params = [];

        if (!empty($status)) {
            $where .= " AND status = %s";
            $params[] = $status;
        }

        if (!empty($type)) {
            $where .= " AND type = %s";
            $params[] = $type;
        }

        if (!empty($search)) {
            $where .= " AND (name LIKE %s OR subject LIKE %s)";
            $search_like = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $search_like;
            $params[] = $search_like;
        }

        // Get total count
        $count_sql = "SELECT COUNT(*) FROM {$table} {$where}";
        if (!empty($params)) {
            $count_sql = $wpdb->prepare($count_sql, ...$params);
        }
        $total = (int) $wpdb->get_var($count_sql);

        // Get items
        $sql = "SELECT * FROM {$table} {$where} ORDER BY created_at DESC LIMIT %d OFFSET %d";
        $params[] = $per_page;
        $params[] = $offset;
        $items = $wpdb->get_results($wpdb->prepare($sql, ...$params));

        // Get all lists for lookup
        $all_lists = $wpdb->get_results(
            "SELECT id, name, (SELECT COUNT(*) FROM {$wpdb->prefix}outreach_subscriber_list WHERE list_id = {$lists_table}.id) as subscriber_count FROM {$lists_table}",
            OBJECT_K
        );

        // Get queue manager for progress
        $queue = new \WPOutreach\Mailer\Queue\QueueManager();

        // Decode JSON fields and add extra info for each item
        foreach ($items as $item) {
            $item->id = (int) $item->id;

            // Decode JSON fields
            if (!empty($item->content_json)) {
                $item->content_json = json_decode($item->content_json, true);
            }
            if (!empty($item->list_ids)) {
                $item->list_ids = array_map('intval', json_decode($item->list_ids, true) ?: []);
            } else {
                $item->list_ids = [];
            }
            if (!empty($item->tag_ids)) {
                $item->tag_ids = array_map('intval', json_decode($item->tag_ids, true) ?: []);
            } else {
                $item->tag_ids = [];
            }
            if (!empty($item->recurring_config)) {
                $item->recurring_config = json_decode($item->recurring_config, true);
            }

            // Add list details
            $item->lists = [];
            $item->total_recipients = 0;
            foreach ($item->list_ids as $list_id) {
                if (isset($all_lists[$list_id])) {
                    $list = $all_lists[$list_id];
                    $item->lists[] = [
                        'id' => (int) $list->id,
                        'name' => $list->name,
                        'subscriber_count' => (int) $list->subscriber_count,
                    ];
                    $item->total_recipients += (int) $list->subscriber_count;
                }
            }

            // Add sending progress for campaigns that have been sent or are sending
            if (in_array($item->status, ['sending', 'paused', 'sent'])) {
                $item->progress = $queue->getCampaignProgress($item->id);
            }

            // Add tracking stats for sent campaigns
            if ($item->status === 'sent') {
                $openStats = \WPOutreach\Tracking\OpenTracker::getCampaignStats($item->id);
                $clickStats = \WPOutreach\Tracking\ClickTracker::getCampaignStats($item->id);

                $item->tracking = [
                    'total_sent' => $openStats['total_sent'],
                    'unique_opens' => $openStats['unique_opens'],
                    'open_rate' => $openStats['open_rate'],
                    'unique_clicks' => $clickStats['unique_clicks'],
                    'click_rate' => $clickStats['click_rate'],
                ];
            }

            // Calculate next send time for recurring campaigns
            if ($item->type === 'recurring' && $item->recurring_config && $item->status !== 'draft') {
                $config = is_array($item->recurring_config)
                    ? $item->recurring_config
                    : json_decode($item->recurring_config, true);
                if (is_array($config)) {
                    $item->next_send_at = $this->calculate_next_send_time($config);
                }
            }

            // Convert dates to ISO 8601 with timezone
            $item->created_at = DateHelper::toIso8601($item->created_at);
            $item->sent_at = DateHelper::toIso8601($item->sent_at);
            $item->scheduled_at = DateHelper::toIso8601($item->scheduled_at);
            if (isset($item->next_send_at)) {
                $item->next_send_at = DateHelper::toIso8601($item->next_send_at);
            }
        }

        return new WP_REST_Response([
            'items' => $items,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page),
        ]);
    }

    /**
     * Calculate next send time for recurring campaign
     */
    private function calculate_next_send_time(array $config): ?string {
        $frequency = $config['frequency'] ?? 'weekly';
        $time = $config['time'] ?? '09:00';

        // Use WordPress timezone for consistent calculations
        $wp_timezone = wp_timezone();
        $now = new \DateTime('now', $wp_timezone);

        switch ($frequency) {
            case 'daily':
                $next = new \DateTime("today {$time}", $wp_timezone);
                if ($next <= $now) {
                    $next = new \DateTime("tomorrow {$time}", $wp_timezone);
                }
                break;

            case 'weekly':
                $day = (int) ($config['day_of_week'] ?? 1);
                $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                $day_name = $days[$day] ?? 'Monday';
                $next = new \DateTime("this {$day_name} {$time}", $wp_timezone);
                if ($next <= $now) {
                    $next = new \DateTime("next {$day_name} {$time}", $wp_timezone);
                }
                break;

            case 'monthly':
                $day_config = $config['day_of_month'] ?? 1;

                if ($day_config === 'last') {
                    // Last day of current month
                    $next = new \DateTime('last day of this month ' . $time, $wp_timezone);
                    if ($next <= $now) {
                        // Last day of next month
                        $next = new \DateTime('last day of next month ' . $time, $wp_timezone);
                    }
                } else {
                    $day = (int) $day_config;
                    $day = max(1, min($day, 31)); // Clamp to valid range 1-31

                    // Get the number of days in current month
                    $days_in_current_month = (int) $now->format('t');
                    $effective_day = min($day, $days_in_current_month);

                    $next = new \DateTime($now->format('Y-m-') . sprintf('%02d', $effective_day) . " {$time}", $wp_timezone);
                    if ($next <= $now) {
                        // Move to next month
                        $next->modify('first day of next month');
                        $days_in_next_month = (int) $next->format('t');
                        $effective_day = min($day, $days_in_next_month);
                        $next->setDate((int)$next->format('Y'), (int)$next->format('m'), $effective_day);
                        $next = new \DateTime($next->format('Y-m-d') . " {$time}", $wp_timezone);
                    }
                }
                break;

            default:
                return null;
        }

        // Return in MySQL datetime format (WordPress local time)
        return $next->format('Y-m-d H:i:s');
    }

    public function create_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';

        $type = sanitize_text_field($request->get_param('type') ?? 'one_time');

        $data = [
            'name' => sanitize_text_field($request->get_param('name')),
            'type' => $type,
            'status' => 'draft',
            'subject' => sanitize_text_field($request->get_param('subject') ?? ''),
            'preheader' => sanitize_text_field($request->get_param('preheader') ?? ''),
            'content' => wp_kses_post($request->get_param('content') ?? ''),
            'content_json' => wp_json_encode($request->get_param('content_json') ?? []),
            'list_ids' => wp_json_encode($request->get_param('list_ids') ?? []),
            'created_at' => current_time('mysql'),
        ];

        // Handle scheduled campaigns
        if ($type === 'scheduled' && $request->has_param('scheduled_at')) {
            $data['scheduled_at'] = sanitize_text_field($request->get_param('scheduled_at'));
            $data['status'] = 'scheduled'; // Set to scheduled so cron can pick it up
        }

        // Handle recurring campaigns
        if ($type === 'recurring' && $request->has_param('recurring_config')) {
            $recurring_config = $request->get_param('recurring_config');
            $data['recurring_config'] = wp_json_encode($recurring_config);
            $data['status'] = 'scheduled'; // Set to scheduled so cron can pick it up
            if (is_array($recurring_config)) {
                $data['next_send_at'] = $this->calculate_next_send_time($recurring_config);
            }
        }

        $result = $wpdb->insert($table, $data);

        if ($result === false) {
            return new WP_Error(
                'db_error',
                'Failed to create campaign: ' . $wpdb->last_error,
                ['status' => 500]
            );
        }

        $data['id'] = $wpdb->insert_id;

        return new WP_REST_Response($data, 201);
    }

    public function get_campaign(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        $item = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$item) {
            return new WP_REST_Response(['message' => 'Not found'], 404);
        }

        // Decode JSON fields
        if (!empty($item->content_json)) {
            $item->content_json = json_decode($item->content_json, true);
        }

        // Ensure list_ids is always an array of integers
        if (!empty($item->list_ids)) {
            $decoded = json_decode($item->list_ids, true);
            $item->list_ids = is_array($decoded) ? array_map('intval', $decoded) : [];
        } else {
            $item->list_ids = [];
        }

        // Ensure tag_ids is always an array of integers
        if (!empty($item->tag_ids)) {
            $decoded = json_decode($item->tag_ids, true);
            $item->tag_ids = is_array($decoded) ? array_map('intval', $decoded) : [];
        } else {
            $item->tag_ids = [];
        }

        if (!empty($item->recurring_config)) {
            $item->recurring_config = json_decode($item->recurring_config, true);
        }

        // Ensure ID is integer
        $item->id = (int) $item->id;

        // Convert dates to ISO 8601 with timezone
        $item->created_at = DateHelper::toIso8601($item->created_at);
        $item->sent_at = DateHelper::toIso8601($item->sent_at);
        $item->scheduled_at = DateHelper::toIso8601($item->scheduled_at);

        return new WP_REST_Response($item);
    }

    public function update_campaign(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        $data = [];
        $json_fields = ['content_json', 'list_ids', 'tag_ids', 'recurring_config'];
        $allowed = ['name', 'type', 'status', 'subject', 'preheader', 'content', 'content_json', 'list_ids', 'tag_ids', 'scheduled_at', 'recurring_config'];

        foreach ($allowed as $field) {
            if ($request->has_param($field)) {
                $value = $request->get_param($field);
                if (in_array($field, $json_fields)) {
                    // Always encode arrays, even empty ones
                    $data[$field] = wp_json_encode($value ?? []);
                } elseif ($field === 'content') {
                    $data[$field] = $value; // Preserve full HTML for email content
                } else {
                    $data[$field] = sanitize_text_field($value ?? '');
                }
            }
        }

        // Auto-set status to 'scheduled' for scheduled campaigns (if not already sending/sent)
        $type = $request->get_param('type');
        $current_status = $request->get_param('status');
        if ($type === 'scheduled' && $request->has_param('scheduled_at') && !in_array($current_status, ['sending', 'sent', 'paused'])) {
            $data['status'] = 'scheduled';
        }

        // Handle recurring campaigns - calculate next_send_at
        if ($type === 'recurring' && $request->has_param('recurring_config')) {
            $recurring_config = $request->get_param('recurring_config');
            if (is_array($recurring_config)) {
                $data['next_send_at'] = $this->calculate_next_send_time($recurring_config);
                // Auto-set status to 'scheduled' if not already sending/sent/paused
                if (!in_array($current_status, ['sending', 'sent', 'paused'])) {
                    $data['status'] = 'scheduled';
                }
            }
        }

        $wpdb->update($table, $data, ['id' => $id]);

        return $this->get_campaign($request);
    }

    public function delete_campaign(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    /**
     * Send a campaign to all subscribers in selected lists
     * Add ?debug=1 to get detailed diagnostic info without actually sending
     */
    public function send_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');
        $debug = (bool) $request->get_param('debug');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        // Debug mode - return diagnostic info without sending
        if ($debug) {
            $list_ids = !empty($campaign->list_ids) ? json_decode($campaign->list_ids, true) : [];
            $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

            // Get subscriber details for debugging
            $subscriber_details = [];
            if (!empty($subscriber_ids)) {
                $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
                $placeholders = implode(',', array_fill(0, count($subscriber_ids), '%d'));
                $subscriber_details = $wpdb->get_results($wpdb->prepare(
                    "SELECT id, email, status FROM {$subscribers_table} WHERE id IN ({$placeholders})",
                    ...$subscriber_ids
                ));
            }

            // Check list subscriber counts
            $list_details = [];
            if (!empty($list_ids)) {
                $lists_table = $wpdb->prefix . 'outreach_lists';
                $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
                foreach ($list_ids as $list_id) {
                    $list = $wpdb->get_row($wpdb->prepare(
                        "SELECT l.id, l.name,
                         (SELECT COUNT(*) FROM {$pivot_table} sl WHERE sl.list_id = l.id) as total_subscribers,
                         (SELECT COUNT(*) FROM {$pivot_table} sl
                          INNER JOIN {$wpdb->prefix}outreach_subscribers s ON s.id = sl.subscriber_id
                          WHERE sl.list_id = l.id AND s.status = 'active') as active_subscribers
                         FROM {$lists_table} l WHERE l.id = %d",
                        $list_id
                    ));
                    if ($list) {
                        $list_details[] = $list;
                    }
                }
            }

            return new WP_REST_Response([
                'debug' => true,
                'campaign_id' => $id,
                'campaign_status' => $campaign->status,
                'has_subject' => !empty($campaign->subject),
                'has_content' => !empty($campaign->content),
                'content_length' => strlen($campaign->content ?? ''),
                'list_ids_raw' => $campaign->list_ids,
                'list_ids_parsed' => $list_ids,
                'lists' => $list_details,
                'subscriber_ids' => $subscriber_ids,
                'subscriber_count' => count($subscriber_ids),
                'subscriber_details' => $subscriber_details,
            ]);
        }

        // Check if already sending or sent
        if ($campaign->status === 'sending') {
            return new WP_Error('already_sending', 'Campaign is already sending', ['status' => 400]);
        }
        if ($campaign->status === 'sent') {
            return new WP_Error('already_sent', 'Campaign has already been sent', ['status' => 400]);
        }

        // Validate required fields
        if (empty($campaign->subject)) {
            return new WP_Error('missing_subject', 'Campaign must have a subject line', ['status' => 400]);
        }
        if (empty($campaign->content)) {
            return new WP_Error('missing_content', 'Campaign must have email content', ['status' => 400]);
        }

        // Get list IDs
        $list_ids = !empty($campaign->list_ids) ? json_decode($campaign->list_ids, true) : [];
        if (empty($list_ids)) {
            return new WP_Error('no_lists', 'Campaign must have at least one list selected', ['status' => 400]);
        }

        // Get subscribers from lists
        $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

        // Debug logging
        error_log("WP Outreach send_campaign: Campaign ID: {$id}");
        error_log("WP Outreach send_campaign: List IDs: " . print_r($list_ids, true));
        error_log("WP Outreach send_campaign: Subscriber IDs from lists: " . print_r($subscriber_ids, true));
        error_log("WP Outreach send_campaign: Subject: " . $campaign->subject);
        error_log("WP Outreach send_campaign: Content length: " . strlen($campaign->content ?? ''));

        if (empty($subscriber_ids)) {
            error_log("WP Outreach send_campaign: No active subscribers found");
            return new WP_Error('no_recipients', 'No active subscribers in selected lists', ['status' => 400]);
        }

        // Update campaign status to 'sending'
        $wpdb->update($table, ['status' => 'sending'], ['id' => $id]);

        // Queue emails using QueueManager
        $queue = new \WPOutreach\Mailer\Queue\QueueManager();
        $queued = $queue->queueCampaign($id, $subscriber_ids, $campaign->subject, $campaign->content);

        error_log("WP Outreach send_campaign: Queued result: {$queued}");

        // Include debug info if queued is 0 (something went wrong)
        $response = [
            'success' => true,
            'campaign_id' => $id,
            'queued' => $queued,
            'total_recipients' => count($subscriber_ids),
            'message' => sprintf('Queued %d emails for sending', $queued),
        ];

        if ($queued === 0) {
            $response['debug'] = array_merge([
                'subscriber_ids' => $subscriber_ids,
                'subject_length' => strlen($campaign->subject ?? ''),
                'content_length' => strlen($campaign->content ?? ''),
                'list_ids' => $list_ids,
            ], $queue->lastDebug);
        }

        return new WP_REST_Response($response);
    }

    /**
     * Get campaign sending progress
     */
    public function get_campaign_progress(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        // Get queue progress
        $queue = new \WPOutreach\Mailer\Queue\QueueManager();
        $progress = $queue->getCampaignProgress($id);

        // Auto-update status when complete
        if ($campaign->status === 'sending' && $progress['pending'] === 0 && $progress['total'] > 0) {
            // For recurring campaigns, reset to 'scheduled' so they get picked up again
            if ($campaign->type === 'recurring' && !empty($campaign->next_send_at)) {
                $wpdb->update($table, [
                    'status' => 'scheduled',
                    'sent_at' => current_time('mysql'),
                ], ['id' => $id]);
                $campaign->status = 'scheduled';
            } else {
                // For one-time and scheduled campaigns, mark as 'sent'
                $wpdb->update($table, [
                    'status' => 'sent',
                    'sent_at' => current_time('mysql'),
                ], ['id' => $id]);
                $campaign->status = 'sent';
            }
        }

        // For sent campaigns or empty queue, get stats from logs (same as campaign list)
        if ($campaign->status === 'sent' || $progress['total'] === 0) {
            $trackingStats = \WPOutreach\Tracking\OpenTracker::getCampaignStats($id);
            $sentFromLogs = (int) $trackingStats['total_sent'];

            // Use logs data if queue is empty but logs have data
            if ($sentFromLogs > 0) {
                $progress['total'] = $sentFromLogs;
                $progress['sent'] = $sentFromLogs;
                $progress['progress'] = 100;
            }
        }

        return new WP_REST_Response([
            'campaign_id' => $id,
            'status' => $campaign->status,
            'total' => $progress['total'],
            'sent' => $progress['sent'],
            'pending' => $progress['pending'],
            'failed' => $progress['failed'],
            'progress' => $progress['progress'],
        ]);
    }

    /**
     * Pause a sending campaign
     */
    public function pause_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        if ($campaign->status !== 'sending') {
            return new WP_Error('not_sending', 'Campaign is not currently sending', ['status' => 400]);
        }

        // Update status to paused
        $wpdb->update($table, ['status' => 'paused'], ['id' => $id]);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Campaign paused',
        ]);
    }

    /**
     * Resume a paused campaign
     */
    public function resume_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        if ($campaign->status !== 'paused') {
            return new WP_Error('not_paused', 'Campaign is not paused', ['status' => 400]);
        }

        // Update status back to sending
        $wpdb->update($table, ['status' => 'sending'], ['id' => $id]);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Campaign resumed',
        ]);
    }

    /**
     * Cancel a campaign and remove pending emails from queue
     */
    public function cancel_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        if (!in_array($campaign->status, ['sending', 'paused'], true)) {
            return new WP_Error('invalid_status', 'Campaign cannot be cancelled (must be sending or paused)', ['status' => 400]);
        }

        // Get current progress before cancelling
        $queue = new \WPOutreach\Mailer\Queue\QueueManager();
        $progress = $queue->getCampaignProgress($id);

        // Remove pending emails from queue
        $cancelled = $queue->cancelCampaign($id);

        // Update campaign status to 'cancelled' or 'sent' based on whether any emails were sent
        $new_status = $progress['sent'] > 0 ? 'sent' : 'draft';
        $wpdb->update($table, ['status' => $new_status], ['id' => $id]);

        return new WP_REST_Response([
            'success' => true,
            'cancelled' => $cancelled,
            'already_sent' => $progress['sent'],
            'status' => $new_status,
            'message' => sprintf('Cancelled %d pending emails', $cancelled),
        ]);
    }

    /**
     * Send a test email for a campaign
     */
    public function test_campaign(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $id = (int) $request->get_param('id');
        $test_email = sanitize_email($request->get_param('email'));

        if (empty($test_email) || !is_email($test_email)) {
            return new WP_Error('invalid_email', 'Please provide a valid email address', ['status' => 400]);
        }

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        // Validate required fields
        if (empty($campaign->subject)) {
            return new WP_Error('missing_subject', 'Campaign must have a subject line', ['status' => 400]);
        }
        if (empty($campaign->content)) {
            return new WP_Error('missing_content', 'Campaign must have email content', ['status' => 400]);
        }

        // Create mock subscriber data for personalization
        $current_user = wp_get_current_user();
        $email_parts = explode('@', $test_email);

        $mock_subscriber = [
            'id' => 0,
            'email' => $test_email,
            'first_name' => $current_user->first_name ?: ucfirst($email_parts[0]),
            'last_name' => $current_user->last_name ?: 'Subscriber',
            'subscriber_id' => 'test-' . time(),
        ];

        // Add site variables
        $mock_subscriber['site_name'] = get_bloginfo('name');
        $mock_subscriber['site_url'] = home_url();
        $mock_subscriber['current_date'] = current_time('Y-m-d');
        $mock_subscriber['current_year'] = current_time('Y');
        $mock_subscriber['unsubscribe_url'] = '#unsubscribe-preview';

        // Parse variables in subject and content
        $template_engine = new \WPOutreach\Mailer\Template\TemplateEngine();
        $parsed_subject = $template_engine->parseVariables('[TEST] ' . $campaign->subject, $mock_subscriber);

        // Campaign content is already a complete HTML document from the email builder,
        // so don't wrap it in the default layout (pass false for wrap parameter)
        $parsed_content = $template_engine->render($campaign->content, $mock_subscriber, false);

        // Send using the configured mailer (direct, not queued)
        $mailer = \WPOutreach\Mailer\MailerFactory::getInstance();
        $sent = $mailer->send($test_email, $parsed_subject, $parsed_content);

        if ($sent) {
            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf('Test email sent to %s', $test_email),
                'mailer' => $mailer->getName(),
            ]);
        }

        return new WP_REST_Response([
            'success' => false,
            'message' => $mailer->getLastError() ?? 'Failed to send test email',
            'mailer' => $mailer->getName(),
        ], 500);
    }

    /**
     * Get campaign statistics
     *
     * Returns aggregated stats for a campaign including open rates, click rates,
     * link breakdown, and timeline data.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_campaign_stats(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $links_table = $wpdb->prefix . 'outreach_links';

        $id = (int) $request->get_param('id');

        // Get campaign
        $campaign = $wpdb->get_row($wpdb->prepare(
            "SELECT id, name, subject, status, sent_at, created_at FROM {$campaigns_table} WHERE id = %d",
            $id
        ));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        // Get aggregate stats from logs
        $stats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total_sent,
                SUM(CASE WHEN status != 'bounced' AND status != 'complained' THEN 1 ELSE 0 END) as delivered,
                SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as unique_opens,
                SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END) as unique_clicks,
                SUM(opens) as total_opens,
                SUM(clicks) as total_clicks,
                SUM(CASE WHEN status = 'bounced' THEN 1 ELSE 0 END) as bounced,
                SUM(CASE WHEN status = 'complained' THEN 1 ELSE 0 END) as complained
             FROM {$logs_table}
             WHERE campaign_id = %d",
            $id
        ));

        $total_sent = (int) ($stats->total_sent ?? 0);
        $delivered = (int) ($stats->delivered ?? 0);
        $unique_opens = (int) ($stats->unique_opens ?? 0);
        $unique_clicks = (int) ($stats->unique_clicks ?? 0);

        // Calculate rates
        $open_rate = $delivered > 0 ? round(($unique_opens / $delivered) * 100, 1) : 0;
        $click_rate = $delivered > 0 ? round(($unique_clicks / $delivered) * 100, 1) : 0;
        $click_to_open_rate = $unique_opens > 0 ? round(($unique_clicks / $unique_opens) * 100, 1) : 0;
        $bounce_rate = $total_sent > 0 ? round(((int) $stats->bounced / $total_sent) * 100, 1) : 0;

        // Get link click breakdown
        $links = $wpdb->get_results($wpdb->prepare(
            "SELECT url, clicks, unique_clicks
             FROM {$links_table}
             WHERE campaign_id = %d
             ORDER BY clicks DESC
             LIMIT 10",
            $id
        ));

        // Get daily stats for chart (last 7 days or since campaign sent)
        $daily_stats = $wpdb->get_results($wpdb->prepare(
            "SELECT
                DATE(sent_at) as date,
                COUNT(*) as sent,
                SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as opens,
                SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END) as clicks
             FROM {$logs_table}
             WHERE campaign_id = %d
             GROUP BY DATE(sent_at)
             ORDER BY date ASC",
            $id
        ));

        // Get hourly open/click distribution
        $hourly_stats = $wpdb->get_results($wpdb->prepare(
            "SELECT
                HOUR(first_opened_at) as hour,
                COUNT(*) as opens
             FROM {$logs_table}
             WHERE campaign_id = %d AND first_opened_at IS NOT NULL
             GROUP BY HOUR(first_opened_at)
             ORDER BY hour ASC",
            $id
        ));

        return new WP_REST_Response([
            'campaign' => [
                'id' => (int) $campaign->id,
                'name' => $campaign->name,
                'subject' => $campaign->subject,
                'status' => $campaign->status,
                'sent_at' => DateHelper::toIso8601($campaign->sent_at),
            ],
            'overview' => [
                'total_sent' => $total_sent,
                'delivered' => $delivered,
                'unique_opens' => $unique_opens,
                'unique_clicks' => $unique_clicks,
                'total_opens' => (int) ($stats->total_opens ?? 0),
                'total_clicks' => (int) ($stats->total_clicks ?? 0),
                'bounced' => (int) ($stats->bounced ?? 0),
                'complained' => (int) ($stats->complained ?? 0),
            ],
            'rates' => [
                'open_rate' => $open_rate,
                'click_rate' => $click_rate,
                'click_to_open_rate' => $click_to_open_rate,
                'bounce_rate' => $bounce_rate,
            ],
            'links' => array_map(function($link) {
                return [
                    'url' => $link->url,
                    'clicks' => (int) $link->clicks,
                    'unique_clicks' => (int) $link->unique_clicks,
                ];
            }, $links),
            'daily_stats' => $daily_stats,
            'hourly_stats' => $hourly_stats,
        ]);
    }

    /**
     * Get campaign recipients with their individual stats
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_campaign_recipients(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        $id = (int) $request->get_param('id');
        $page = max(1, (int) $request->get_param('page'));
        $per_page = min(100, max(1, (int) $request->get_param('per_page')));
        $offset = ($page - 1) * $per_page;
        $status = sanitize_text_field($request->get_param('status'));
        $search = sanitize_text_field($request->get_param('search'));

        // Verify campaign exists
        $campaign = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$campaigns_table} WHERE id = %d",
            $id
        ));

        if (!$campaign) {
            return new WP_Error('not_found', 'Campaign not found', ['status' => 404]);
        }

        // Build WHERE clause
        $where = ['l.campaign_id = %d'];
        $params = [$id];

        if (!empty($status)) {
            if ($status === 'opened') {
                $where[] = 'l.opens > 0';
            } elseif ($status === 'clicked') {
                $where[] = 'l.clicks > 0';
            } elseif ($status === 'not_opened') {
                $where[] = 'l.opens = 0';
            } else {
                $where[] = 'l.status = %s';
                $params[] = $status;
            }
        }

        if (!empty($search)) {
            $where[] = '(l.email LIKE %s OR s.first_name LIKE %s OR s.last_name LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $search_term;
            $params[] = $search_term;
            $params[] = $search_term;
        }

        $where_clause = implode(' AND ', $where);

        // Get total count
        $count_sql = "SELECT COUNT(*) FROM {$logs_table} l
                      LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
                      WHERE {$where_clause}";
        $total = (int) $wpdb->get_var($wpdb->prepare($count_sql, ...$params));

        // Get recipients
        $sql = "SELECT
                    l.id,
                    l.email,
                    l.subscriber_id,
                    s.first_name,
                    s.last_name,
                    l.status,
                    l.opens,
                    l.clicks,
                    l.first_opened_at,
                    l.last_opened_at,
                    l.sent_at
                FROM {$logs_table} l
                LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
                WHERE {$where_clause}
                ORDER BY l.sent_at DESC
                LIMIT %d OFFSET %d";

        $query_params = array_merge($params, [$per_page, $offset]);
        $recipients = $wpdb->get_results($wpdb->prepare($sql, ...$query_params));

        // Format recipients
        $formatted = array_map(function($recipient) {
            return [
                'id' => (int) $recipient->id,
                'email' => $recipient->email,
                'subscriber_id' => $recipient->subscriber_id ? (int) $recipient->subscriber_id : null,
                'name' => trim(($recipient->first_name ?? '') . ' ' . ($recipient->last_name ?? '')) ?: null,
                'status' => $recipient->status,
                'opens' => (int) $recipient->opens,
                'clicks' => (int) $recipient->clicks,
                'first_opened_at' => DateHelper::toIso8601($recipient->first_opened_at),
                'last_opened_at' => DateHelper::toIso8601($recipient->last_opened_at),
                'sent_at' => DateHelper::toIso8601($recipient->sent_at),
            ];
        }, $recipients);

        return new WP_REST_Response([
            'items' => $formatted,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => (int) ceil($total / $per_page),
        ]);
    }

    /**
     * Get subscribers for given list IDs
     */
    private function get_subscribers_for_lists(array $list_ids): array {
        global $wpdb;

        if (empty($list_ids)) {
            return [];
        }

        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';

        $placeholders = implode(',', array_fill(0, count($list_ids), '%d'));

        return $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT s.id
             FROM {$subscribers_table} s
             INNER JOIN {$pivot_table} sl ON s.id = sl.subscriber_id
             WHERE sl.list_id IN ({$placeholders})
             AND s.status = 'active'",
            ...$list_ids
        ));
    }

    public function get_automations(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';
        $queue_table = $wpdb->prefix . 'outreach_automation_queue';
        $items = $wpdb->get_results("SELECT * FROM {$table} ORDER BY created_at DESC");

        // Get queue stats for each automation
        foreach ($items as $item) {
            $item->id = (int) $item->id;

            // Parse JSON fields
            if (!empty($item->trigger_config)) {
                $item->trigger_config = json_decode($item->trigger_config, true);
            } else {
                $item->trigger_config = [];
            }
            if (!empty($item->steps)) {
                $item->steps = json_decode($item->steps, true);
            } else {
                $item->steps = [];
            }
            if (!empty($item->stats)) {
                $item->stats = json_decode($item->stats, true);
            }

            // Get active subscriber count in this automation
            $item->active_subscribers = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$queue_table} WHERE automation_id = %d AND status = 'active'",
                $item->id
            ));

            // Get completed count
            $item->completed_count = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$queue_table} WHERE automation_id = %d AND status = 'completed'",
                $item->id
            ));

            // Convert dates to ISO 8601 with timezone
            $item->created_at = DateHelper::toIso8601($item->created_at);
        }

        return new WP_REST_Response($items);
    }

    public function create_automation(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $trigger_type = sanitize_text_field($request->get_param('trigger_type') ?? 'added_to_list');

        // Check if trigger is available for current license
        if (!FeatureManager::canUseTrigger($trigger_type)) {
            return new WP_Error(
                'feature_restricted',
                FeatureManager::getRestrictionMessage('advanced_triggers'),
                [
                    'status' => 403,
                    'upgrade_url' => FeatureManager::getUpgradeUrl(),
                    'allowed_triggers' => FeatureManager::getAvailableTriggers(),
                ]
            );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';

        // Get trigger config and auto-generate secret key for incoming_webhook
        $trigger_config = $request->get_param('trigger_config') ?? [];
        if ($trigger_type === 'incoming_webhook' && empty($trigger_config['secret_key'])) {
            $trigger_config['secret_key'] = 'whsec_' . wp_generate_password(32, false);
        }

        $data = [
            'name' => sanitize_text_field($request->get_param('name')),
            'status' => sanitize_text_field($request->get_param('status') ?? 'draft'),
            'trigger_type' => $trigger_type,
            'trigger_config' => wp_json_encode($trigger_config),
            'steps' => wp_json_encode($request->get_param('steps') ?? []),
            'created_at' => current_time('mysql'),
        ];

        $wpdb->insert($table, $data);
        $data['id'] = $wpdb->insert_id;

        return new WP_REST_Response($data, 201);
    }

    public function get_automation(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';
        $id = (int) $request->get_param('id');

        $item = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$item) {
            return new WP_REST_Response(['message' => 'Not found'], 404);
        }

        // Parse JSON fields
        $item->id = (int) $item->id;
        if (!empty($item->trigger_config)) {
            $item->trigger_config = json_decode($item->trigger_config, true);
        } else {
            $item->trigger_config = [];
        }
        if (!empty($item->steps)) {
            $item->steps = json_decode($item->steps, true);
        } else {
            $item->steps = [];
        }
        if (!empty($item->stats)) {
            $item->stats = json_decode($item->stats, true);
        }

        // Ensure webhook URL is always current for incoming_webhook triggers
        if ($item->trigger_type === 'incoming_webhook' && !empty($item->trigger_config['webhook_key'])) {
            $item->trigger_config['webhook_url'] = rest_url("outreach/v1/webhooks/incoming/{$item->trigger_config['webhook_key']}");
        }

        // Convert dates to ISO 8601 with timezone
        $item->created_at = DateHelper::toIso8601($item->created_at);

        return new WP_REST_Response($item);
    }

    public function update_automation(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';
        $id = (int) $request->get_param('id');

        // Check trigger restrictions for free users
        if ($request->has_param('trigger_type')) {
            $trigger_type = sanitize_text_field($request->get_param('trigger_type'));

            if (!FeatureManager::canUseTrigger($trigger_type)) {
                return new WP_Error(
                    'feature_restricted',
                    FeatureManager::getRestrictionMessage('advanced_triggers'),
                    [
                        'status' => 403,
                        'upgrade_url' => FeatureManager::getUpgradeUrl(),
                        'allowed_triggers' => FeatureManager::getAvailableTriggers(),
                    ]
                );
            }
        }

        $data = [];
        if ($request->has_param('name')) {
            $data['name'] = sanitize_text_field($request->get_param('name'));
        }
        if ($request->has_param('status')) {
            $data['status'] = sanitize_text_field($request->get_param('status'));
        }
        if ($request->has_param('trigger_type')) {
            $data['trigger_type'] = sanitize_text_field($request->get_param('trigger_type'));
        }
        if ($request->has_param('trigger_config')) {
            $trigger_config = $request->get_param('trigger_config');

            // Auto-generate secret key for incoming_webhook if not present
            $new_trigger_type = $request->get_param('trigger_type') ?? '';
            if ($new_trigger_type === 'incoming_webhook' && empty($trigger_config['secret_key'])) {
                $trigger_config['secret_key'] = 'whsec_' . wp_generate_password(32, false);
            }

            $data['trigger_config'] = wp_json_encode($trigger_config);
        }
        if ($request->has_param('steps')) {
            $data['steps'] = wp_json_encode($request->get_param('steps'));
        }

        $wpdb->update($table, $data, ['id' => $id]);

        return $this->get_automation($request);
    }

    public function delete_automation(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    /**
     * Get available automation triggers
     *
     * Returns all registered triggers grouped by category.
     * Third-party plugins can register their own triggers via the
     * 'wp_outreach_register_triggers' action.
     *
     * @since 1.0.0
     */
    public function get_automation_triggers(WP_REST_Request $request): WP_REST_Response {
        $triggers = \WPOutreach\Automation\TriggerRegistry::forApi();
        return new WP_REST_Response($triggers);
    }

    /**
     * Get available automation actions
     *
     * Returns all registered actions grouped by category.
     * Third-party plugins can register their own actions via the
     * 'wp_outreach_register_actions' action.
     *
     * @since 1.0.0
     */
    public function get_automation_actions(WP_REST_Request $request): WP_REST_Response {
        $actions = \WPOutreach\Automation\ActionRegistry::forApi();
        return new WP_REST_Response($actions);
    }

    /**
     * Get condition options for the condition builder UI
     *
     * Returns available operators and fields for building conditions.
     * When trigger_type is provided, returns trigger-specific fields with
     * full metadata (type, description, options, etc.)
     *
     * @since 1.3.0
     *
     * @param WP_REST_Request $request {
     *     @type string $trigger_type Optional. Trigger type to get context-specific fields.
     * }
     *
     * @return WP_REST_Response {
     *     @type array  $operators      All available comparison operators.
     *     @type array  $fields         Available fields grouped by category.
     *     @type array  $field_types    Field type configurations with operators.
     *     @type array  $unary_operators Operators that don't require a value.
     * }
     */
    public function get_condition_options(WP_REST_Request $request): WP_REST_Response {
        $trigger_type = $request->get_param('trigger_type');

        $operators = \WPOutreach\Automation\ConditionEvaluator::OPERATORS;
        $fields = \WPOutreach\Automation\ConditionEvaluator::getFieldsForTrigger($trigger_type);
        $field_types = \WPOutreach\Automation\ConditionEvaluator::FIELD_TYPES;
        $unary_operators = \WPOutreach\Automation\ConditionEvaluator::UNARY_OPERATORS;

        // Format fields for frontend - preserve full metadata
        $formatted_fields = [];
        foreach ($fields as $group_key => $group) {
            $formatted_fields[] = [
                'key' => $group_key,
                'label' => $group['label'] ?? ucfirst($group_key),
                'icon' => $group['icon'] ?? 'circle',
                'description' => $group['description'] ?? '',
                'dynamic' => $group['dynamic'] ?? false,
                'fields' => array_map(function ($field) {
                    return [
                        'value' => $field['value'],
                        'label' => $field['label'],
                        'type' => $field['type'] ?? 'string',
                        'description' => $field['description'] ?? '',
                        'options' => $field['options'] ?? [],
                        'placeholder' => $field['placeholder'] ?? '',
                        'suffix' => $field['suffix'] ?? '',
                        'computed' => $field['computed'] ?? false,
                        'editable' => $field['editable'] ?? false,
                    ];
                }, $group['fields'] ?? []),
            ];
        }

        return new WP_REST_Response([
            'operators' => $operators,
            'fields' => $formatted_fields,
            'field_types' => $field_types,
            'unary_operators' => $unary_operators,
        ]);
    }

    /**
     * Duplicate an automation
     *
     * Creates a copy of the specified automation with "(Copy)" appended to the name
     * and status set to 'draft'. Queue data is not copied.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Required. Automation ID to duplicate.
     * }
     *
     * @return WP_REST_Response|WP_Error The newly created automation or error if not found.
     */
    public function duplicate_automation(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_automations';
        $id = (int) $request->get_param('id');

        // Get the original automation
        $original = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $id
        ));

        if (!$original) {
            return new WP_Error('not_found', 'Automation not found', ['status' => 404]);
        }

        // Create the duplicate with "(Copy)" suffix and draft status
        $data = [
            'name' => $original->name . ' (Copy)',
            'status' => 'draft',
            'trigger_type' => $original->trigger_type,
            'trigger_config' => $original->trigger_config,
            'steps' => $original->steps,
            'stats' => null, // Reset stats for the copy
            'created_at' => current_time('mysql'),
        ];

        $wpdb->insert($table, $data);
        $new_id = $wpdb->insert_id;

        // Return the new automation
        $new_request = new WP_REST_Request();
        $new_request->set_param('id', $new_id);
        return $this->get_automation($new_request);
    }

    public function get_templates(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';
        $items = $wpdb->get_results("SELECT * FROM {$table} ORDER BY created_at DESC");

        // Decode JSON fields for each item
        foreach ($items as $item) {
            if (!empty($item->content_json)) {
                $item->content_json = json_decode($item->content_json, true);
            }
            // Convert dates to ISO 8601 with timezone
            $item->created_at = DateHelper::toIso8601($item->created_at);
        }

        return new WP_REST_Response($items);
    }

    public function create_template(WP_REST_Request $request): WP_REST_Response|WP_Error {
        // Check if templates feature is available
        if (!FeatureManager::can('templates')) {
            return new WP_Error(
                'feature_restricted',
                FeatureManager::getRestrictionMessage('templates'),
                [
                    'status' => 403,
                    'upgrade_url' => FeatureManager::getUpgradeUrl(),
                ]
            );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';

        $name = sanitize_text_field($request->get_param('name'));
        $data = [
            'name' => $name,
            'slug' => sanitize_title($name),
            'subject' => sanitize_text_field($request->get_param('subject') ?? ''),
            'content' => $request->get_param('content') ?? '', // Preserve full HTML for email templates
            'content_json' => wp_json_encode($request->get_param('content_json') ?? []),
            'created_at' => current_time('mysql'),
        ];

        $wpdb->insert($table, $data);
        $data['id'] = $wpdb->insert_id;

        return new WP_REST_Response($data, 201);
    }

    public function get_template(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';
        $id = (int) $request->get_param('id');

        $item = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id));

        if (!$item) {
            return new WP_REST_Response(['message' => 'Not found'], 404);
        }

        // Decode JSON fields
        if (!empty($item->content_json)) {
            $item->content_json = json_decode($item->content_json, true);
        }

        // Convert dates to ISO 8601 with timezone
        $item->created_at = DateHelper::toIso8601($item->created_at);

        return new WP_REST_Response($item);
    }

    public function update_template(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';
        $id = (int) $request->get_param('id');

        $data = [];
        $allowed = ['name', 'subject', 'content', 'content_json'];

        foreach ($allowed as $field) {
            if ($request->has_param($field)) {
                $value = $request->get_param($field);
                if ($field === 'content_json') {
                    $data[$field] = wp_json_encode($value);
                } elseif ($field === 'content') {
                    $data[$field] = $value; // Preserve full HTML for email content
                } else {
                    $data[$field] = sanitize_text_field($value);
                }
            }
        }

        $wpdb->update($table, $data, ['id' => $id]);

        return $this->get_template($request);
    }

    public function delete_template(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_templates';
        $id = (int) $request->get_param('id');

        $wpdb->delete($table, ['id' => $id]);

        return new WP_REST_Response(['deleted' => true]);
    }

    public function get_settings(WP_REST_Request $request): WP_REST_Response {
        // Ensure cron secret exists
        $cron_settings = get_option('wp_outreach_cron', []);
        if (empty($cron_settings['secret'])) {
            $cron_settings['secret'] = wp_generate_password(32, false);
            $cron_settings['mode'] = $cron_settings['mode'] ?? 'wp_cron';
            update_option('wp_outreach_cron', $cron_settings);
        }

        $settings = [
            'general' => get_option('wp_outreach_general', []),
            'mailer' => get_option('wp_outreach_mailer', []),
            'smtp' => get_option('wp_outreach_smtp', []),
            'ses' => get_option('wp_outreach_ses', []),
            'subscription' => get_option('wp_outreach_subscription', []),
            'tracking' => get_option('wp_outreach_tracking', []),
            'cron' => $cron_settings,
            'post_subscriptions' => get_option('wp_outreach_post_subscriptions', [
                'enabled' => false,
                'post_types' => ['post'],
                'display_mode' => 'after_content',
                'show_on' => 'single',
                'heading' => 'Get notified when this content is updated',
                'description' => 'Enter your email to receive updates about this post.',
                'button_text' => 'Notify Me',
                'success_message' => 'You\'ll be notified when this is updated!',
                'show_subscriber_count' => true,
                'float_button_text' => 'Get Updates',
                'float_position' => 'bottom-right',
            ]),
        ];

        // Add cron URL for display
        $settings['cron']['url'] = rest_url('outreach/v1/cron') . '?key=' . $cron_settings['secret'];

        // Remove sensitive data
        if (isset($settings['smtp']['password'])) {
            $settings['smtp']['password'] = $settings['smtp']['password'] ? '********' : '';
        }
        if (isset($settings['ses']['secret_key'])) {
            $settings['ses']['secret_key'] = $settings['ses']['secret_key'] ? '********' : '';
        }

        return new WP_REST_Response($settings);
    }

    public function update_settings(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $sections = ['general', 'mailer', 'smtp', 'ses', 'subscription', 'tracking', 'cron', 'post_subscriptions'];

        foreach ($sections as $section) {
            if ($request->has_param($section)) {
                $data = $request->get_param($section);

                // Silently disable pro features for free users (since all sections are saved at once)
                if ($section === 'post_subscriptions' && !FeatureManager::can('post_subscriptions')) {
                    $data['enabled'] = false;
                }

                if ($section === 'mailer') {
                    $driver = $data['driver'] ?? 'wp_mail';
                    // Reset to wp_mail if trying to use a pro mailer without license
                    if ($driver === 'smtp' && !FeatureManager::can('smtp')) {
                        $data['driver'] = 'wp_mail';
                    }
                    if ($driver === 'ses' && !FeatureManager::can('ses')) {
                        $data['driver'] = 'wp_mail';
                    }
                }

                if ($section === 'tracking' && !FeatureManager::can('tracking')) {
                    // Disable tracking features for free users
                    $data['open_tracking'] = false;
                    $data['click_tracking'] = false;
                }

                // Don't overwrite passwords if masked
                if ($section === 'smtp' && isset($data['password']) && $data['password'] === '********') {
                    $existing = get_option('wp_outreach_smtp', []);
                    $data['password'] = $existing['password'] ?? '';
                }
                if ($section === 'ses' && isset($data['secret_key']) && $data['secret_key'] === '********') {
                    $existing = get_option('wp_outreach_ses', []);
                    $data['secret_key'] = $existing['secret_key'] ?? '';
                }
                // Preserve cron secret (it's managed separately)
                if ($section === 'cron') {
                    $existing = get_option('wp_outreach_cron', []);
                    $data['secret'] = $existing['secret'] ?? wp_generate_password(32, false);
                }

                update_option('wp_outreach_' . $section, $data);
            }
        }

        return $this->get_settings($request);
    }

    public function send_test_email(WP_REST_Request $request): WP_REST_Response {
        $to = sanitize_email($request->get_param('email'));
        $subject = __('Test Email from WP Outreach', 'outreach');
        $message = __('This is a test email to verify your email configuration is working correctly.', 'outreach');

        // Use the configured mailer
        $mailer = \WPOutreach\Mailer\MailerFactory::getInstance();
        $sent = $mailer->send($to, $subject, $message);

        if ($sent) {
            return new WP_REST_Response([
                'success' => true,
                'message' => __('Test email sent successfully!', 'outreach'),
                'mailer' => $mailer->getName(),
            ]);
        }

        return new WP_REST_Response([
            'success' => false,
            'message' => $mailer->getLastError() ?? __('Failed to send test email.', 'outreach'),
            'mailer' => $mailer->getName(),
        ], 500);
    }

    /**
     * Regenerate cron secret key
     */
    public function regenerate_cron_key(WP_REST_Request $request): WP_REST_Response {
        $cron_settings = get_option('wp_outreach_cron', []);
        $cron_settings['secret'] = wp_generate_password(32, false);
        update_option('wp_outreach_cron', $cron_settings);

        return new WP_REST_Response([
            'success' => true,
            'secret' => $cron_settings['secret'],
            'url' => rest_url('outreach/v1/cron') . '?key=' . $cron_settings['secret'],
        ]);
    }

    /**
     * Get available post types for post subscriptions settings
     */
    public function get_available_post_types(WP_REST_Request $request): WP_REST_Response {
        $post_types = get_post_types(['public' => true], 'objects');
        $result = [];

        foreach ($post_types as $post_type) {
            // Exclude attachment
            if ($post_type->name === 'attachment') {
                continue;
            }

            $result[] = [
                'name' => $post_type->name,
                'label' => $post_type->labels->singular_name,
                'label_plural' => $post_type->labels->name,
            ];
        }

        return new WP_REST_Response($result);
    }

    /**
     * Get posts of a specific type for admin dropdowns
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $post_type Required. Post type to fetch.
     *     @type int    $per_page  Optional. Items per page (default: 50).
     *     @type string $search    Optional. Search query for title.
     * }
     *
     * @return WP_REST_Response Array of posts with id and title.
     */
    public function get_posts_by_type(WP_REST_Request $request): WP_REST_Response {
        $post_type = sanitize_key($request->get_param('post_type'));
        $per_page = min(100, max(1, (int) $request->get_param('per_page')));
        $search = sanitize_text_field($request->get_param('search'));

        // Validate post type
        if (!post_type_exists($post_type)) {
            return new WP_REST_Response([]);
        }

        $args = [
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'orderby' => 'title',
            'order' => 'ASC',
        ];

        if (!empty($search)) {
            $args['s'] = $search;
        }

        $posts = get_posts($args);
        $result = [];

        foreach ($posts as $post) {
            $result[] = [
                'id' => $post->ID,
                'title' => $post->post_title ?: '(no title)',
            ];
        }

        return new WP_REST_Response($result);
    }

    // =========================================================================
    // USER ACCESS MANAGEMENT HANDLERS
    // =========================================================================

    /**
     * Get list of users with WP Outreach access
     */
    public function get_allowed_users(WP_REST_Request $request): WP_REST_Response {
        $allowed_user_ids = get_option('wp_outreach_allowed_users', []);
        $users = [];

        foreach ($allowed_user_ids as $user_id) {
            $user = get_user_by('ID', $user_id);
            if ($user) {
                $users[] = [
                    'id' => $user->ID,
                    'display_name' => $user->display_name,
                    'email' => $user->user_email,
                    'avatar' => get_avatar_url($user->ID, ['size' => 40]),
                    'roles' => $user->roles,
                ];
            }
        }

        return new WP_REST_Response($users);
    }

    /**
     * Add a user to the allowed users list
     */
    public function add_allowed_user(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $user_id = (int) $request->get_param('user_id');

        if (!$user_id) {
            return new WP_Error('invalid_user', 'User ID is required', ['status' => 400]);
        }

        $user = get_user_by('ID', $user_id);
        if (!$user) {
            return new WP_Error('user_not_found', 'User not found', ['status' => 404]);
        }

        // Don't add users who already have manage_options capability
        if (user_can($user_id, 'manage_options')) {
            return new WP_Error('already_admin', 'This user already has full access as a site administrator', ['status' => 400]);
        }

        $allowed_users = get_option('wp_outreach_allowed_users', []);

        if (in_array($user_id, $allowed_users, true)) {
            return new WP_Error('already_added', 'User already has access', ['status' => 400]);
        }

        $allowed_users[] = $user_id;
        update_option('wp_outreach_allowed_users', $allowed_users);

        return new WP_REST_Response([
            'success' => true,
            'user' => [
                'id' => $user->ID,
                'display_name' => $user->display_name,
                'email' => $user->user_email,
                'avatar' => get_avatar_url($user->ID, ['size' => 40]),
                'roles' => $user->roles,
            ],
        ]);
    }

    /**
     * Remove a user from the allowed users list
     */
    public function remove_allowed_user(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $user_id = (int) $request->get_param('user_id');

        if (!$user_id) {
            return new WP_Error('invalid_user', 'User ID is required', ['status' => 400]);
        }

        $allowed_users = get_option('wp_outreach_allowed_users', []);
        $key = array_search($user_id, $allowed_users, true);

        if ($key === false) {
            return new WP_Error('not_found', 'User is not in the allowed list', ['status' => 404]);
        }

        unset($allowed_users[$key]);
        $allowed_users = array_values($allowed_users); // Re-index array
        update_option('wp_outreach_allowed_users', $allowed_users);

        return new WP_REST_Response(['success' => true]);
    }

    /**
     * Search users for adding to allowed list
     */
    public function search_users(WP_REST_Request $request): WP_REST_Response {
        $search = sanitize_text_field($request->get_param('search'));

        if (strlen($search) < 2) {
            return new WP_REST_Response([]);
        }

        $allowed_users = get_option('wp_outreach_allowed_users', []);

        $args = [
            'search' => '*' . $search . '*',
            'search_columns' => ['user_login', 'user_email', 'display_name'],
            'number' => 10,
            'orderby' => 'display_name',
            'order' => 'ASC',
        ];

        $user_query = new \WP_User_Query($args);
        $users = [];

        foreach ($user_query->get_results() as $user) {
            // Skip users who already have access
            if (user_can($user->ID, 'manage_options') || in_array($user->ID, $allowed_users, true)) {
                continue;
            }

            $users[] = [
                'id' => $user->ID,
                'display_name' => $user->display_name,
                'email' => $user->user_email,
                'avatar' => get_avatar_url($user->ID, ['size' => 40]),
                'roles' => $user->roles,
            ];
        }

        return new WP_REST_Response($users);
    }

    // =========================================================================
    // SES MANAGEMENT HANDLERS
    // =========================================================================

    /**
     * Get available SES regions
     */
    public function get_ses_regions(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response(\WPOutreach\Mailer\SES\SESRegions::toArray());
    }

    /**
     * Get SES account status
     */
    public function get_ses_status(WP_REST_Request $request): WP_REST_Response {
        $manager = new \WPOutreach\Mailer\SES\SESManager();
        return new WP_REST_Response($manager->getAccountStatus());
    }

    /**
     * Get SES sending statistics
     */
    public function get_ses_statistics(WP_REST_Request $request): WP_REST_Response {
        $manager = new \WPOutreach\Mailer\SES\SESManager();
        return new WP_REST_Response($manager->getSendStatistics());
    }

    /**
     * Test SES connection
     */
    public function test_ses_connection(WP_REST_Request $request): WP_REST_Response {
        // Check if temporary credentials are provided
        $accessKey = $request->get_param('access_key');
        $secretKey = $request->get_param('secret_key');
        $region = $request->get_param('region');
        $saveOnSuccess = $request->get_param('save_on_success') ?? false;

        if ($accessKey && $secretKey && $secretKey !== '********') {
            // Save temporary credentials for testing
            $tempSettings = [
                'access_key' => sanitize_text_field($accessKey),
                'secret_key' => sanitize_text_field($secretKey),
                'region' => sanitize_text_field($region ?: 'us-east-1'),
            ];

            // Temporarily update the option
            $oldSettings = get_option('wp_outreach_ses', []);
            update_option('wp_outreach_ses', $tempSettings);

            $credentials = new \WPOutreach\Mailer\SES\SESCredentials();
            $manager = new \WPOutreach\Mailer\SES\SESManager($credentials);
            $result = $manager->testConnection();

            // If successful, include account status in response
            if ($result['success']) {
                $result['account_status'] = $manager->getAccountStatus();

                // If save_on_success is true, keep the new settings
                if ($saveOnSuccess) {
                    // Settings already saved, don't restore
                } else {
                    // Restore old settings
                    update_option('wp_outreach_ses', $oldSettings);
                }
            } else {
                // Restore old settings on failure
                update_option('wp_outreach_ses', $oldSettings);
            }

            return new WP_REST_Response($result);
        }

        // Use stored credentials
        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->testConnection();

        // Include account status on success
        if ($result['success']) {
            $result['account_status'] = $manager->getAccountStatus();
        }

        return new WP_REST_Response($result);
    }

    /**
     * Get all SES identities
     */
    public function get_ses_identities(WP_REST_Request $request): WP_REST_Response {
        $manager = new \WPOutreach\Mailer\SES\SESManager();
        return new WP_REST_Response($manager->getAllIdentitiesWithStatus());
    }

    /**
     * Check if a sender email is verified (either the email itself or its domain)
     */
    public function check_ses_sender(WP_REST_Request $request): WP_REST_Response {
        $email = $request->get_param('email');

        if (empty($email)) {
            return new WP_REST_Response([
                'verified' => false,
                'error' => __('Email address is required', 'outreach'),
            ], 400);
        }

        // Check if SES is configured
        $manager = new \WPOutreach\Mailer\SES\SESManager();
        if (!$manager->isConfigured()) {
            return new WP_REST_Response([
                'verified' => false,
                'configured' => false,
                'message' => __('SES is not configured', 'outreach'),
            ]);
        }

        // Get all verified identities
        $identities = $manager->getAllIdentitiesWithStatus();

        // Extract domain from email
        $emailParts = explode('@', $email);
        $domain = $emailParts[1] ?? '';

        // Check if domain is verified
        $domainVerified = false;
        foreach ($identities['domains'] ?? [] as $domainIdentity) {
            if (strtolower($domainIdentity['identity']) === strtolower($domain) &&
                $domainIdentity['status'] === 'Success') {
                $domainVerified = true;
                break;
            }
        }

        if ($domainVerified) {
            return new WP_REST_Response([
                'verified' => true,
                'verified_by' => 'domain',
                'domain' => $domain,
            ]);
        }

        // Check if email is verified
        $emailVerified = false;
        foreach ($identities['emails'] ?? [] as $emailIdentity) {
            if (strtolower($emailIdentity['identity']) === strtolower($email) &&
                $emailIdentity['status'] === 'Success') {
                $emailVerified = true;
                break;
            }
        }

        if ($emailVerified) {
            return new WP_REST_Response([
                'verified' => true,
                'verified_by' => 'email',
                'email' => $email,
            ]);
        }

        // Not verified
        return new WP_REST_Response([
            'verified' => false,
            'domain' => $domain,
            'email' => $email,
            'message' => sprintf(
                __('Neither "%s" nor the domain "%s" is verified in SES.', 'outreach'),
                $email,
                $domain
            ),
        ]);
    }

    /**
     * Verify email identity
     */
    public function verify_ses_email(WP_REST_Request $request): WP_REST_Response {
        $email = sanitize_email($request->get_param('email'));

        if (empty($email)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Email address is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->verifyEmail($email);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Verify domain identity
     */
    public function verify_ses_domain(WP_REST_Request $request): WP_REST_Response {
        $domain = sanitize_text_field($request->get_param('domain'));

        if (empty($domain)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Domain is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->verifyDomain($domain);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Refresh identity verification status
     */
    public function refresh_ses_identity(WP_REST_Request $request): WP_REST_Response {
        $identity = urldecode($request->get_param('identity'));

        if (empty($identity)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Identity is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->refreshIdentityStatus($identity);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Delete SES identity
     */
    public function delete_ses_identity(WP_REST_Request $request): WP_REST_Response {
        $identity = urldecode($request->get_param('identity'));

        if (empty($identity)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Identity is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->deleteIdentity($identity);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Get SES suppression list
     */
    public function get_ses_suppression_list(WP_REST_Request $request): WP_REST_Response {
        $reason = $request->get_param('reason') ?? '';
        $nextToken = $request->get_param('next_token') ?? '';
        $pageSize = (int) ($request->get_param('page_size') ?? 100);

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->listSuppressedDestinations($reason, $nextToken, $pageSize);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Add email to SES suppression list
     */
    public function add_ses_suppression(WP_REST_Request $request): WP_REST_Response {
        $email = sanitize_email($request->get_param('email') ?? '');
        $reason = strtoupper($request->get_param('reason') ?? 'BOUNCE');

        if (empty($email)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Email is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->putSuppressedDestination($email, $reason);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    /**
     * Remove email from SES suppression list
     */
    public function delete_ses_suppression(WP_REST_Request $request): WP_REST_Response {
        $email = urldecode($request->get_param('email'));

        if (empty($email)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => __('Email is required', 'outreach'),
            ], 400);
        }

        $manager = new \WPOutreach\Mailer\SES\SESManager();
        $result = $manager->deleteSuppressedDestination($email);

        return new WP_REST_Response($result, $result['success'] ? 200 : 400);
    }

    // =========================================================================
    // END SES MANAGEMENT HANDLERS
    // =========================================================================

    /**
     * External cron endpoint - processes ALL queue types
     *
     * IMPORTANT: This method MUST stay in sync with Bootstrap::init_cron_handlers()
     * Both process the same 4 tasks in the same order:
     *
     * 1. Scheduled Campaigns  - One-time campaigns scheduled for future delivery
     * 2. Recurring Campaigns  - Campaigns that repeat on a schedule (daily/weekly/monthly)
     * 3. Automation Queue     - Automation workflow steps (wp_outreach_automation_queue table)
     * 4. Email Queue          - Pending emails waiting to be sent (wp_outreach_queue table)
     *
     * Queue Processing Flow:
     * ┌─────────────────────────────────────────────────────────────────────┐
     * │ TRIGGERS (create queue entries)                                      │
     * │ ├── Subscriber events → AutomationEngine → wp_outreach_automation_queue │
     * │ ├── Post published    → AutomationEngine → wp_outreach_automation_queue │
     * │ ├── Campaign send     → QueueManager     → wp_outreach_queue            │
     * │ └── Scheduled campaign→ This method      → wp_outreach_queue            │
     * ├─────────────────────────────────────────────────────────────────────┤
     * │ PROCESSING (this method / WP Cron)                                   │
     * │ ├── AutomationEngine::processQueue() → Executes automation steps     │
     * │ │   └── May add emails to wp_outreach_queue via send_email action    │
     * │ └── QueueWorker::processQueue()      → Sends emails via mailer       │
     * └─────────────────────────────────────────────────────────────────────┘
     *
     * @see Bootstrap::init_cron_handlers() - WP Cron version (must match this)
     * @see AutomationEngine::processQueue() - Processes automation steps
     * @see QueueWorker::processQueue() - Sends queued emails
     *
     * Add ?debug=1 to get queue stats without processing
     */
    public function run_cron(WP_REST_Request $request): WP_REST_Response {
        $provided_key = $request->get_param('key');
        $cron_settings = get_option('wp_outreach_cron', []);
        $stored_key = $cron_settings['secret'] ?? '';

        // Validate secret key
        if (empty($provided_key) || !hash_equals($stored_key, $provided_key)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Invalid or missing cron key',
            ], 403);
        }

        // Debug mode - just return queue stats without processing
        if ($request->get_param('debug')) {
            global $wpdb;
            $table = $wpdb->prefix . 'outreach_queue';

            // Get status counts
            $statusCounts = $wpdb->get_results(
                "SELECT status, COUNT(*) as count FROM {$table} GROUP BY status",
                OBJECT_K
            );

            // Get sample pending items
            $pendingItems = $wpdb->get_results(
                "SELECT id, campaign_id, to_email, status, attempts, scheduled_at, error
                 FROM {$table}
                 WHERE status IN ('pending', 'failed')
                 ORDER BY scheduled_at ASC
                 LIMIT 5"
            );

            return new WP_REST_Response([
                'success' => true,
                'debug' => true,
                'queue_stats' => [
                    'pending' => (int) ($statusCounts['pending']->count ?? 0),
                    'processing' => (int) ($statusCounts['processing']->count ?? 0),
                    'sent' => (int) ($statusCounts['sent']->count ?? 0),
                    'failed' => (int) ($statusCounts['failed']->count ?? 0),
                ],
                'paused' => \WPOutreach\Mailer\Queue\QueueWorker::isPaused(),
                'sample_pending' => $pendingItems,
                'current_time' => current_time('mysql'),
            ]);
        }

        $results = [
            'scheduled_campaigns' => 0,
            'recurring_campaigns' => 0,
            'automations_processed' => 0,
            'emails_sent' => 0,
            'emails_failed' => 0,
            'timestamp' => current_time('mysql'),
        ];

        // =====================================================================
        // CRON PROCESSING - 4 STEPS (must match Bootstrap::init_cron_handlers)
        // =====================================================================

        // STEP 1: Scheduled Campaigns
        // - Finds campaigns with type='scheduled', status='scheduled', scheduled_at <= now
        // - Queues emails to wp_outreach_queue for each subscriber
        // - Updates campaign status to 'sending'
        $results['scheduled_campaigns'] = $this->process_scheduled_campaigns();

        // STEP 2: Recurring Campaigns
        // - Finds campaigns with type='recurring', status='scheduled', next_send_at <= now
        // - Queues emails to wp_outreach_queue for each subscriber
        // - Calculates and sets next_send_at based on frequency (daily/weekly/monthly)
        $results['recurring_campaigns'] = $this->process_recurring_campaigns();

        // STEP 3: Automation Queue (CRITICAL - don't forget this!)
        // - Processes wp_outreach_automation_queue table
        // - Executes automation steps (send_email, wait, add_tag, webhook, etc.)
        // - Triggered by: subscriber_created, tag_added, post_published, post_updated, etc.
        // - send_email action adds entries to wp_outreach_queue (processed in step 4)
        $results['automations_processed'] = \WPOutreach\Automation\AutomationEngine::processQueue();

        // STEP 4: Email Queue
        // - Processes wp_outreach_queue table (entries from campaigns + automations)
        // - Sends emails via configured mailer (WP Mail, SMTP, or SES)
        // - Updates status to 'sent' or 'failed', logs results
        if (!\WPOutreach\Mailer\Queue\QueueWorker::isPaused()) {
            $queueResult = \WPOutreach\Mailer\Queue\QueueWorker::processQueue();
            $results['emails_sent'] = $queueResult['sent'] ?? 0;
            $results['emails_failed'] = $queueResult['failed'] ?? 0;
        }

        // STEP 5: Unopened Email Triggers
        // - Checks for emails that haven't been opened after X days
        // - Fires email_not_opened trigger for each matching email
        $results['unopened_triggers'] = $this->process_unopened_email_triggers();

        return new WP_REST_Response([
            'success' => true,
            'results' => $results,
        ]);
    }

    /**
     * Process scheduled campaigns that are due
     */
    private function process_scheduled_campaigns(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $now = current_time('mysql');

        // Find scheduled campaigns that are due
        $campaigns = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table}
             WHERE type = 'scheduled'
             AND status = 'scheduled'
             AND scheduled_at <= %s",
            $now
        ));

        $processed = 0;

        foreach ($campaigns as $campaign) {
            // Get subscribers from lists
            $list_ids = json_decode($campaign->list_ids, true) ?: [];

            if (empty($list_ids)) {
                // No lists selected, mark as failed/draft
                $wpdb->update($table, ['status' => 'draft'], ['id' => $campaign->id]);
                continue;
            }

            $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

            if (empty($subscriber_ids)) {
                // No subscribers, mark as sent (nothing to send)
                $wpdb->update($table, ['status' => 'sent', 'sent_at' => $now], ['id' => $campaign->id]);
                continue;
            }

            // Update status to 'sending'
            $wpdb->update($table, ['status' => 'sending'], ['id' => $campaign->id]);

            // Queue emails
            $queue = new \WPOutreach\Mailer\Queue\QueueManager();
            $queue->queueCampaign($campaign->id, $subscriber_ids, $campaign->subject, $campaign->content);

            $processed++;
        }

        return $processed;
    }

    /**
     * Process recurring campaigns that are due
     */
    private function process_recurring_campaigns(): int {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_campaigns';
        $now = current_time('mysql');
        $now_timestamp = current_time('timestamp');

        // Find recurring campaigns that are due
        $campaigns = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table}
             WHERE type = 'recurring'
             AND status = 'scheduled'
             AND next_send_at IS NOT NULL
             AND next_send_at <= %s",
            $now
        ));

        $processed = 0;

        foreach ($campaigns as $campaign) {
            $recurring_config = json_decode($campaign->recurring_config, true);

            if (!$recurring_config) {
                continue;
            }

            // Check start_date constraint
            if (!empty($recurring_config['start_date'])) {
                $start_date = strtotime($recurring_config['start_date']);
                if ($start_date > $now_timestamp) {
                    continue; // Not started yet
                }
            }

            // Check end_date constraint
            if (!empty($recurring_config['end_date'])) {
                $end_date = strtotime($recurring_config['end_date'] . ' 23:59:59');
                if ($end_date < $now_timestamp) {
                    // Campaign has ended - mark as sent
                    $wpdb->update($table, ['status' => 'sent'], ['id' => $campaign->id]);
                    continue;
                }
            }

            // Get subscribers from lists
            $list_ids = json_decode($campaign->list_ids, true) ?: [];

            if (empty($list_ids)) {
                continue;
            }

            $subscriber_ids = $this->get_subscribers_for_lists($list_ids);

            if (empty($subscriber_ids)) {
                // No subscribers, just update next_send_at
                $next_send_at = $this->calculate_next_send_time($recurring_config);
                $wpdb->update($table, [
                    'last_sent_at' => $now,
                    'next_send_at' => $next_send_at,
                ], ['id' => $campaign->id]);
                continue;
            }

            // Update status to 'sending'
            $wpdb->update($table, ['status' => 'sending'], ['id' => $campaign->id]);

            // Queue emails
            $queue = new \WPOutreach\Mailer\Queue\QueueManager();
            $queue->queueCampaign($campaign->id, $subscriber_ids, $campaign->subject, $campaign->content);

            // Calculate next send time and update campaign
            $next_send_at = $this->calculate_next_send_time($recurring_config);

            // Check if next send is after end_date
            $campaign_ended = false;
            if (!empty($recurring_config['end_date'])) {
                $end_date = strtotime($recurring_config['end_date'] . ' 23:59:59');
                $next_timestamp = strtotime($next_send_at);
                if ($next_timestamp > $end_date) {
                    $campaign_ended = true;
                }
            }

            // Update last_sent_at and next_send_at (status will be updated when sending completes)
            $update_data = [
                'last_sent_at' => $now,
                'next_send_at' => $campaign_ended ? null : $next_send_at,
            ];

            $wpdb->update($table, $update_data, ['id' => $campaign->id]);

            $processed++;
        }

        return $processed;
    }

    /**
     * Process unopened email triggers
     *
     * Checks for emails that haven't been opened after X days and fires
     * the email_not_opened trigger for each matching email.
     *
     * @return int Number of triggers fired
     */
    private function process_unopened_email_triggers(): int {
        global $wpdb;

        $automations_table = $wpdb->prefix . 'outreach_automations';
        $logs_table = $wpdb->prefix . 'outreach_logs';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        // Find all active automations using email_not_opened trigger
        $automations = $wpdb->get_results(
            "SELECT id, trigger_config FROM {$automations_table}
             WHERE status = 'active'
             AND trigger_type = 'email_not_opened'"
        );

        if (empty($automations)) {
            return 0;
        }

        // Group automations by days threshold
        $thresholds = [];
        foreach ($automations as $automation) {
            $config = json_decode($automation->trigger_config, true) ?: [];
            $days = (int) ($config['days'] ?? 0);
            if ($days > 0) {
                if (!isset($thresholds[$days])) {
                    $thresholds[$days] = [];
                }
                $thresholds[$days][] = $automation->id;
            }
        }

        if (empty($thresholds)) {
            return 0;
        }

        $triggered = 0;

        // Process each threshold
        foreach ($thresholds as $days => $automation_ids) {
            $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

            // Find unopened emails that haven't triggered yet
            $logs = $wpdb->get_results($wpdb->prepare(
                "SELECT l.*, s.first_name, s.last_name, s.email as subscriber_email
                 FROM {$logs_table} l
                 LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
                 WHERE l.opens = 0
                 AND l.sent_at <= %s
                 AND l.not_opened_triggered IS NULL
                 AND l.subscriber_id IS NOT NULL
                 LIMIT 100",
                $cutoff_date
            ));

            foreach ($logs as $log) {
                // Calculate exact days unopened
                $sent_timestamp = strtotime($log->sent_at);
                $days_unopened = floor((time() - $sent_timestamp) / 86400);

                $context = [
                    'campaign_id'   => $log->campaign_id,
                    'automation_id' => $log->automation_id,
                    'email'         => $log->email,
                    'subject'       => $log->subject ?? '',
                    'first_name'    => $log->first_name ?? '',
                    'last_name'     => $log->last_name ?? '',
                    'sent_at'       => $log->sent_at,
                    'days_unopened' => $days_unopened,
                ];

                // Fire the trigger
                $result = \WPOutreach\Automation\TriggerRegistry::fire('email_not_opened', (int) $log->subscriber_id, $context);

                if ($result > 0) {
                    // Mark as triggered
                    $wpdb->update(
                        $logs_table,
                        ['not_opened_triggered' => current_time('mysql')],
                        ['id' => $log->id]
                    );
                    $triggered++;
                }
            }
        }

        return $triggered;
    }

    /**
     * Register queue routes
     */
    private function register_queue_routes(): void {
        register_rest_route($this->namespace, '/queue/stats', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_queue_stats'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/queue/process', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'process_queue'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/queue/pause', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'pause_queue'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/queue/resume', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'resume_queue'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/mailers', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_available_mailers'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        register_rest_route($this->namespace, '/merge-tags', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_merge_tags'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Get queue statistics
     */
    public function get_queue_stats(): WP_REST_Response {
        $manager = new \WPOutreach\Mailer\Queue\QueueManager();
        $stats = $manager->getStats();
        $stats['paused'] = \WPOutreach\Mailer\Queue\QueueWorker::isPaused();

        return new WP_REST_Response($stats);
    }

    /**
     * Manually trigger queue processing
     */
    public function process_queue(): WP_REST_Response {
        $worker = new \WPOutreach\Mailer\Queue\QueueWorker();
        $results = $worker->run();

        return new WP_REST_Response($results);
    }

    /**
     * Pause queue processing
     */
    public function pause_queue(): WP_REST_Response {
        \WPOutreach\Mailer\Queue\QueueWorker::pause();
        return new WP_REST_Response(['paused' => true]);
    }

    /**
     * Resume queue processing
     */
    public function resume_queue(): WP_REST_Response {
        \WPOutreach\Mailer\Queue\QueueWorker::resume();
        return new WP_REST_Response(['paused' => false]);
    }

    /**
     * Get available mailers
     */
    public function get_available_mailers(): WP_REST_Response {
        $mailers = \WPOutreach\Mailer\MailerFactory::getAvailableMailers();
        $current = get_option('wp_outreach_mailer', []);

        return new WP_REST_Response([
            'mailers' => $mailers,
            'current' => $current['driver'] ?? 'wp_mail',
        ]);
    }

    /**
     * Get available merge tags for email personalization
     */
    public function get_merge_tags(): WP_REST_Response {
        $tags = \WPOutreach\Mailer\Template\VariableParser::getAvailableVariables();

        return new WP_REST_Response($tags);
    }

    // =========================================================================
    // TRACKING ENDPOINTS (PUBLIC)
    // =========================================================================

    /**
     * Handle unsubscribe request (GET)
     *
     * Displays the subscription management page where users can choose
     * which subscriptions to keep or remove.
     *
     * @param WP_REST_Request $request
     * @return void
     */
    public function handle_unsubscribe(WP_REST_Request $request): void {
        $token = sanitize_text_field($request->get_param('token') ?? '');

        if (empty($token)) {
            $html = \WPOutreach\Tracking\UnsubscribeHandler::renderConfirmationPage([
                'success' => false,
                'message' => 'Missing unsubscribe token.',
            ]);
        } else {
            // Validate token and get subscriber ID
            $subscriberId = \WPOutreach\Tracking\UnsubscribeHandler::validateToken($token);

            if ($subscriberId === false) {
                $html = \WPOutreach\Tracking\UnsubscribeHandler::renderConfirmationPage([
                    'success' => false,
                    'message' => 'Invalid or expired unsubscribe link.',
                ]);
            } else {
                // Show the subscription management page
                $html = \WPOutreach\Tracking\UnsubscribeHandler::renderManagementPage($subscriberId, $token);
            }
        }

        // Output HTML directly (WP_REST_Response JSON-encodes the content)
        header('Content-Type: text/html; charset=utf-8');
        echo $html;
        exit;
    }

    /**
     * Handle unsubscribe form submission (POST)
     *
     * Processes the subscription management form to update user preferences.
     *
     * @param WP_REST_Request $request
     * @return void
     */
    public function handle_unsubscribe_form(WP_REST_Request $request): void {
        $token = sanitize_text_field($request->get_param('token') ?? '');
        $action = sanitize_text_field($request->get_param('action') ?? '');
        $unsubscribeAll = !empty($request->get_param('unsubscribe_all'));

        // Get arrays of lists and posts to keep
        $keepLists = $request->get_param('keep_lists');
        $keepPosts = $request->get_param('keep_posts');

        // Ensure they are arrays
        $keepLists = is_array($keepLists) ? array_map('intval', $keepLists) : [];
        $keepPosts = is_array($keepPosts) ? array_map('intval', $keepPosts) : [];

        if (empty($token)) {
            $result = [
                'success' => false,
                'message' => 'Missing unsubscribe token.',
            ];
        } else {
            $result = \WPOutreach\Tracking\UnsubscribeHandler::processSubscriptionUpdate(
                $token,
                $keepLists,
                $keepPosts,
                $unsubscribeAll
            );
        }

        // Render the confirmation page HTML
        $html = \WPOutreach\Tracking\UnsubscribeHandler::renderConfirmationPage($result);

        // Output HTML directly (WP_REST_Response JSON-encodes the content)
        header('Content-Type: text/html; charset=utf-8');
        echo $html;
        exit;
    }

    /**
     * Track email open (tracking pixel)
     *
     * Returns a 1x1 transparent GIF and records the open event.
     * This is a public endpoint that doesn't require authentication.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function track_open(WP_REST_Request $request): WP_REST_Response {
        $trackingId = sanitize_text_field($request->get_param('t') ?? '');

        if (!empty($trackingId)) {
            // Record the open event
            \WPOutreach\Tracking\OpenTracker::recordOpen($trackingId);
        }

        // Return a 1x1 transparent GIF
        $gif = base64_decode('R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7');

        $response = new WP_REST_Response($gif);
        $response->header('Content-Type', 'image/gif');
        $response->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
        $response->header('Pragma', 'no-cache');
        $response->header('Expires', 'Thu, 01 Jan 1970 00:00:00 GMT');

        return $response;
    }

    /**
     * Track click and redirect
     *
     * Records the click event and redirects to the target URL.
     * This is a public endpoint that doesn't require authentication.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function track_click(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $trackingId = sanitize_text_field($request->get_param('t') ?? '');
        $linkHash = sanitize_text_field($request->get_param('l') ?? '');

        // If we have tracking params, use the new tracking system
        if (!empty($trackingId) && !empty($linkHash)) {
            $url = \WPOutreach\Tracking\ClickTracker::recordClick($trackingId, $linkHash);

            if ($url) {
                // Redirect to the original URL
                $response = new WP_REST_Response(null, 302);
                $response->header('Location', $url);
                $response->header('Cache-Control', 'no-store, no-cache, must-revalidate');
                return $response;
            }

            // Link not found - redirect to home
            $response = new WP_REST_Response(null, 302);
            $response->header('Location', home_url());
            return $response;
        }

        // Legacy support: direct URL parameter
        $url = $request->get_param('url');

        if (empty($url)) {
            return new WP_Error('missing_params', 'Missing tracking parameters', ['status' => 400]);
        }

        // Decode the URL if it's base64 encoded
        if (preg_match('/^[a-zA-Z0-9+\/=]+$/', $url) && strlen($url) > 20) {
            $decoded = base64_decode($url, true);
            if ($decoded !== false && filter_var($decoded, FILTER_VALIDATE_URL)) {
                $url = $decoded;
            }
        }

        // Validate URL format
        $url = filter_var($url, FILTER_VALIDATE_URL);
        if (!$url) {
            return new WP_Error('invalid_url', 'Invalid redirect URL', ['status' => 400]);
        }

        // Redirect to the target URL
        $response = new WP_REST_Response(null, 302);
        $response->header('Location', $url);
        $response->header('Cache-Control', 'no-store, no-cache, must-revalidate');

        return $response;
    }

    /**
     * Handle public subscription request
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function handle_subscribe(WP_REST_Request $request): WP_REST_Response {
        $data = [
            'email' => $request->get_param('email'),
            'first_name' => $request->get_param('first_name'),
            'last_name' => $request->get_param('last_name'),
            'list_id' => $request->get_param('list_id'),
            'website' => $request->get_param('website'), // Honeypot
        ];

        $result = \WPOutreach\Forms\SubscriptionForm::processSubscription($data);

        return new WP_REST_Response($result);
    }

    /**
     * Handle email confirmation request
     *
     * Validates the confirmation token and activates the subscriber.
     * Returns an HTML confirmation page.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $token Required. Base64-encoded confirmation token.
     * }
     *
     * @return WP_REST_Response HTML confirmation page
     */
    public function handle_confirm(WP_REST_Request $request): WP_REST_Response {
        $token = $request->get_param('token');

        if (empty($token)) {
            $result = [
                'success' => false,
                'message' => __('Missing confirmation token.', 'outreach'),
            ];
        } else {
            $result = \WPOutreach\Forms\DoubleOptIn::processConfirmation($token);
        }

        // Return HTML page
        $html = \WPOutreach\Forms\DoubleOptIn::renderConfirmationPage($result);

        $response = new WP_REST_Response($html);
        $response->header('Content-Type', 'text/html; charset=utf-8');

        return $response;
    }

    /**
     * Resend confirmation email
     *
     * Allows users to request a new confirmation email if the original
     * expired or was lost. Generates a new token for security.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $email Required. Subscriber email address.
     * }
     *
     * @return WP_REST_Response {
     *     @type bool   $success Always true to prevent email enumeration.
     *     @type string $message Generic success message.
     * }
     */
    public function resend_confirmation(WP_REST_Request $request): WP_REST_Response {
        $email = sanitize_email($request->get_param('email'));

        if (empty($email) || !is_email($email)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Please enter a valid email address.', 'outreach'),
            ]);
        }

        $result = \WPOutreach\Forms\DoubleOptIn::resendConfirmation($email);

        return new WP_REST_Response($result);
    }

    // =========================================================================
    // POST SUBSCRIPTION ENDPOINTS
    // =========================================================================

    /**
     * Handle post subscription request
     *
     * Allows users to subscribe to post/page/CPT updates.
     * Creates a subscriber if needed, then creates the post subscription.
     * Includes honeypot spam protection and rate limiting.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $email            Required. Subscriber email address.
     *     @type int    $post_id          Optional. Specific post ID (0 for all of type).
     *     @type string $post_type        Required. Post type (post, page, or CPT).
     *     @type string $subscription_type Optional. 'new', 'update', or 'both'. Default: 'both'.
     *     @type string $scope            Optional. 'post' or 'type'. Default: 'post'.
     *     @type string $website          Optional. Honeypot field (should be empty).
     * }
     *
     * @return WP_REST_Response
     */
    public function handle_post_subscribe(WP_REST_Request $request): WP_REST_Response {
        $data = [
            'email' => $request->get_param('email'),
            'post_id' => $request->get_param('post_id'),
            'post_type' => $request->get_param('post_type'),
            'subscription_type' => $request->get_param('subscription_type'),
            'scope' => $request->get_param('scope'),
            'website' => $request->get_param('website'), // Honeypot
        ];

        $result = \WPOutreach\Forms\PostSubscriptionForm::processSubscription($data);

        return new WP_REST_Response($result);
    }

    /**
     * Handle post unsubscribe request
     *
     * Unsubscribes a user from post updates using a secure token.
     * Returns an HTML confirmation page.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type string $token Required. Subscription token.
     * }
     *
     * @return WP_REST_Response HTML confirmation page
     */
    public function handle_post_unsubscribe(WP_REST_Request $request): WP_REST_Response {
        $token = sanitize_text_field($request->get_param('token') ?? '');

        if (empty($token)) {
            $result = [
                'success' => false,
                'message' => __('Missing unsubscribe token.', 'outreach'),
            ];
        } else {
            $handler = new \WPOutreach\Subscriptions\PostSubscriptionHandler();
            $success = $handler->unsubscribeByToken($token);

            $result = [
                'success' => $success,
                'message' => $success
                    ? __('You have been unsubscribed from content updates.', 'outreach')
                    : __('Invalid or expired unsubscribe link.', 'outreach'),
            ];
        }

        // Render HTML confirmation page
        $html = $this->render_post_unsubscribe_page($result);

        $response = new WP_REST_Response($html);
        $response->header('Content-Type', 'text/html; charset=utf-8');

        return $response;
    }

    /**
     * Render post unsubscribe confirmation page
     *
     * @param array $result Result array with success and message
     * @return string HTML content
     */
    private function render_post_unsubscribe_page(array $result): string {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $success = $result['success'] ?? false;
        $message = $result['message'] ?? '';

        $icon = $success
            ? '<svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#10b981" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>'
            : '<svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#ef4444" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>';

        $title = $success ? __('Unsubscribed', 'outreach') : __('Error', 'outreach');

        ob_start();
        include WP_OUTREACH_PATH . 'templates/pages/unsubscribe.php';
        return ob_get_clean();
    }

    /**
     * Get available post types for automation trigger config
     *
     * Returns public post types with labels for the dropdown.
     *
     * @since 1.0.0
     *
     * @return WP_REST_Response Array of post types with value/label
     */
    public function get_post_types(): WP_REST_Response {
        $post_types = \WPOutreach\Subscriptions\PostSubscriptionHandler::getAvailablePostTypes();

        return new WP_REST_Response($post_types);
    }

    /**
     * Get available user roles for trigger configuration
     *
     * Returns roles in the format expected by the TriggerConfigModal dropdown.
     *
     * @since 2.6.5
     *
     * @return WP_REST_Response Array of roles with value/label
     */
    public function get_user_roles(): WP_REST_Response {
        global $wp_roles;

        if (!isset($wp_roles)) {
            $wp_roles = new \WP_Roles();
        }

        $roles = [];
        foreach ($wp_roles->role_names as $role_slug => $role_name) {
            $roles[] = [
                'value' => $role_slug,
                'label' => translate_user_role($role_name),
            ];
        }

        return new WP_REST_Response($roles);
    }

    /**
     * Register post subscription routes (admin management)
     */
    private function register_post_subscription_routes(): void {
        // List post subscriptions with pagination and filtering
        register_rest_route($this->namespace, '/post-subscriptions', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_post_subscriptions'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => array_merge($this->get_collection_params(), [
                    'post_type' => [
                        'type' => 'string',
                        'default' => '',
                    ],
                    'subscription_type' => [
                        'type' => 'string',
                        'default' => '',
                    ],
                    'search' => [
                        'type' => 'string',
                        'default' => '',
                    ],
                ]),
            ],
            // Create post subscription from admin
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'create_post_subscription'],
                'permission_callback' => [$this, 'check_admin_permission'],
                'args' => [
                    'subscriber_id' => [
                        'required' => true,
                        'type' => 'integer',
                        'sanitize_callback' => 'absint',
                    ],
                    'post_id' => [
                        'type' => 'integer',
                        'default' => 0,
                        'sanitize_callback' => 'absint',
                    ],
                    'post_type' => [
                        'required' => true,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_key',
                    ],
                    'subscription_type' => [
                        'required' => true,
                        'type' => 'string',
                        'enum' => ['new', 'update', 'both'],
                        'sanitize_callback' => 'sanitize_key',
                    ],
                ],
            ],
            // Bulk delete
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'bulk_delete_post_subscriptions'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Single post subscription operations
        register_rest_route($this->namespace, '/post-subscriptions/(?P<id>\d+)', [
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'delete_post_subscription'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);

        // Get post subscriptions for a specific subscriber
        register_rest_route($this->namespace, '/subscribers/(?P<id>\d+)/post-subscriptions', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_subscriber_post_subscriptions'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Get post subscriptions with pagination and filtering
     *
     * Returns a paginated list of post subscriptions with subscriber and post details.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int    $page              Page number (default: 1)
     *     @type int    $per_page          Items per page (default: 20)
     *     @type string $post_type         Filter by post type
     *     @type string $subscription_type Filter by subscription type (new, update, both)
     *     @type string $search            Search by email or post title
     * }
     *
     * @return WP_REST_Response {
     *     @type array $items       Array of post subscriptions
     *     @type int   $total       Total items count
     *     @type int   $page        Current page
     *     @type int   $per_page    Items per page
     *     @type int   $total_pages Total pages
     * }
     */
    public function get_post_subscriptions(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_post_subscriptions';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        $page = (int) $request->get_param('page') ?: 1;
        $per_page = (int) $request->get_param('per_page') ?: 20;
        $offset = ($page - 1) * $per_page;
        $post_type = sanitize_key($request->get_param('post_type'));
        $subscription_type = sanitize_key($request->get_param('subscription_type'));
        $search = sanitize_text_field($request->get_param('search'));

        // Build WHERE clause
        // Only show active post subscriptions for active subscribers
        $where = ['ps.status = 1', "s.status = 'active'"];
        $params = [];

        if (!empty($post_type)) {
            $where[] = 'ps.post_type = %s';
            $params[] = $post_type;
        }

        if (!empty($subscription_type)) {
            $where[] = 'ps.subscription_type = %s';
            $params[] = $subscription_type;
        }

        if (!empty($search)) {
            $where[] = '(s.email LIKE %s OR p.post_title LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $search_term;
            $params[] = $search_term;
        }

        $where_clause = implode(' AND ', $where);

        // Get total count
        $count_sql = "SELECT COUNT(*) FROM {$table} ps
                      LEFT JOIN {$subscribers_table} s ON ps.subscriber_id = s.id
                      LEFT JOIN {$wpdb->posts} p ON ps.post_id = p.ID
                      WHERE {$where_clause}";

        if (!empty($params)) {
            $count_sql = $wpdb->prepare($count_sql, ...$params);
        }
        $total = (int) $wpdb->get_var($count_sql);

        // Get items with pagination
        $sql = "SELECT ps.*, s.email, s.first_name, s.last_name, p.post_title, p.post_type as wp_post_type
                FROM {$table} ps
                LEFT JOIN {$subscribers_table} s ON ps.subscriber_id = s.id
                LEFT JOIN {$wpdb->posts} p ON ps.post_id = p.ID
                WHERE {$where_clause}
                ORDER BY ps.created_at DESC
                LIMIT %d OFFSET %d";

        $query_params = array_merge($params, [$per_page, $offset]);
        $items = $wpdb->get_results($wpdb->prepare($sql, ...$query_params));

        // Format items
        $formatted_items = array_map(function($item) {
            // Get post type label
            $post_type_obj = get_post_type_object($item->post_type);
            $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $item->post_type;

            return [
                'id' => (int) $item->id,
                'subscriber_id' => (int) $item->subscriber_id,
                'email' => $item->email,
                'subscriber_name' => trim(($item->first_name ?? '') . ' ' . ($item->last_name ?? '')) ?: null,
                'post_id' => (int) $item->post_id,
                'post_title' => $item->post_title ?: ($item->post_id === 0 ? 'All ' . $post_type_label . 's' : 'Deleted Post'),
                'post_type' => $item->post_type,
                'post_type_label' => $post_type_label,
                'subscription_type' => $item->subscription_type,
                'created_at' => DateHelper::toIso8601($item->created_at),
            ];
        }, $items);

        return new WP_REST_Response([
            'items' => $formatted_items,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page),
        ]);
    }

    /**
     * Delete a single post subscription
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Post subscription ID from URL parameter
     * }
     *
     * @return WP_REST_Response|WP_Error Success message or error
     */
    public function delete_post_subscription(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';
        $id = (int) $request->get_param('id');

        // Check if exists
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE id = %d",
            $id
        ));

        if (!$exists) {
            return new WP_Error('not_found', 'Post subscription not found', ['status' => 404]);
        }

        // Delete the subscription
        $wpdb->delete($table, ['id' => $id], ['%d']);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Post subscription deleted',
        ]);
    }

    /**
     * Bulk delete post subscriptions
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type array $ids Array of post subscription IDs to delete
     * }
     *
     * @return WP_REST_Response|WP_Error Success with count or error
     */
    public function bulk_delete_post_subscriptions(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;
        $table = $wpdb->prefix . 'outreach_post_subscriptions';
        $ids = $request->get_param('ids');

        if (!is_array($ids) || empty($ids)) {
            return new WP_Error('invalid_ids', 'Please provide an array of IDs to delete', ['status' => 400]);
        }

        // Sanitize IDs
        $ids = array_map('intval', $ids);
        $ids = array_filter($ids); // Remove zeros

        if (empty($ids)) {
            return new WP_Error('invalid_ids', 'No valid IDs provided', ['status' => 400]);
        }

        // Delete the subscriptions
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table} WHERE id IN ({$placeholders})",
            ...$ids
        ));

        return new WP_REST_Response([
            'success' => true,
            'deleted' => $deleted,
            'message' => sprintf('Deleted %d post subscription(s)', $deleted),
        ]);
    }

    /**
     * Get post subscriptions for a specific subscriber
     *
     * Returns all post subscriptions for a subscriber with post details.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int $id Subscriber ID from URL parameter
     * }
     *
     * @return WP_REST_Response Array of post subscriptions
     */
    public function get_subscriber_post_subscriptions(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_post_subscriptions';
        $subscriber_id = (int) $request->get_param('id');

        $subscriptions = $wpdb->get_results($wpdb->prepare(
            "SELECT ps.*, p.post_title, p.post_type as wp_post_type
             FROM {$table} ps
             LEFT JOIN {$wpdb->posts} p ON ps.post_id = p.ID
             WHERE ps.subscriber_id = %d AND ps.status = 1
             ORDER BY ps.created_at DESC",
            $subscriber_id
        ));

        $formatted = array_map(function($item) {
            $post_type_obj = get_post_type_object($item->post_type);
            $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : $item->post_type;
            $post_id = (int) $item->post_id;

            return [
                'id' => (int) $item->id,
                'post_id' => $post_id,
                'post_title' => $item->post_title ?: '',
                'post_url' => $post_id > 0 ? get_permalink($post_id) : '',
                'post_type' => $item->post_type,
                'post_type_label' => $post_type_label,
                'subscription_type' => $item->subscription_type,
                'created_at' => DateHelper::toIso8601($item->created_at),
            ];
        }, $subscriptions);

        return new WP_REST_Response($formatted);
    }

    /**
     * Create a post subscription from admin
     *
     * Allows admins to manually subscribe users to post updates for testing
     * or manual subscription management.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int    $subscriber_id     Required. Subscriber ID to subscribe.
     *     @type int    $post_id           Optional. Specific post ID (0 = all posts of type).
     *     @type string $post_type         Required. Post type to subscribe to.
     *     @type string $subscription_type Required. Type: 'new', 'update', or 'both'.
     * }
     *
     * @return WP_REST_Response|WP_Error Success response or error.
     */
    public function create_post_subscription(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_post_subscriptions';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';

        $subscriber_id = (int) $request->get_param('subscriber_id');
        $post_id = (int) $request->get_param('post_id');
        $post_type = sanitize_key($request->get_param('post_type'));
        $subscription_type = sanitize_key($request->get_param('subscription_type'));

        // Validate subscriber exists
        $subscriber = $wpdb->get_row($wpdb->prepare(
            "SELECT id, email FROM {$subscribers_table} WHERE id = %d",
            $subscriber_id
        ));

        if (!$subscriber) {
            return new WP_Error('invalid_subscriber', 'Subscriber not found', ['status' => 404]);
        }

        // Validate post exists if post_id provided
        if ($post_id > 0) {
            $post = get_post($post_id);
            if (!$post) {
                return new WP_Error('invalid_post', 'Post not found', ['status' => 404]);
            }
            // Use the post's actual type
            $post_type = $post->post_type;
        }

        // Check for existing subscription
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table}
             WHERE subscriber_id = %d AND post_id = %d AND post_type = %s AND status = 1",
            $subscriber_id,
            $post_id,
            $post_type
        ));

        if ($existing) {
            return new WP_Error('duplicate', 'This subscription already exists', ['status' => 400]);
        }

        // Insert subscription
        $result = $wpdb->insert($table, [
            'subscriber_id' => $subscriber_id,
            'post_id' => $post_id,
            'post_type' => $post_type,
            'subscription_type' => $subscription_type,
            'status' => 1,
            'created_at' => current_time('mysql'),
        ], ['%d', '%d', '%s', '%s', '%d', '%s']);

        if (!$result) {
            return new WP_Error('db_error', 'Failed to create subscription', ['status' => 500]);
        }

        $subscription_id = $wpdb->insert_id;

        // Fire hook for automation trigger
        do_action('wp_outreach_post_subscription_created', $subscriber_id, [
            'subscription_id' => $subscription_id,
            'post_id' => $post_id,
            'post_type' => $post_type,
            'subscription_type' => $subscription_type,
        ]);

        return new WP_REST_Response([
            'success' => true,
            'id' => $subscription_id,
            'message' => 'Subscription created successfully',
        ], 201);
    }

    // =========================================================================
    // EMAIL LOGS HANDLERS
    // =========================================================================

    /**
     * Get email logs with filtering and pagination
     *
     * Returns logs for WordPress emails, automation emails, and campaign emails
     * with filtering by type, status, and search.
     *
     * @since 1.0.0
     *
     * @param WP_REST_Request $request {
     *     @type int    $page     Page number (default: 1)
     *     @type int    $per_page Items per page (default: 20)
     *     @type string $type     Filter by type: wordpress, automation, campaign
     *     @type string $status   Filter by status: sent, opened, clicked, bounced, complained
     *     @type string $search   Search by email or subject
     * }
     *
     * @return WP_REST_Response Paginated logs with metadata
     */
    public function get_logs(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_logs';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $automations_table = $wpdb->prefix . 'outreach_automations';

        $page = max(1, (int) $request->get_param('page'));
        $per_page = min(100, max(1, (int) $request->get_param('per_page')));
        $offset = ($page - 1) * $per_page;

        $type = sanitize_text_field($request->get_param('type'));
        $status = sanitize_text_field($request->get_param('status'));
        $search = sanitize_text_field($request->get_param('search'));

        // Build WHERE clause
        // By default, exclude campaign emails - they are shown on the campaign stats page
        $where = ["(l.type IS NULL OR l.type != 'campaign')"];
        $params = [];

        if (!empty($type)) {
            // When filtering by specific type, override the default exclusion
            $where = ['l.type = %s'];
            $params[] = $type;
        }

        if (!empty($status)) {
            $where[] = 'l.status = %s';
            $params[] = $status;
        }

        if (!empty($search)) {
            $where[] = '(l.email LIKE %s OR l.subject LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $search_term;
            $params[] = $search_term;
        }

        $where_clause = implode(' AND ', $where);

        // Get total count
        $count_sql = "SELECT COUNT(*) FROM {$table} l WHERE {$where_clause}";
        if (!empty($params)) {
            $count_sql = $wpdb->prepare($count_sql, ...$params);
        }
        $total = (int) $wpdb->get_var($count_sql);

        // Get items with pagination
        $sql = "SELECT l.*,
                       s.first_name, s.last_name,
                       c.name as campaign_name,
                       a.name as automation_name
                FROM {$table} l
                LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
                LEFT JOIN {$campaigns_table} c ON l.campaign_id = c.id
                LEFT JOIN {$automations_table} a ON l.automation_id = a.id
                WHERE {$where_clause}
                ORDER BY l.sent_at DESC
                LIMIT %d OFFSET %d";

        $query_params = array_merge($params, [$per_page, $offset]);
        $items = $wpdb->get_results($wpdb->prepare($sql, ...$query_params));

        // Format items
        $formatted_items = array_map(function($item) {
            $type_label = match($item->type ?? 'campaign') {
                'wordpress' => 'WordPress',
                'automation' => 'Automation',
                'campaign' => 'Campaign',
                default => ucfirst($item->type ?? 'Unknown'),
            };

            return [
                'id' => (int) $item->id,
                'type' => $item->type ?? 'campaign',
                'type_label' => $type_label,
                'email' => $item->email,
                'subscriber_id' => $item->subscriber_id ? (int) $item->subscriber_id : null,
                'subscriber_name' => trim(($item->first_name ?? '') . ' ' . ($item->last_name ?? '')) ?: null,
                'subject' => $item->subject,
                'status' => $item->status,
                'error' => $item->error ?? null,
                'opens' => (int) $item->opens,
                'clicks' => (int) $item->clicks,
                'campaign_id' => $item->campaign_id ? (int) $item->campaign_id : null,
                'campaign_name' => $item->campaign_name,
                'automation_id' => $item->automation_id ? (int) $item->automation_id : null,
                'automation_name' => $item->automation_name,
                'first_opened_at' => DateHelper::toIso8601($item->first_opened_at),
                'last_opened_at' => DateHelper::toIso8601($item->last_opened_at),
                'sent_at' => DateHelper::toIso8601($item->sent_at),
            ];
        }, $items);

        return new WP_REST_Response([
            'items' => $formatted_items,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => (int) ceil($total / $per_page),
        ]);
    }

    /**
     * Get a single log entry
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_log(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_logs';
        $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
        $campaigns_table = $wpdb->prefix . 'outreach_campaigns';
        $automations_table = $wpdb->prefix . 'outreach_automations';

        $id = (int) $request->get_param('id');

        $log = $wpdb->get_row($wpdb->prepare(
            "SELECT l.*,
                    s.first_name, s.last_name, s.email as subscriber_email,
                    c.name as campaign_name,
                    a.name as automation_name
             FROM {$table} l
             LEFT JOIN {$subscribers_table} s ON l.subscriber_id = s.id
             LEFT JOIN {$campaigns_table} c ON l.campaign_id = c.id
             LEFT JOIN {$automations_table} a ON l.automation_id = a.id
             WHERE l.id = %d",
            $id
        ));

        if (!$log) {
            return new WP_Error('not_found', 'Log entry not found', ['status' => 404]);
        }

        // Parse clicked_links JSON
        $clicked_links = [];
        if (!empty($log->clicked_links)) {
            $clicked_links = json_decode($log->clicked_links, true) ?: [];
        }

        return new WP_REST_Response([
            'id' => (int) $log->id,
            'type' => $log->type ?? 'campaign',
            'email' => $log->email,
            'subscriber_id' => $log->subscriber_id ? (int) $log->subscriber_id : null,
            'subscriber_name' => trim(($log->first_name ?? '') . ' ' . ($log->last_name ?? '')) ?: null,
            'subject' => $log->subject,
            'status' => $log->status,
            'opens' => (int) $log->opens,
            'clicks' => (int) $log->clicks,
            'clicked_links' => $clicked_links,
            'campaign_id' => $log->campaign_id ? (int) $log->campaign_id : null,
            'campaign_name' => $log->campaign_name,
            'automation_id' => $log->automation_id ? (int) $log->automation_id : null,
            'automation_name' => $log->automation_name,
            'first_opened_at' => DateHelper::toIso8601($log->first_opened_at),
            'last_opened_at' => DateHelper::toIso8601($log->last_opened_at),
            'sent_at' => DateHelper::toIso8601($log->sent_at),
        ]);
    }

    /**
     * Delete a log entry
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function delete_log(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_logs';
        $id = (int) $request->get_param('id');

        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE id = %d",
            $id
        ));

        if (!$exists) {
            return new WP_Error('not_found', 'Log entry not found', ['status' => 404]);
        }

        $wpdb->delete($table, ['id' => $id], ['%d']);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Log entry deleted',
        ]);
    }

    /**
     * Get logs statistics for dashboard
     *
     * Returns aggregated stats by type and status.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    public function get_logs_stats(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        $table = $wpdb->prefix . 'outreach_logs';

        // Exclude campaign emails - they are shown on the campaign stats page
        $exclude_campaign = "(type IS NULL OR type != 'campaign')";

        // Get counts by type (excluding campaign)
        $type_counts = $wpdb->get_results(
            "SELECT COALESCE(type, 'unknown') as type, COUNT(*) as count
             FROM {$table}
             WHERE {$exclude_campaign}
             GROUP BY type",
            OBJECT_K
        );

        // Get counts by status (excluding campaign)
        $status_counts = $wpdb->get_results(
            "SELECT status, COUNT(*) as count
             FROM {$table}
             WHERE {$exclude_campaign}
             GROUP BY status",
            OBJECT_K
        );

        // Get today's stats (excluding campaign)
        $today = current_time('Y-m-d');
        $today_stats = $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total,
                SUM(CASE WHEN status = 'sent' THEN 1 ELSE 0 END) as sent,
                SUM(CASE WHEN opens > 0 THEN 1 ELSE 0 END) as opened,
                SUM(CASE WHEN clicks > 0 THEN 1 ELSE 0 END) as clicked
             FROM {$table}
             WHERE DATE(sent_at) = %s AND {$exclude_campaign}",
            $today
        ));

        // Get last 7 days trend (excluding campaign)
        $seven_days_ago = date('Y-m-d', strtotime('-7 days'));
        $daily_stats = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(sent_at) as date, COUNT(*) as count
             FROM {$table}
             WHERE sent_at >= %s AND {$exclude_campaign}
             GROUP BY DATE(sent_at)
             ORDER BY date ASC",
            $seven_days_ago
        ));

        return new WP_REST_Response([
            'by_type' => [
                'wordpress' => (int) ($type_counts['wordpress']->count ?? 0),
                'automation' => (int) ($type_counts['automation']->count ?? 0),
            ],
            'by_status' => [
                'sent' => (int) ($status_counts['sent']->count ?? 0),
                'opened' => (int) ($status_counts['opened']->count ?? 0),
                'clicked' => (int) ($status_counts['clicked']->count ?? 0),
                'bounced' => (int) ($status_counts['bounced']->count ?? 0),
                'complained' => (int) ($status_counts['complained']->count ?? 0),
            ],
            'today' => [
                'total' => (int) ($today_stats->total ?? 0),
                'sent' => (int) ($today_stats->sent ?? 0),
                'opened' => (int) ($today_stats->opened ?? 0),
                'clicked' => (int) ($today_stats->clicked ?? 0),
            ],
            'daily_trend' => $daily_stats,
        ]);
    }

    /**
     * Get license info with feature availability
     */
    public function get_license(WP_REST_Request $request): WP_REST_Response {
        $license = \WPOutreach\Admin\License::get_info();

        // Add feature manager status
        $license['features'] = FeatureManager::getStatus();

        return new WP_REST_Response($license);
    }

    /**
     * Validate and activate license
     */
    public function validate_license(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $license_key = $request->get_param('license_key');

        if (empty($license_key)) {
            return new WP_Error(
                'missing_license_key',
                'Please provide a license key',
                ['status' => 400]
            );
        }

        $result = \WPOutreach\Admin\License::validate($license_key);

        if (!$result['success']) {
            return new WP_Error(
                'license_validation_failed',
                $result['message'],
                ['status' => 400]
            );
        }

        return new WP_REST_Response($result);
    }

    /**
     * Deactivate license
     */
    public function deactivate_license(WP_REST_Request $request): WP_REST_Response {
        $result = \WPOutreach\Admin\License::deactivate();
        return new WP_REST_Response($result);
    }

    // =========================================================================
    // CSV Import/Export Methods
    // =========================================================================

    /**
     * Export subscribers to CSV
     *
     * @param WP_REST_Request $request {
     *     @type int    $list_id Optional. Filter by list ID.
     *     @type string $status  Optional. Filter by status (active, pending, unsubscribed, bounced).
     * }
     *
     * @return WP_REST_Response|WP_Error CSV file download
     */
    public function export_subscribers(WP_REST_Request $request): WP_REST_Response|WP_Error {
        // Check if import/export feature is available
        if (!FeatureManager::can('import_export')) {
            return new WP_Error(
                'feature_restricted',
                FeatureManager::getRestrictionMessage('import_export'),
                [
                    'status' => 403,
                    'upgrade_url' => FeatureManager::getUpgradeUrl(),
                ]
            );
        }

        global $wpdb;

        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';
        $lists_table = $wpdb->prefix . 'outreach_lists';
        $tag_pivot_table = $wpdb->prefix . 'outreach_subscriber_tag';
        $tags_table = $wpdb->prefix . 'outreach_tags';

        $list_id = (int) $request->get_param('list_id');
        $status = sanitize_text_field($request->get_param('status') ?? '');

        // Build query
        $where = [];
        $params = [];

        if ($status) {
            $where[] = 's.status = %s';
            $params[] = $status;
        }

        if ($list_id > 0) {
            $where[] = 's.id IN (SELECT subscriber_id FROM ' . $pivot_table . ' WHERE list_id = %d)';
            $params[] = $list_id;
        }

        $where_clause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

        $query = "SELECT s.* FROM {$table} s {$where_clause} ORDER BY s.created_at DESC";

        if (!empty($params)) {
            $query = $wpdb->prepare($query, ...$params);
        }

        $subscribers = $wpdb->get_results($query, ARRAY_A);

        if (empty($subscribers)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('No subscribers found to export', 'outreach'),
            ], 404);
        }

        // Get lists and tags for each subscriber
        foreach ($subscribers as &$subscriber) {
            // Get lists
            $lists = $wpdb->get_col($wpdb->prepare(
                "SELECT l.name FROM {$pivot_table} sl
                 INNER JOIN {$lists_table} l ON sl.list_id = l.id
                 WHERE sl.subscriber_id = %d",
                $subscriber['id']
            ));
            $subscriber['lists'] = implode(', ', $lists);

            // Get tags
            $tags = $wpdb->get_col($wpdb->prepare(
                "SELECT t.name FROM {$tag_pivot_table} st
                 INNER JOIN {$tags_table} t ON st.tag_id = t.id
                 WHERE st.subscriber_id = %d",
                $subscriber['id']
            ));
            $subscriber['tags'] = implode(', ', $tags);

            // Remove internal fields
            unset($subscriber['token']);
        }

        // Build CSV
        $csv_headers = ['email', 'first_name', 'last_name', 'status', 'source', 'lists', 'tags', 'created_at'];
        $csv_rows = [];

        // Header row
        $csv_rows[] = $csv_headers;

        // Data rows
        foreach ($subscribers as $subscriber) {
            $row = [];
            foreach ($csv_headers as $header) {
                $row[] = $subscriber[$header] ?? '';
            }
            $csv_rows[] = $row;
        }

        // Generate CSV content
        $output = fopen('php://temp', 'r+');
        foreach ($csv_rows as $row) {
            fputcsv($output, $row);
        }
        rewind($output);
        $csv_content = stream_get_contents($output);
        fclose($output);

        // Generate filename
        $filename = 'subscribers';
        if ($list_id > 0) {
            $list_name = $wpdb->get_var($wpdb->prepare(
                "SELECT name FROM {$lists_table} WHERE id = %d",
                $list_id
            ));
            if ($list_name) {
                $filename .= '-' . sanitize_title($list_name);
            }
        }
        $filename .= '-' . gmdate('Y-m-d') . '.csv';

        return new WP_REST_Response([
            'success' => true,
            'filename' => $filename,
            'content' => base64_encode($csv_content),
            'count' => count($subscribers),
        ]);
    }

    /**
     * Preview CSV import - analyze file and return column mapping options
     *
     * @param WP_REST_Request $request File upload
     *
     * @return WP_REST_Response Preview data with columns and sample rows
     */
    public function preview_import_subscribers(WP_REST_Request $request): WP_REST_Response {
        $files = $request->get_file_params();

        if (empty($files['file'])) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('No file uploaded', 'outreach'),
            ], 400);
        }

        $file = $files['file'];

        // Validate file type
        $allowed_types = ['text/csv', 'text/plain', 'application/csv', 'application/vnd.ms-excel'];
        $file_type = $file['type'] ?? '';
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        if (!in_array($file_type, $allowed_types) && $file_ext !== 'csv') {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Invalid file type. Please upload a CSV file.', 'outreach'),
            ], 400);
        }

        // Read CSV file
        $handle = fopen($file['tmp_name'], 'r');
        if (!$handle) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Failed to read file', 'outreach'),
            ], 400);
        }

        // Detect delimiter
        $first_line = fgets($handle);
        rewind($handle);
        $delimiter = $this->detect_csv_delimiter($first_line);

        // Read header row
        $headers = fgetcsv($handle, 0, $delimiter);
        if (!$headers || count($headers) < 1) {
            fclose($handle);
            return new WP_REST_Response([
                'success' => false,
                'message' => __('CSV file appears to be empty or invalid', 'outreach'),
            ], 400);
        }

        // Clean headers
        $headers = array_map('trim', $headers);
        $headers = array_map(function($h) {
            return preg_replace('/^\xEF\xBB\xBF/', '', $h); // Remove BOM
        }, $headers);

        // Read sample rows (first 5)
        $sample_rows = [];
        $row_count = 0;
        while (($row = fgetcsv($handle, 0, $delimiter)) !== false && count($sample_rows) < 5) {
            $sample_rows[] = array_map('trim', $row);
            $row_count++;
        }

        // Count total rows
        while (fgetcsv($handle, 0, $delimiter) !== false) {
            $row_count++;
        }

        fclose($handle);

        // Auto-detect column mappings
        $suggested_mappings = $this->suggest_column_mappings($headers);

        // Available fields for mapping
        $available_fields = [
            ['value' => '', 'label' => __('-- Skip --', 'outreach')],
            ['value' => 'email', 'label' => __('Email (required)', 'outreach')],
            ['value' => 'first_name', 'label' => __('First Name', 'outreach')],
            ['value' => 'last_name', 'label' => __('Last Name', 'outreach')],
            ['value' => 'status', 'label' => __('Status', 'outreach')],
            ['value' => 'source', 'label' => __('Source', 'outreach')],
        ];

        // Store file temporarily for actual import using WordPress upload handler
        $upload_overrides = [
            'test_form' => false,
            'unique_filename_callback' => function($dir, $name, $ext) {
                return 'wpo-import-' . wp_generate_password(16, false) . $ext;
            },
            'mimes' => [
                'csv' => 'text/csv',
                'txt' => 'text/plain',
            ],
            'test_type' => false, // Allow CSV files regardless of MIME type detection
        ];

        // Load WordPress file handling functions if not already loaded
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        // Use wp_handle_upload for WordPress.org compliance
        $uploaded = wp_handle_upload($file, $upload_overrides);

        if (isset($uploaded['error'])) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $uploaded['error'],
            ], 400);
        }

        $temp_file = $uploaded['file'];

        // Store temp file reference in transient
        $import_key = wp_generate_password(32, false);
        set_transient('wpo_import_' . $import_key, [
            'file' => $temp_file,
            'delimiter' => $delimiter,
            'headers' => $headers,
        ], HOUR_IN_SECONDS);

        return new WP_REST_Response([
            'success' => true,
            'import_key' => $import_key,
            'headers' => $headers,
            'sample_rows' => $sample_rows,
            'total_rows' => $row_count,
            'suggested_mappings' => $suggested_mappings,
            'available_fields' => $available_fields,
        ]);
    }

    /**
     * Import subscribers from CSV
     *
     * @param WP_REST_Request $request {
     *     @type string $import_key      Import key from preview.
     *     @type array  $mappings        Column to field mappings.
     *     @type array  $list_ids        Lists to add subscribers to.
     *     @type string $duplicate_action How to handle duplicates: skip, update.
     *     @type string $status          Default status for new subscribers.
     * }
     *
     * @return WP_REST_Response|WP_Error Import results
     */
    public function import_subscribers(WP_REST_Request $request): WP_REST_Response|WP_Error {
        // Check if import/export feature is available
        if (!FeatureManager::can('import_export')) {
            return new WP_Error(
                'feature_restricted',
                FeatureManager::getRestrictionMessage('import_export'),
                [
                    'status' => 403,
                    'upgrade_url' => FeatureManager::getUpgradeUrl(),
                ]
            );
        }

        global $wpdb;

        $import_key = sanitize_text_field($request->get_param('import_key') ?? '');
        $mappings = $request->get_param('mappings') ?? [];
        $list_ids = $request->get_param('list_ids') ?? [];
        $duplicate_action = sanitize_text_field($request->get_param('duplicate_action') ?? 'skip');
        $default_status = sanitize_text_field($request->get_param('status') ?? 'active');

        // Validate import key
        $import_data = get_transient('wpo_import_' . $import_key);
        if (!$import_data || !file_exists($import_data['file'])) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Import session expired. Please upload the file again.', 'outreach'),
            ], 400);
        }

        // Validate email mapping exists
        $email_column = array_search('email', $mappings);
        if ($email_column === false) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Email column mapping is required', 'outreach'),
            ], 400);
        }

        $table = $wpdb->prefix . 'outreach_subscribers';
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';

        // Read CSV file
        $handle = fopen($import_data['file'], 'r');
        if (!$handle) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Failed to read import file', 'outreach'),
            ], 400);
        }

        $delimiter = $import_data['delimiter'];

        // Skip header row
        fgetcsv($handle, 0, $delimiter);

        $stats = [
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => 0,
            'error_details' => [],
        ];

        $row_number = 1;

        while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
            $row_number++;
            $row = array_map('trim', $row);

            // Extract email
            $email = isset($row[$email_column]) ? sanitize_email($row[$email_column]) : '';

            if (empty($email) || !is_email($email)) {
                $stats['errors']++;
                if (count($stats['error_details']) < 10) {
                    $stats['error_details'][] = sprintf(
                        __('Row %d: Invalid or empty email address', 'outreach'),
                        $row_number
                    );
                }
                continue;
            }

            // Check for existing subscriber
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT id FROM {$table} WHERE email = %s",
                $email
            ));

            if ($existing) {
                if ($duplicate_action === 'skip') {
                    $stats['skipped']++;
                    continue;
                }

                // Update existing subscriber
                $update_data = [];
                foreach ($mappings as $col_index => $field) {
                    if ($field && $field !== 'email' && isset($row[$col_index])) {
                        $update_data[$field] = sanitize_text_field($row[$col_index]);
                    }
                }

                if (!empty($update_data)) {
                    $wpdb->update($table, $update_data, ['id' => $existing->id]);
                }

                // Add to lists
                $this->add_subscriber_to_lists($existing->id, $list_ids);

                $stats['updated']++;
            } else {
                // Create new subscriber
                $subscriber_data = [
                    'email' => $email,
                    'first_name' => '',
                    'last_name' => '',
                    'status' => $default_status,
                    'source' => 'import',
                    'token' => wp_generate_password(32, false),
                    'created_at' => current_time('mysql'),
                ];

                // Map fields
                foreach ($mappings as $col_index => $field) {
                    if ($field && $field !== 'email' && isset($row[$col_index]) && array_key_exists($field, $subscriber_data)) {
                        $subscriber_data[$field] = sanitize_text_field($row[$col_index]);
                    }
                }

                // Validate status
                if (!in_array($subscriber_data['status'], ['active', 'pending', 'unsubscribed', 'bounced'])) {
                    $subscriber_data['status'] = $default_status;
                }

                $result = $wpdb->insert($table, $subscriber_data);

                if ($result) {
                    $subscriber_id = $wpdb->insert_id;

                    // Add to lists
                    $this->add_subscriber_to_lists($subscriber_id, $list_ids);

                    // Fire automation trigger
                    do_action('wp_outreach_subscriber_created', $subscriber_id, $subscriber_data);

                    $stats['imported']++;
                } else {
                    $stats['errors']++;
                    if (count($stats['error_details']) < 10) {
                        $stats['error_details'][] = sprintf(
                            __('Row %d: Failed to insert subscriber', 'outreach'),
                            $row_number
                        );
                    }
                }
            }
        }

        fclose($handle);

        // Cleanup temp file
        @unlink($import_data['file']);
        delete_transient('wpo_import_' . $import_key);

        return new WP_REST_Response([
            'success' => true,
            'stats' => $stats,
            'message' => sprintf(
                __('Import complete: %d imported, %d updated, %d skipped, %d errors', 'outreach'),
                $stats['imported'],
                $stats['updated'],
                $stats['skipped'],
                $stats['errors']
            ),
        ]);
    }

    /**
     * Add subscriber to multiple lists
     */
    private function add_subscriber_to_lists(int $subscriber_id, array $list_ids): void {
        global $wpdb;
        $pivot_table = $wpdb->prefix . 'outreach_subscriber_list';

        foreach ($list_ids as $list_id) {
            $list_id = (int) $list_id;
            if ($list_id <= 0) continue;

            // Check if already in list
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT 1 FROM {$pivot_table} WHERE subscriber_id = %d AND list_id = %d",
                $subscriber_id,
                $list_id
            ));

            if (!$exists) {
                $wpdb->insert($pivot_table, [
                    'subscriber_id' => $subscriber_id,
                    'list_id' => $list_id,
                    'subscribed_at' => current_time('mysql'),
                ]);
            }
        }
    }

    /**
     * Detect CSV delimiter from first line
     */
    private function detect_csv_delimiter(string $line): string {
        $delimiters = [',', ';', "\t", '|'];
        $counts = [];

        foreach ($delimiters as $delimiter) {
            $counts[$delimiter] = substr_count($line, $delimiter);
        }

        return array_search(max($counts), $counts) ?: ',';
    }

    /**
     * Suggest column mappings based on header names
     */
    private function suggest_column_mappings(array $headers): array {
        $mappings = [];

        $field_patterns = [
            'email' => ['email', 'e-mail', 'email_address', 'emailaddress', 'mail'],
            'first_name' => ['first_name', 'firstname', 'first', 'fname', 'given_name', 'givenname'],
            'last_name' => ['last_name', 'lastname', 'last', 'lname', 'surname', 'family_name', 'familyname'],
            'status' => ['status', 'subscription_status', 'sub_status'],
            'source' => ['source', 'origin', 'signup_source'],
        ];

        foreach ($headers as $index => $header) {
            $header_lower = strtolower(trim($header));
            $mappings[$index] = '';

            foreach ($field_patterns as $field => $patterns) {
                if (in_array($header_lower, $patterns)) {
                    $mappings[$index] = $field;
                    break;
                }
            }
        }

        return $mappings;
    }

    /**
     * Preview WordPress users for import
     */
    public function preview_import_wp_users(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $roles = $request->get_param('roles') ?: [];
        $date_from = $request->get_param('date_from');
        $date_to = $request->get_param('date_to');
        $exclude_existing = $request->get_param('exclude_existing');

        // Build user query args
        $args = [
            'number' => -1,
            'fields' => ['ID', 'user_email', 'user_registered', 'display_name'],
        ];

        // Filter by roles
        if (!empty($roles)) {
            $args['role__in'] = $roles;
        }

        // Filter by date
        if (!empty($date_from)) {
            $args['date_query'] = $args['date_query'] ?? [];
            $args['date_query'][] = [
                'after' => $date_from,
                'inclusive' => true,
            ];
        }

        if (!empty($date_to)) {
            $args['date_query'] = $args['date_query'] ?? [];
            $args['date_query'][] = [
                'before' => $date_to,
                'inclusive' => true,
            ];
        }

        $users = get_users($args);
        $total_users = count($users);

        // Get existing subscriber emails if we need to exclude them
        $existing_emails = [];
        if ($exclude_existing) {
            $table = $wpdb->prefix . 'outreach_subscribers';
            $existing_emails = $wpdb->get_col("SELECT LOWER(email) FROM {$table}");
        }

        // Filter out existing subscribers
        $users_to_import = [];
        $existing_count = 0;

        foreach ($users as $user) {
            if ($exclude_existing && in_array(strtolower($user->user_email), $existing_emails)) {
                $existing_count++;
                continue;
            }

            $users_to_import[] = [
                'id' => $user->ID,
                'email' => $user->user_email,
                'display_name' => $user->display_name,
                'registered' => $user->user_registered,
            ];
        }

        // Get available roles for the filter dropdown
        $available_roles = [];
        foreach (wp_roles()->roles as $role_key => $role) {
            $available_roles[] = [
                'key' => $role_key,
                'name' => $role['name'],
            ];
        }

        return new WP_REST_Response([
            'success' => true,
            'total_users' => $total_users,
            'importable_count' => count($users_to_import),
            'existing_count' => $existing_count,
            'preview' => array_slice($users_to_import, 0, 10), // First 10 for preview
            'available_roles' => $available_roles,
        ]);
    }

    /**
     * Import WordPress users as subscribers
     */
    public function import_wp_users(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $roles = $request->get_param('roles') ?: [];
        $date_from = $request->get_param('date_from');
        $date_to = $request->get_param('date_to');
        $exclude_existing = $request->get_param('exclude_existing') ?? true;
        $list_ids = $request->get_param('list_ids') ?: [];

        if (empty($list_ids)) {
            return new WP_Error('no_lists', __('Please select at least one list', 'outreach'), ['status' => 400]);
        }

        // Build user query args
        $args = [
            'number' => -1,
            'fields' => 'all',
        ];

        if (!empty($roles)) {
            $args['role__in'] = $roles;
        }

        if (!empty($date_from)) {
            $args['date_query'] = $args['date_query'] ?? [];
            $args['date_query'][] = [
                'after' => $date_from,
                'inclusive' => true,
            ];
        }

        if (!empty($date_to)) {
            $args['date_query'] = $args['date_query'] ?? [];
            $args['date_query'][] = [
                'before' => $date_to,
                'inclusive' => true,
            ];
        }

        $users = get_users($args);
        $table = $wpdb->prefix . 'outreach_subscribers';

        // Get existing subscriber emails
        $existing_emails = [];
        if ($exclude_existing) {
            $existing_emails = $wpdb->get_col("SELECT LOWER(email) FROM {$table}");
        }

        $stats = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
        ];

        foreach ($users as $user) {
            $email_lower = strtolower($user->user_email);

            // Skip if already exists and exclude_existing is true
            if ($exclude_existing && in_array($email_lower, $existing_emails)) {
                $stats['skipped']++;
                continue;
            }

            // Check if email already exists
            $existing_id = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$table} WHERE email = %s",
                $user->user_email
            ));

            if ($existing_id) {
                // Add existing subscriber to lists
                $this->add_subscriber_to_lists($existing_id, $list_ids);
                $stats['skipped']++;
                continue;
            }

            // Create new subscriber
            $subscriber_data = [
                'email' => $user->user_email,
                'first_name' => get_user_meta($user->ID, 'first_name', true) ?: '',
                'last_name' => get_user_meta($user->ID, 'last_name', true) ?: '',
                'status' => 'active',
                'source' => 'wp_users',
                'user_id' => $user->ID,
                'created_at' => current_time('mysql'),
            ];

            $result = $wpdb->insert($table, $subscriber_data);

            if ($result) {
                $subscriber_id = $wpdb->insert_id;
                $this->add_subscriber_to_lists($subscriber_id, $list_ids);

                // Fire automation trigger
                do_action('wp_outreach_subscriber_created', $subscriber_id, $subscriber_data);

                $stats['imported']++;
            } else {
                $stats['errors']++;
            }
        }

        return new WP_REST_Response([
            'success' => true,
            'stats' => $stats,
            'message' => sprintf(
                __('Import complete: %d imported, %d skipped, %d errors', 'outreach'),
                $stats['imported'],
                $stats['skipped'],
                $stats['errors']
            ),
        ]);
    }

    /**
     * Register WPDM integration routes
     */
    private function register_wpdm_routes(): void {
        // Get WPDM products for condition dropdowns
        register_rest_route($this->namespace, '/wpdm/products', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_wpdm_products'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Get WPDM products for automation condition configuration
     *
     * @return WP_REST_Response
     */
    public function get_wpdm_products(): WP_REST_Response {
        // Check if WPDM is active
        if (!class_exists('WPDMPP\\Libs\\Order') && !function_exists('wpdmpp_effective_price')) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'WPDM Premium Packages is not active',
                'products' => [],
            ]);
        }

        // Use WPDMIntegration::getProducts() if available
        if (class_exists('WPOutreach\\Integrations\\WPDMIntegration')) {
            $products = \WPOutreach\Integrations\WPDMIntegration::getProducts();

            return new WP_REST_Response([
                'success' => true,
                'products' => $products,
            ]);
        }

        // Fallback: Get products directly
        $products = get_posts([
            'post_type' => 'wpdmpro',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);

        $result = [];
        foreach ($products as $product) {
            $price = 0;
            if (function_exists('wpdmpp_effective_price')) {
                $price = wpdmpp_effective_price($product->ID);
            }

            // Only include products with a price (premium packages)
            if ($price > 0) {
                $result[] = [
                    'id' => $product->ID,
                    'name' => $product->post_title,
                    'price' => $price,
                ];
            }
        }

        return new WP_REST_Response([
            'success' => true,
            'products' => $result,
        ]);
    }

    /**
     * Register incoming webhook routes
     *
     * These routes allow external systems (Zapier, Make, custom apps) to trigger
     * automations via HTTP POST requests.
     */
    private function register_webhook_routes(): void {
        // Incoming webhook endpoint (public with signature validation)
        // Uses a unique webhook_key instead of automation_id for security
        register_rest_route($this->namespace, '/webhooks/incoming/(?P<webhook_key>[a-zA-Z0-9]+)', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'handle_incoming_webhook'],
                'permission_callback' => '__return_true', // Signature validated in handler
                'args' => [
                    'webhook_key' => [
                        'required' => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);
    }

    /**
     * Handle incoming webhook request
     *
     * Validates the request signature, extracts subscriber data from the payload,
     * and triggers the automation for the specified subscriber.
     *
     * @param WP_REST_Request $request The incoming request
     * @return WP_REST_Response|WP_Error Response or error
     */
    public function handle_incoming_webhook(WP_REST_Request $request): WP_REST_Response|WP_Error {
        global $wpdb;

        $webhook_key = sanitize_text_field($request->get_param('webhook_key'));

        // Find automation by webhook_key stored in trigger_config
        // We search for automations with incoming_webhook trigger and matching webhook_key
        $automations = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}outreach_automations
             WHERE trigger_type = 'incoming_webhook'",
            []
        ));

        $automation = null;
        $trigger_config = [];

        foreach ($automations as $a) {
            $config = json_decode($a->trigger_config, true) ?: [];
            if (!empty($config['webhook_key']) && $config['webhook_key'] === $webhook_key) {
                $automation = $a;
                $trigger_config = $config;
                break;
            }
        }

        if (!$automation) {
            return new WP_Error(
                'webhook_not_found',
                __('Invalid webhook URL', 'outreach'),
                ['status' => 404]
            );
        }

        // Check if automation is active
        if ($automation->status !== 'active') {
            return new WP_Error(
                'automation_inactive',
                __('Automation is not active', 'outreach'),
                ['status' => 400]
            );
        }

        $secret_key = $trigger_config['secret_key'] ?? '';
        $require_signature = !empty($trigger_config['require_signature']);
        $signature_header_name = !empty($trigger_config['signature_header'])
            ? $trigger_config['signature_header']
            : 'X-Webhook-Signature';

        // Validate signature only if require_signature is enabled
        if ($require_signature && !empty($secret_key)) {
            $signature_header = $request->get_header($signature_header_name);

            if (empty($signature_header)) {
                return new WP_Error(
                    'missing_signature',
                    sprintf(__('Missing %s header', 'outreach'), $signature_header_name),
                    ['status' => 401]
                );
            }

            // Get raw body for signature verification
            $raw_body = $request->get_body();

            // Calculate expected signature (raw hash without prefix)
            $expected_hash = hash_hmac('sha256', $raw_body, $secret_key);

            // Normalize incoming signature - strip any algorithm prefix (sha256=, sha1=, etc.)
            $incoming_signature = $signature_header;
            if (preg_match('/^(sha256|sha1|sha512)=(.+)$/i', $signature_header, $matches)) {
                $incoming_signature = $matches[2];
            }

            // Timing-safe comparison
            if (!hash_equals($expected_hash, $incoming_signature)) {
                return new WP_Error(
                    'invalid_signature',
                    __('Invalid webhook signature', 'outreach'),
                    ['status' => 401]
                );
            }
        }

        // Check rate limit
        $rate_limit = (int) ($trigger_config['rate_limit'] ?? 100);
        if ($rate_limit > 0) {
            $rate_limit_key = 'outreach_webhook_rate_' . $automation->id;
            $current_count = (int) get_transient($rate_limit_key);

            if ($current_count >= $rate_limit) {
                return new WP_Error(
                    'rate_limit_exceeded',
                    sprintf(__('Rate limit exceeded. Maximum %d requests per minute.', 'outreach'), $rate_limit),
                    ['status' => 429]
                );
            }

            // Increment counter
            set_transient($rate_limit_key, $current_count + 1, MINUTE_IN_SECONDS);
        }

        // Parse request body
        $payload = $request->get_json_params();

        if (empty($payload)) {
            return new WP_Error(
                'empty_payload',
                __('Request body is empty or invalid JSON', 'outreach'),
                ['status' => 400]
            );
        }

        // Extract email from payload (optional - allows webhook without subscriber)
        $email_field = $trigger_config['subscriber_email_field'] ?? '';
        $email = '';
        $subscriber_id = 0;
        $subscriber = null;

        if (!empty($email_field)) {
            $email = $this->get_nested_value($payload, $email_field);
            if (!empty($email) && is_email($email)) {
                $email = sanitize_email($email);

                // Find or create subscriber
                $subscribers_table = $wpdb->prefix . 'outreach_subscribers';
                $subscriber = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$subscribers_table} WHERE email = %s",
                    $email
                ));

                $create_if_not_found = !empty($trigger_config['create_if_not_found']);

                if (!$subscriber && $create_if_not_found) {
                    // Extract name fields from payload
                    $first_name = sanitize_text_field($payload['first_name'] ?? '');
                    $last_name = sanitize_text_field($payload['last_name'] ?? '');

                    // Fallback to "name" field if first_name/last_name not provided
                    if (empty($first_name) && empty($last_name) && !empty($payload['name'])) {
                        $name_parts = explode(' ', sanitize_text_field($payload['name']), 2);
                        $first_name = $name_parts[0] ?? '';
                        $last_name = $name_parts[1] ?? '';
                    }

                    // Create new subscriber
                    $wpdb->insert($subscribers_table, [
                        'email' => $email,
                        'first_name' => $first_name,
                        'last_name' => $last_name,
                        'status' => 'active',
                        'source' => 'webhook',
                        'created_at' => current_time('mysql'),
                    ]);

                    $subscriber_id = $wpdb->insert_id;

                    // Fire subscriber created action
                    do_action('wp_outreach_subscriber_created', $subscriber_id, [
                        'email' => $email,
                        'first_name' => $first_name,
                        'last_name' => $last_name,
                        'source' => 'webhook',
                    ]);
                } elseif ($subscriber) {
                    $subscriber_id = (int) $subscriber->id;
                }
            }
        }

        // Build context from payload (works with or without subscriber)
        $context = [
            'email' => $email,
            'first_name' => $payload['first_name'] ?? ($subscriber->first_name ?? ''),
            'last_name' => $payload['last_name'] ?? ($subscriber->last_name ?? ''),
            'source' => 'webhook',
            'automation_id' => (int) $automation->id,
            'webhook_payload' => $payload,
            'has_subscriber' => $subscriber_id > 0,
        ];

        // Add all payload fields to context with webhook_ prefix
        foreach ($payload as $key => $value) {
            if (!in_array($key, ['email', 'first_name', 'last_name', 'name'])) {
                $context['webhook_' . $key] = $value;
            }
        }

        // Fire the webhook received action
        do_action('wp_outreach_webhook_received', (int) $automation->id, $subscriber_id, $context);

        // Enqueue subscriber into the automation
        $result = \WPOutreach\Automation\AutomationEngine::enqueueSubscriber(
            (int) $automation->id,
            $subscriber_id,
            $context
        );

        if (!$result) {
            return new WP_Error(
                'enqueue_failed',
                __('Failed to enqueue subscriber into automation', 'outreach'),
                ['status' => 500]
            );
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Webhook received and automation triggered', 'outreach'),
            'subscriber_id' => $subscriber_id,
            'automation_id' => (int) $automation->id,
        ], 200);
    }

    /**
     * Get a nested value from an array using dot notation
     *
     * @param array $array The array to search
     * @param string $path The dot-notation path (e.g., "user.email" or "data.contact.email")
     * @return mixed|null The value at the path or null if not found
     */
    private function get_nested_value(array $array, string $path) {
        $keys = explode('.', $path);
        $value = $array;

        foreach ($keys as $key) {
            if (!is_array($value) || !array_key_exists($key, $value)) {
                return null;
            }
            $value = $value[$key];
        }

        return $value;
    }

    /**
     * Get all available keys from payload (including nested paths)
     * For debugging webhook payloads
     *
     * @param array $array The payload array
     * @param string $prefix Current path prefix for nested keys
     * @param int $depth Max depth to traverse
     * @return array List of available field paths
     */
    private function get_payload_keys(array $array, string $prefix = '', int $depth = 3): array {
        $keys = [];

        if ($depth <= 0) {
            return $keys;
        }

        foreach ($array as $key => $value) {
            $path = $prefix ? "{$prefix}.{$key}" : $key;

            if (is_array($value) && !empty($value)) {
                // Add the key itself
                $keys[] = $path;
                // Recursively get nested keys
                $keys = array_merge($keys, $this->get_payload_keys($value, $path, $depth - 1));
            } else {
                $keys[] = $path;
            }
        }

        return $keys;
    }
}
